"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateUpgrades = exports.generateSuccessfulChanges = exports.updateDependencies = void 0;
const pathLib = require("path");
const debugLib = require("debug");
const pipenvPipfileFix = require("@snyk/fix-pipenv-pipfile");
const no_fixes_applied_1 = require("../../../../../lib/errors/no-fixes-applied");
const standardize_package_name_1 = require("../../pip-requirements/update-dependencies/standardize-package-name");
const command_failed_to_run_error_1 = require("../../../../../lib/errors/command-failed-to-run-error");
const validate_required_data_1 = require("../../validate-required-data");
const debug = debugLib('snyk-fix:python:Pipfile');
async function updateDependencies(entity, options) {
    const handlerResult = {
        succeeded: [],
        failed: [],
        skipped: [],
    };
    let pipenvCommand;
    try {
        const { remediation, targetFile } = validate_required_data_1.validateRequiredData(entity);
        const { dir } = pathLib.parse(pathLib.resolve(entity.workspace.path, targetFile));
        // TODO: for better support we need to:
        // 1. parse the manifest and extract original requirements, version spec etc
        // 2. swap out only the version and retain original spec
        // 3. re-lock the lockfile
        // Currently this is not possible as there is no Pipfile parser that would do this.
        const upgrades = generateUpgrades(remediation.pin);
        if (!options.dryRun) {
            const res = await pipenvPipfileFix.pipenvInstall(dir, upgrades, {
                python: entity.options.command,
            });
            if (res.exitCode !== 0) {
                pipenvCommand = res.command;
                throwPipenvError(res.stderr, res.command);
            }
        }
        const changes = generateSuccessfulChanges(remediation.pin);
        handlerResult.succeeded.push({ original: entity, changes });
    }
    catch (error) {
        debug(`Failed to fix ${entity.scanResult.identity.targetFile}.\nERROR: ${error}`);
        handlerResult.failed.push({
            original: entity,
            error,
            tip: pipenvCommand ? `Try running \`${pipenvCommand}\`` : undefined,
        });
    }
    return handlerResult;
}
exports.updateDependencies = updateDependencies;
function generateSuccessfulChanges(pins) {
    const changes = [];
    for (const pkgAtVersion of Object.keys(pins)) {
        const pin = pins[pkgAtVersion];
        const updatedMessage = pin.isTransitive ? 'Pinned' : 'Upgraded';
        const newVersion = pin.upgradeTo.split('@')[1];
        const [pkgName, version] = pkgAtVersion.split('@');
        changes.push({
            success: true,
            userMessage: `${updatedMessage} ${pkgName} from ${version} to ${newVersion}`,
            issueIds: pin.vulns,
            from: pkgAtVersion,
            to: `${pkgName}@${newVersion}`,
        });
    }
    return changes;
}
exports.generateSuccessfulChanges = generateSuccessfulChanges;
function generateUpgrades(pins) {
    const upgrades = [];
    for (const pkgAtVersion of Object.keys(pins)) {
        const pin = pins[pkgAtVersion];
        const newVersion = pin.upgradeTo.split('@')[1];
        const [pkgName] = pkgAtVersion.split('@');
        upgrades.push(`${standardize_package_name_1.standardizePackageName(pkgName)}==${newVersion}`);
    }
    return upgrades;
}
exports.generateUpgrades = generateUpgrades;
function throwPipenvError(stderr, command) {
    const errorStr = stderr.toLowerCase();
    const incompatibleDeps = 'There are incompatible versions in the resolved dependencies';
    const lockingFailed = 'Locking failed';
    const versionNotFound = 'Could not find a version that matches';
    if (stderr.includes(incompatibleDeps.toLocaleLowerCase())) {
        throw new command_failed_to_run_error_1.CommandFailedError(incompatibleDeps, command);
    }
    if (errorStr.includes(lockingFailed.toLocaleLowerCase())) {
        throw new command_failed_to_run_error_1.CommandFailedError(lockingFailed, command);
    }
    if (stderr.includes(versionNotFound.toLocaleLowerCase())) {
        throw new command_failed_to_run_error_1.CommandFailedError(versionNotFound, command);
    }
    throw new no_fixes_applied_1.NoFixesCouldBeAppliedError();
}
//# sourceMappingURL=index.js.map