"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToDetectYamlConfigError = exports.MissingRequiredFieldsInKubernetesYamlError = exports.HelmFileNotSupportedError = exports.checkRequiredFieldsMatch = exports.detectConfigType = exports.assertHelmAndThrow = exports.REQUIRED_CLOUDFORMATION_FIELDS = exports.REQUIRED_K8S_FIELDS = void 0;
const errors_1 = require("../../../../../lib/errors");
const constants_1 = require("../../../../../lib/iac/constants");
const error_utils_1 = require("../error-utils");
const types_1 = require("../types");
exports.REQUIRED_K8S_FIELDS = ['apiVersion', 'kind', 'metadata'];
exports.REQUIRED_CLOUDFORMATION_FIELDS = ['Resources'];
function assertHelmAndThrow(fileData) {
    const lines = fileData.fileContent.split(/\r\n|\r|\n/);
    lines.forEach((line) => {
        const isHelmFile = line.includes('{{') && line.includes('}}');
        if (isHelmFile) {
            throw new HelmFileNotSupportedError(fileData.filePath);
        }
    });
}
exports.assertHelmAndThrow = assertHelmAndThrow;
function detectConfigType(fileData, parsedIacFiles) {
    assertHelmAndThrow(fileData);
    return parsedIacFiles.map((parsedIaCFile, docId) => {
        if (checkRequiredFieldsMatch(parsedIaCFile, exports.REQUIRED_CLOUDFORMATION_FIELDS)) {
            return Object.assign(Object.assign({}, fileData), { jsonContent: parsedIaCFile, projectType: constants_1.IacProjectType.CLOUDFORMATION, engineType: types_1.EngineType.CloudFormation, docId });
        }
        else if (checkRequiredFieldsMatch(parsedIaCFile, exports.REQUIRED_K8S_FIELDS)) {
            return Object.assign(Object.assign({}, fileData), { jsonContent: parsedIaCFile, projectType: constants_1.IacProjectType.K8S, engineType: types_1.EngineType.Kubernetes, docId });
        }
        else {
            throw new FailedToDetectYamlConfigError(fileData.filePath);
        }
    });
}
exports.detectConfigType = detectConfigType;
function checkRequiredFieldsMatch(parsedDocument, requiredFields) {
    return requiredFields.every((requiredField) => parsedDocument.hasOwnProperty(requiredField));
}
exports.checkRequiredFieldsMatch = checkRequiredFieldsMatch;
class HelmFileNotSupportedError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to parse Helm file');
        this.code = types_1.IaCErrorCodes.FailedToParseHelmError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to parse the YAML file "${filename}" as we currently do not support scanning of Helm files. More information can be found through our documentation:\nhttps://support.snyk.io/hc/en-us/articles/360012429477-Test-your-Kubernetes-files-with-our-CLI-tool`;
    }
}
exports.HelmFileNotSupportedError = HelmFileNotSupportedError;
class MissingRequiredFieldsInKubernetesYamlError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to detect Kubernetes file, missing required fields');
        this.code = types_1.IaCErrorCodes.MissingRequiredFieldsInKubernetesYamlError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to detect whether the YAML file "${filename}" is a valid Kubernetes file, it is missing the following fields: "${exports.REQUIRED_K8S_FIELDS.join('", "')}"`;
    }
}
exports.MissingRequiredFieldsInKubernetesYamlError = MissingRequiredFieldsInKubernetesYamlError;
class FailedToDetectYamlConfigError extends errors_1.CustomError {
    constructor(filename) {
        super('Failed to detect either a Kubernetes or CloudFormation file, missing required fields');
        this.code = types_1.IaCErrorCodes.FailedToDetectYamlConfigError;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage = `We were unable to detect whether the YAML file "${filename}" is a valid Kubernetes or CloudFormation file. For Kubernetes required fields are: "${exports.REQUIRED_K8S_FIELDS.join('", "')}". For CloudFormation required fields are: "${exports.REQUIRED_CLOUDFORMATION_FIELDS.join('", "')}". Please contact support@snyk.io, if possible with a redacted version of the file`;
    }
}
exports.FailedToDetectYamlConfigError = FailedToDetectYamlConfigError;
//# sourceMappingURL=k8s-or-cloudformation-parser.js.map