"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeDirectoryIterator = void 0;
const fs = require("fs");
const path_1 = require("path");
/**
 * Walk the provided directory tree, depth first, yielding the matched filename
 * to the caller. An optional `maxDepth` argument can be provided to limit how
 * deep in the file tree the search will go.
 */
function* makeDirectoryIterator(root, options = {}) {
    if (!isDirectory(root)) {
        throw new Error(`Path "${root}" is not a directory`);
    }
    // Internal function to allow us to track the recursion depth.
    function* walk(dirpath, currentDepth) {
        const filenames = fs.readdirSync(dirpath);
        for (const filename of filenames) {
            // NOTE: We filter filenames that start with a period to maintain
            // backwards compatibility with the original implementation that used the
            // "glob" package.
            if (!options.includeDotfiles && filename.startsWith('.')) {
                continue;
            }
            const resolved = path_1.resolve(dirpath, filename);
            if (isDirectory(resolved)) {
                // Skip this directory if max depth has been reached.
                if (options.maxDepth === currentDepth) {
                    continue;
                }
                yield* walk(resolved, currentDepth + 1);
            }
            else {
                yield resolved;
            }
        }
    }
    yield* walk(root, 1);
}
exports.makeDirectoryIterator = makeDirectoryIterator;
// NOTE: We use isDirectory here instead of isLocalFolder() in order to
// follow symlinks and match the original glob() implementation.
function isDirectory(path) {
    try {
        // statSync will resolve symlinks, lstatSync will not.
        return fs.statSync(path).isDirectory();
    }
    catch (e) {
        return false;
    }
}
//# sourceMappingURL=makeDirectoryIterator.js.map