"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.build = void 0;
const dep_graph_1 = require("@snyk/dep-graph");
// Poetry uses the virtualenv to create an environment and this comes with these
// packages pre-installed, therefore they won't be part of the lockfile.
// See: https://github.com/python-poetry/poetry/issues/3075#issuecomment-703334427
const IGNORED_DEPENDENCIES = [
    'setuptools',
    'distribute',
    'pip',
    'wheel',
];
function build(pkgDetails, pkgDependencyNames, pkgSpecs) {
    const builder = new dep_graph_1.DepGraphBuilder({ name: 'poetry' }, pkgDetails);
    addDependenciesToGraph(pkgDependencyNames, pkgSpecs, builder.rootNodeId, builder);
    return builder.build();
}
exports.build = build;
function addDependenciesToGraph(pkgNames, pkgSpecs, parentNodeId, builder) {
    for (const pkgName of pkgNames) {
        addDependenciesForPkg(pkgName, pkgSpecs, parentNodeId, builder);
    }
}
function addDependenciesForPkg(pkgName, pkgSpecs, parentNodeId, builder) {
    if (IGNORED_DEPENDENCIES.includes(pkgName)) {
        return;
    }
    const pkg = pkgLockInfoFor(pkgName, pkgSpecs);
    if (!pkg) {
        return;
    }
    if (isPkgAlreadyInGraph(pkg, builder)) {
        builder.connectDep(parentNodeId, pkg.name);
        return;
    }
    const pkgInfo = { name: pkg.name, version: pkg.version };
    builder.addPkgNode(pkgInfo, pkg.name).connectDep(parentNodeId, pkg.name);
    addDependenciesToGraph(pkg.dependencies, pkgSpecs, pkg.name, builder);
}
function isPkgAlreadyInGraph(pkg, builder) {
    return builder
        .getPkgs()
        .some((existingPkg) => existingPkg.name === pkg.name && existingPkg.version === pkg.version);
}
function pkgLockInfoFor(pkgName, pkgSpecs) {
    // From PEP 426 https://www.python.org/dev/peps/pep-0426/#name
    // All comparisons of distribution names MUST be case insensitive, and MUST
    // consider hyphens and underscores to be equivalent
    const pkgLockInfo = pkgSpecs.find((lockItem) => lockItem.name.toLowerCase().replace(/_/g, '-') ===
        pkgName.toLowerCase().replace(/_/g, '-') ||
        lockItem.name.toLowerCase().replace(/-/g, '_') ===
            pkgName.toLowerCase().replace(/-/g, '_'));
    if (!pkgLockInfo) {
        console.warn(`Could not find any lockfile metadata for package: ${pkgName}. This package will not be represented in the dependency graph.`);
    }
    return pkgLockInfo;
}
//# sourceMappingURL=poetry-dep-graph-builder.js.map