/*
Copyright 2019 The Tekton Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package tektonpipeline

import (
	"context"
	"fmt"
	"os"
	"path/filepath"

	mf "github.com/manifestival/manifestival"
	"github.com/tektoncd/operator/pkg/apis/operator/v1alpha1"
	clientset "github.com/tektoncd/operator/pkg/client/clientset/versioned"
	tektonpipelinereconciler "github.com/tektoncd/operator/pkg/client/injection/reconciler/operator/v1alpha1/tektonpipeline"
	"github.com/tektoncd/operator/pkg/reconciler/common"
	oscommon "github.com/tektoncd/operator/pkg/reconciler/openshift/common"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/client-go/kubernetes"
	"knative.dev/pkg/logging"
	pkgreconciler "knative.dev/pkg/reconciler"
)

const enableMetrics = "enableMetrics"

// Reconciler implements controller.Reconciler for TektonPipeline resources.
type Reconciler struct {
	// kubeClientSet allows us to talk to the k8s for core APIs
	kubeClientSet kubernetes.Interface
	// operatorClientSet allows us to configure operator objects
	operatorClientSet clientset.Interface
	// manifest is empty, but with a valid client and logger. all
	// manifests are immutable, and any created during reconcile are
	// expected to be appended to this one, obviating the passing of
	// client & logger
	manifest mf.Manifest
	// Platform-specific behavior to affect the transform
	extension common.Extension
}

// Check that our Reconciler implements controller.Reconciler
var _ tektonpipelinereconciler.Interface = (*Reconciler)(nil)
var _ tektonpipelinereconciler.Finalizer = (*Reconciler)(nil)

// FinalizeKind removes all resources after deletion of a TektonPipeline.
func (r *Reconciler) FinalizeKind(ctx context.Context, original *v1alpha1.TektonPipeline) pkgreconciler.Event {
	logger := logging.FromContext(ctx)

	// List all TektonPipelines to determine if cluster-scoped resources should be deleted.
	tps, err := r.operatorClientSet.OperatorV1alpha1().TektonPipelines().List(ctx, metav1.ListOptions{})
	if err != nil {
		return fmt.Errorf("failed to list all TektonPipelines: %w", err)
	}

	for _, tp := range tps.Items {
		if tp.GetDeletionTimestamp().IsZero() {
			// Not deleting all TektonPipelines. Nothing to do here.
			return nil
		}
	}

	if err := r.extension.Finalize(ctx, original); err != nil {
		logger.Error("Failed to finalize platform resources", err)
	}
	logger.Info("Deleting cluster-scoped resources")
	manifest, err := r.installed(ctx, original)
	if err != nil {
		logger.Error("Unable to fetch installed manifest; no cluster-scoped resources will be finalized", err)
		return nil
	}
	if err := common.Uninstall(ctx, manifest, nil); err != nil {
		logger.Error("Failed to finalize platform resources", err)
	}
	return nil
}

// ReconcileKind compares the actual state with the desired, and attempts to
// converge the two.
func (r *Reconciler) ReconcileKind(ctx context.Context, tp *v1alpha1.TektonPipeline) pkgreconciler.Event {
	logger := logging.FromContext(ctx)
	tp.Status.InitializeConditions()
	tp.Status.ObservedGeneration = tp.Generation

	logger.Infow("Reconciling TektonPipeline", "status", tp.Status)
	if tp.GetName() != common.PipelineResourceName {
		msg := fmt.Sprintf("Resource ignored, Expected Name: %s, Got Name: %s",
			common.PipelineResourceName,
			tp.GetName(),
		)
		logger.Error(msg)
		tp.GetStatus().MarkInstallFailed(msg)
		return nil
	}

	if err := r.extension.PreReconcile(ctx, tp); err != nil {
		// If prereconcile updates the TektonConfig CR, it returns an error
		// to reconcile
		if err.Error() == "reconcile" {
			return err
		}
		tp.GetStatus().MarkInstallFailed(err.Error())
		return err
	}
	stages := common.Stages{
		r.appendPipelineTarget,
		r.transform,
		r.filterAndInstall,
		common.CheckDeployments,
	}
	manifest := r.manifest.Append()
	return stages.Execute(ctx, &manifest, tp)
}

// appendPipelineTarget mutates the passed manifest by appending one
// appropriate for the passed TektonComponent
func (r *Reconciler) appendPipelineTarget(ctx context.Context, manifest *mf.Manifest, comp v1alpha1.TektonComponent) error {
	if err := common.AppendTarget(ctx, manifest, comp); err != nil {
		return err
	}
	// add proxy configs to pipeline if any
	return addProxy(manifest)
}

func addProxy(manifest *mf.Manifest) error {
	koDataDir := os.Getenv(common.KoEnvKey)
	proxyLocation := filepath.Join(koDataDir, "webhook")
	return common.AppendManifest(manifest, proxyLocation)
}

// transform mutates the passed manifest to one with common, component
// and platform transformations applied
func (r *Reconciler) transform(ctx context.Context, manifest *mf.Manifest, comp v1alpha1.TektonComponent) error {
	//logger := logging.FromContext(ctx)
	images := common.ToLowerCaseKeys(common.ImagesFromEnv(common.PipelinesImagePrefix))
	instance := comp.(*v1alpha1.TektonPipeline)
	extra := []mf.Transformer{
		common.ApplyProxySettings,
		common.DeploymentImages(images),
		common.InjectLabelOnNamespace(),
	}
	extra = append(extra, r.extension.Transformers(instance)...)

	// TODO: Move this to OpenShift Extension
	tp := comp.(*v1alpha1.TektonPipeline)
	if len(tp.Spec.Params) != 0 && tp.Spec.Params[0].Name == enableMetrics && tp.Spec.Params[0].Value == "false" {
		extra = append(extra, oscommon.RemoveMonitoringLabel())
	}

	return common.Transform(ctx, manifest, instance, extra...)
}

func (r *Reconciler) installed(ctx context.Context, instance v1alpha1.TektonComponent) (*mf.Manifest, error) {
	// Create new, empty manifest with valid client and logger
	installed := r.manifest.Append()
	// TODO: add ingress, etc
	stages := common.Stages{r.appendPipelineTarget, r.transform}
	err := stages.Execute(ctx, &installed, instance)
	return &installed, err
}

var (
	serviceMonitor mf.Predicate = mf.ByKind("ServiceMonitor")
	smRole         mf.Predicate = mf.All(mf.ByKind("Role"), mf.ByName("openshift-pipelines-read"))
	smRoleBinding  mf.Predicate = mf.All(mf.ByKind("RoleBinding"), mf.ByName("openshift-pipelines-prometheus-k8s-read-binding"))
)

func (r *Reconciler) filterAndInstall(ctx context.Context, manifest *mf.Manifest, comp v1alpha1.TektonComponent) error {
	tp := comp.(*v1alpha1.TektonPipeline)

	var installManifest, uninstallManifest mf.Manifest
	installManifest = *manifest

	// TODO: Move this to OpenShift Extension
	if len(tp.Spec.Params) != 0 && tp.Spec.Params[0].Name == enableMetrics && tp.Spec.Params[0].Value == "false" {
		installManifest = manifest.Filter(mf.Not(mf.Any(serviceMonitor, smRole, smRoleBinding)))
		uninstallManifest = manifest.Filter(mf.Any(serviceMonitor, smRole, smRoleBinding))
	}

	if err := common.Install(ctx, &installManifest, tp); err != nil {
		return err
	}
	if err := common.Uninstall(ctx, &uninstallManifest, tp); err != nil {
		return err
	}

	return nil
}
