<!--
---
linkTitle: "Trigger Templates"
weight: 3
---
-->
# TriggerTemplates

A `TriggerTemplate` is a resource that can template resources.
`TriggerTemplate`s have [parameters](#parameters) that can be substituted
**anywhere** within the resource template.

<!-- FILE: examples/triggertemplates/triggertemplate.yaml -->
```YAML
apiVersion: triggers.tekton.dev/v1alpha1
kind: TriggerTemplate
metadata:
  name: pipeline-template
spec:
  params:
  - name: gitrevision
    description: The git revision
    default: master
  - name: gitrepositoryurl
    description: The git repository url
  - name: message
    description: The message to print
    default: This is the default message
  - name: contenttype
    description: The Content-Type of the event
  resourcetemplates:
  - apiVersion: tekton.dev/v1beta1
    kind: PipelineRun
    metadata:
      generateName: simple-pipeline-run-
    spec:
      pipelineRef:
        name: simple-pipeline
      params:
      - name: message
        value: $(tt.params.message)
      - name: contenttype
        value: $(tt.params.contenttype)
      resources:
      - name: git-source
        resourceSpec:
          type: git
          params:
          - name: revision
            value: $(tt.params.gitrevision)
          - name: url
            value: $(tt.params.gitrepositoryurl)
```

`TriggerTemplates` currently support the following [Tekton Pipelines](https://github.com/tektoncd/pipeline) resources:

v1alpha1          | v1beta1
------------------|---------
pipelines         | pipelines
pipelineruns      | pipelineruns
tasks             | tasks
taskruns          | taskruns
clustertasks      | clustertasks
conditions        |
pipelineresources |

Similar to
[Pipelines](https://github.com/tektoncd/pipeline/blob/master/docs/pipelines.md),`TriggerTemplate`s
do not do any actual work, but instead act as the blueprint for what resources
should be created.

If the namespace is omitted, it will be resolved to the `EventListener`'s
namespace.

The `$(uid)` variable is implicitly available throughout a `TriggerTemplate`'s
resource templates. A random string value is assigned to `$(uid)` like the
postfix generated by the Kubernetes `generateName` metadata field. One instance
where there is useful is when resources in a `TriggerTemplate` have internal
references.

The following are additional labels added to all `TriggerTemplate` resource
templates:

- To help with housekeeping/garbage collection: `tekton.dev/eventlistener`:
  `<EventListenerName>`
- To track resources created by the same event: `tekton.dev/triggers-eventid`:
  `<EventID>`

To enable support for arbitrary resource types, the resource templates are
internally resolved as byte blobs. As a result, validation on these resources is
only done at event processing time (rather than during `TriggerTemplate`
creation). :rotating_light: As of now, only Tekton resources can be defined
within a `TriggerTemplate` :rotating_light:

## Parameters

`TriggerTemplate`s can declare parameters that are supplied by a
`TriggerBinding` and/or `EventListener`. `params` must have a `name`, and can
have an optional `description` and `default` value.

`params` can be referenced in the `TriggerTemplate` using the following variable
substitution syntax, where `<name>` is the name of the parameter:

```YAML
$(tt.params.<name>)
```

`tt.params` can be referenced in the `resourceTemplates` section of a
`TriggerTemplate`. The purpose of `tt.params` is to make `TriggerTemplates`
reusable.

The value of the `default` field for each entry of the `params` array defined in a `TriggerTemplate` will
be applied if a corresponding entry in the `params` array in a `TriggerBinding` is either missing or cannot 
be satisfied in the cases where the entry's value comes from an HTTP header or body. 

## Best Practices

As of Tekton Pipelines version
[v0.8.0](https://github.com/tektoncd/pipeline/releases/tag/v0.8.0), users can
embed resource specs. It is a best practice to embed each resource specs in the
PipelineRun or TaskRun that uses the resource spec. Embedding the resource spec
avoids a race condition between creating and using resources.

## Templating Params

When templating parameters into resources, a simple replacement on the string
with the parameter name e.g. `$(tt.params.name)` is carried out.

This means that for simple string / number values, replacements in the
YAML should work fine.

If the string could begin with a number e.g. `012abcd`, it might be misinterpreted by YAML as a
number, which could cause an error, in which case you can put quotes around the
templated parameter key, and it should solve the problem.

## Escaping quoted strings.

TriggerTemplate parameter values were previously escaped by simply replacing
`"` with `\"` this could lead to problems when strings were already escaped, and
generating invalid resources from the TriggerTemplate.

No escaping is done on the templated variables, if you are inserting a JSON
object as a template var, then you should not put it within a quoted string.

This behaviour has been deprecated, if this breaks your templates, you can add
an annotation to the TriggerTemplate.

For example with the following JSON body:

```json
{
  "title": "this is \"demo\" body",
  "object": {
    "name": "testing"
  }
}
```

If you have a TriggerBinding that extracts `$(body.title)` then when it's
inserted into a TriggerTemplate it will be embedded as `this is a \"demo\"
body`.

By annotating the TriggerTemplate.

```yaml
apiVersion: triggers.tekton.dev/v1alpha1
kind: TriggerTemplate
metadata:
  name: escaped-tt
  annotations:
    triggers.tekton.dev/old-escape-quotes: "true"
spec:
  params:
  - name: title
    description: The title from the incoming body
```

This would pass the same body through as `this is a \""demo\"" body`, which is
invalid JSON, but, if you were to use a value with `$(body.object)` in a
template, and want it passed through as a quoted string, then this will work.

This might be useful if you want a string of JSON that you want to parse in a
command.
