// Copyright 2018 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package pb

import (
	"fmt"
	"reflect"
	"strings"
	"sync/atomic"

	"github.com/golang/protobuf/proto"
	descpb "github.com/golang/protobuf/protoc-gen-go/descriptor"
	exprpb "google.golang.org/genproto/googleapis/api/expr/v1alpha1"
)

// TypeDescription is a collection of type metadata relevant to expression
// checking and evaluation.
type TypeDescription struct {
	typeName string
	file     *FileDescription
	desc     *descpb.DescriptorProto
	metadata *atomic.Value
}

type typeMetadata struct {
	fields          map[string]*FieldDescription // fields by name (proto)
	fieldIndices    map[int][]*FieldDescription  // fields by Go struct idx
	fieldProperties *proto.StructProperties
	reflectedType   *reflect.Type
	reflectedVal    *reflect.Value
}

// FieldCount returns the number of fields declared within the type.
func (td *TypeDescription) FieldCount() int {
	// The number of keys in the field indices map corresponds to the number
	// of fields on the proto message.
	return len(td.getMetadata().fieldIndices)
}

// FieldByName returns the FieldDescription associated with a field name.
func (td *TypeDescription) FieldByName(name string) (*FieldDescription, bool) {
	fd, found := td.getMetadata().fields[name]
	return fd, found
}

// Name of the type.
func (td *TypeDescription) Name() string {
	return td.typeName
}

// ReflectType returns the reflected struct type of the generated proto struct.
func (td *TypeDescription) ReflectType() reflect.Type {
	if td.getMetadata().reflectedType == nil {
		return nil
	}
	return *td.getMetadata().reflectedType
}

// DefaultValue returns an empty instance of the proto message associated with the type.
func (td *TypeDescription) DefaultValue() proto.Message {
	refVal := td.getMetadata().reflectedVal
	if refVal == nil {
		return nil
	}
	return (*refVal).Interface().(proto.Message)
}

func (td *TypeDescription) getMetadata() *typeMetadata {
	meta, ok := td.metadata.Load().(*typeMetadata)
	if ok {
		return meta
	}
	meta = td.makeMetadata()
	td.metadata.Store(meta)
	return meta
}

func (td *TypeDescription) makeMetadata() *typeMetadata {
	refType := proto.MessageType(td.typeName)
	meta := &typeMetadata{
		fields:       make(map[string]*FieldDescription),
		fieldIndices: make(map[int][]*FieldDescription),
	}
	if refType != nil {
		// Set the reflected type if non-nil.
		meta.reflectedType = &refType

		// Unwrap the pointer reference for the sake of later checks.
		elemType := refType
		if elemType.Kind() == reflect.Ptr {
			elemType = elemType.Elem()
		}
		if elemType.Kind() == reflect.Struct {
			meta.fieldProperties = proto.GetProperties(elemType)
		}
		refVal := reflect.New(elemType)
		meta.reflectedVal = &refVal
	}

	fieldIndexMap := make(map[string]int)
	fieldDescMap := make(map[string]*descpb.FieldDescriptorProto)
	for i, f := range td.desc.Field {
		fieldDescMap[f.GetName()] = f
		fieldIndexMap[f.GetName()] = i
	}
	if meta.fieldProperties != nil {
		// This is a proper message type.
		for i, prop := range meta.fieldProperties.Prop {
			if strings.HasPrefix(prop.OrigName, "XXX_") {
				// Book-keeping fields generated by protoc start with XXX_
				continue
			}
			desc := fieldDescMap[prop.OrigName]
			fd := td.newFieldDesc(desc, prop, i)
			meta.fields[prop.OrigName] = fd
			meta.fieldIndices[i] = append(meta.fieldIndices[i], fd)
		}
		for _, oneofProp := range meta.fieldProperties.OneofTypes {
			desc := fieldDescMap[oneofProp.Prop.OrigName]
			fd := td.newOneofFieldDesc(desc, oneofProp, oneofProp.Field)
			meta.fields[oneofProp.Prop.OrigName] = fd
			meta.fieldIndices[oneofProp.Field] = append(meta.fieldIndices[oneofProp.Field], fd)
		}
	} else {
		for fieldName, desc := range fieldDescMap {
			fd := td.newMapFieldDesc(desc)
			meta.fields[fieldName] = fd
			index := fieldIndexMap[fieldName]
			meta.fieldIndices[index] = append(meta.fieldIndices[index], fd)
		}
	}
	return meta
}

func (td *TypeDescription) newFieldDesc(
	desc *descpb.FieldDescriptorProto,
	prop *proto.Properties,
	index int) *FieldDescription {
	fieldDesc := &FieldDescription{
		desc:      desc,
		index:     index,
		getter:    fmt.Sprintf("Get%s", prop.Name),
		prop:      prop,
		isProto3:  td.file.desc.GetSyntax() == "proto3",
		isWrapper: isWrapperType(desc),
	}
	if desc.GetType() == descpb.FieldDescriptorProto_TYPE_MESSAGE {
		typeName := sanitizeProtoName(desc.GetTypeName())
		fieldType, _ := td.file.pbdb.DescribeType(typeName)
		fieldDesc.td = fieldType
		return fieldDesc
	}
	return fieldDesc
}

func (td *TypeDescription) newOneofFieldDesc(
	desc *descpb.FieldDescriptorProto,
	oneofProp *proto.OneofProperties,
	index int) *FieldDescription {
	fieldDesc := td.newFieldDesc(desc, oneofProp.Prop, index)
	fieldDesc.oneofProp = oneofProp
	return fieldDesc
}

func (td *TypeDescription) newMapFieldDesc(desc *descpb.FieldDescriptorProto) *FieldDescription {
	return &FieldDescription{
		desc:     desc,
		index:    int(desc.GetNumber()),
		isProto3: td.file.desc.GetSyntax() == "proto3",
	}
}

func isWrapperType(desc *descpb.FieldDescriptorProto) bool {
	if desc.GetType() != descpb.FieldDescriptorProto_TYPE_MESSAGE {
		return false
	}
	switch sanitizeProtoName(desc.GetTypeName()) {
	case "google.protobuf.BoolValue",
		"google.protobuf.BytesValue",
		"google.protobuf.DoubleValue",
		"google.protobuf.FloatValue",
		"google.protobuf.Int32Value",
		"google.protobuf.Int64Value",
		"google.protobuf.StringValue",
		"google.protobuf.UInt32Value",
		"google.protobuf.UInt64Value":
		return true
	}
	return false
}

// FieldDescription holds metadata related to fields declared within a type.
type FieldDescription struct {
	// getter is the name of the accessor method to obtain the field value.
	getter string
	// isProto3 indicates whether the field is defined in a proto3 syntax.
	isProto3 bool
	// isWrapper indicates whether the field is a wrapper type.
	isWrapper bool

	// td is the type description for message typed fields.
	td *TypeDescription

	// proto descriptor data.
	desc      *descpb.FieldDescriptorProto
	index     int
	prop      *proto.Properties
	oneofProp *proto.OneofProperties
}

// CheckedType returns the type-definition used at type-check time.
func (fd *FieldDescription) CheckedType() *exprpb.Type {
	if fd.IsMap() {
		// Get the FieldDescriptors for the type arranged by their index within the
		// generated Go struct.
		fieldIndices := fd.getFieldIndicies()
		// Map keys and values are represented as repeated entries in a list.
		key := fieldIndices[0][0]
		val := fieldIndices[1][0]
		return &exprpb.Type{
			TypeKind: &exprpb.Type_MapType_{
				MapType: &exprpb.Type_MapType{
					KeyType:   key.typeDefToType(),
					ValueType: val.typeDefToType()}}}
	}
	if fd.IsRepeated() {
		return &exprpb.Type{
			TypeKind: &exprpb.Type_ListType_{
				ListType: &exprpb.Type_ListType{
					ElemType: fd.typeDefToType()}}}
	}
	return fd.typeDefToType()
}

// GetterName returns the accessor method name associated with the field
// on the proto generated struct.
func (fd *FieldDescription) GetterName() string {
	return fd.getter
}

// Index returns the field index within a reflected value.
func (fd *FieldDescription) Index() int {
	return fd.index
}

// IsEnum returns true if the field type refers to an enum value.
func (fd *FieldDescription) IsEnum() bool {
	return fd.desc.GetType() == descpb.FieldDescriptorProto_TYPE_ENUM
}

// IsMap returns true if the field is of map type.
func (fd *FieldDescription) IsMap() bool {
	if !fd.IsRepeated() || !fd.IsMessage() {
		return false
	}
	if fd.td == nil {
		return false
	}
	return fd.td.desc.GetOptions().GetMapEntry()
}

// IsMessage returns true if the field is of message type.
func (fd *FieldDescription) IsMessage() bool {
	return fd.desc.GetType() == descpb.FieldDescriptorProto_TYPE_MESSAGE
}

// IsOneof returns true if the field is declared within a oneof block.
func (fd *FieldDescription) IsOneof() bool {
	if fd.desc != nil {
		return fd.desc.OneofIndex != nil
	}
	return fd.oneofProp != nil
}

// IsRepeated returns true if the field is a repeated value.
//
// This method will also return true for map values, so check whether the
// field is also a map.
func (fd *FieldDescription) IsRepeated() bool {
	return *fd.desc.Label == descpb.FieldDescriptorProto_LABEL_REPEATED
}

// IsWrapper returns true if the field type is a primitive wrapper type.
func (fd *FieldDescription) IsWrapper() bool {
	return fd.isWrapper
}

// OneofType returns the reflect.Type value of a oneof field.
//
// Oneof field values are wrapped in a struct which contains one field whose
// value is a proto.Message.
func (fd *FieldDescription) OneofType() reflect.Type {
	return fd.oneofProp.Type
}

// OrigName returns the snake_case name of the field as it was declared within
// the proto. This is the same name format that is expected within expressions.
func (fd *FieldDescription) OrigName() string {
	if fd.desc != nil && fd.desc.Name != nil {
		return *fd.desc.Name
	}
	return fd.prop.OrigName
}

// Name returns the CamelCase name of the field within the proto-based struct.
func (fd *FieldDescription) Name() string {
	return fd.prop.Name
}

// SupportsPresence returns true if the field supports presence detection.
func (fd *FieldDescription) SupportsPresence() bool {
	return !fd.IsRepeated() && (fd.IsMessage() || !fd.isProto3)
}

// String returns a struct-like field definition string.
func (fd *FieldDescription) String() string {
	return fmt.Sprintf("%s %s `oneof=%t`",
		fd.TypeName(), fd.OrigName(), fd.IsOneof())
}

// Type returns the TypeDescription for the field.
func (fd *FieldDescription) Type() *TypeDescription {
	return fd.td
}

// TypeName returns the type name of the field.
func (fd *FieldDescription) TypeName() string {
	return sanitizeProtoName(fd.desc.GetTypeName())
}

func (fd *FieldDescription) getFieldIndicies() map[int][]*FieldDescription {
	return fd.td.getMetadata().fieldIndices
}

func (fd *FieldDescription) typeDefToType() *exprpb.Type {
	if fd.IsMessage() {
		if wk, found := CheckedWellKnowns[fd.TypeName()]; found {
			return wk
		}
		return checkedMessageType(fd.TypeName())
	}
	if fd.IsEnum() {
		return checkedInt
	}
	if p, found := CheckedPrimitives[fd.desc.GetType()]; found {
		return p
	}
	return CheckedPrimitives[fd.desc.GetType()]
}

func checkedMessageType(name string) *exprpb.Type {
	return &exprpb.Type{
		TypeKind: &exprpb.Type_MessageType{MessageType: name}}
}

func checkedPrimitive(primitive exprpb.Type_PrimitiveType) *exprpb.Type {
	return &exprpb.Type{
		TypeKind: &exprpb.Type_Primitive{Primitive: primitive}}
}

func checkedWellKnown(wellKnown exprpb.Type_WellKnownType) *exprpb.Type {
	return &exprpb.Type{
		TypeKind: &exprpb.Type_WellKnown{WellKnown: wellKnown}}
}

func checkedWrap(t *exprpb.Type) *exprpb.Type {
	return &exprpb.Type{
		TypeKind: &exprpb.Type_Wrapper{Wrapper: t.GetPrimitive()}}
}
