package bitbucketbuildtrigger

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strings"

	"github.com/quay/config-tool/pkg/lib/shared"
)

// Validate checks the configuration settings for this field group
func (fg *BitbucketBuildTriggerFieldGroup) Validate(opts shared.Options) []shared.ValidationError {

	fgName := "BitbucketBuildTrigger"

	// Make empty errors
	errors := []shared.ValidationError{}

	// If build suppport is off, dont validate
	if !fg.FeatureBuildSupport {
		return errors
	}

	// If bitbucket build support is off, dont validate
	if !fg.FeatureBitbucketBuild {
		return errors
	}

	// Make sure config is set up correctly
	if fg.BitbucketTriggerConfig == nil {
		newError := shared.ValidationError{
			Tags:       []string{"BITBUCKET_TRIGGER_CONFIG"},
			FieldGroup: fgName,
			Message:    "BITBUCKET_TRIGGER_CONFIG is required",
		}
		errors = append(errors, newError)
		return errors
	}

	// Check for consumer key
	if fg.BitbucketTriggerConfig.ConsumerKey == "" {
		newError := shared.ValidationError{
			Tags:       []string{"BITBUCKET_TRIGGER_CONFIG.CONSUMER_KEY"},
			FieldGroup: fgName,
			Message:    "BITBUCKET_TRIGGER_CONFIG.CONSUMER_KEY is required",
		}
		errors = append(errors, newError)
	}

	// Check consumer secret
	if fg.BitbucketTriggerConfig.ConsumerSecret == "" {
		newError := shared.ValidationError{
			Tags:       []string{"BITBUCKET_TRIGGER_CONFIG.CONSUMER_SECRET"},
			FieldGroup: fgName,
			Message:    "BITBUCKET_TRIGGER_CONFIG.CONSUMER_SECRET is required",
		}
		errors = append(errors, newError)
	}

	// Check OAuth credentials
	if !ValidateBitbucketOAuth(fg.BitbucketTriggerConfig.ConsumerKey, fg.BitbucketTriggerConfig.ConsumerSecret) {
		newError := shared.ValidationError{
			Tags:       []string{"BITBUCKET_TRIGGER_CONFIG.CONSUMER_ID", "BITBUCKET_TRIGGER_CONFIG.CONSUMER_SECRET"},
			FieldGroup: fgName,
			Message:    "Cannot validate BITBUCKET_TRIGGER_CONFIG credentials",
		}
		errors = append(errors, newError)
	}

	// Return errors
	return errors

}

// ValidateBitbucketOAuth checks that the Bitbucker OAuth credentials are correct
func ValidateBitbucketOAuth(clientID, clientSecret string) bool {

	// Generated by curl-to-Go: https://mholt.github.io/curl-to-go
	body := strings.NewReader(`grant_type=authorization_code&code={code}`)
	req, _ := http.NewRequest("POST", "https://bitbucket.org/site/oauth2/access_token", body)

	req.SetBasicAuth(clientID, clientSecret)
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")

	resp, _ := http.DefaultClient.Do(req)

	respBody, _ := ioutil.ReadAll(resp.Body)

	// Load response into json
	var responseJSON map[string]interface{}
	_ = json.Unmarshal(respBody, &responseJSON)

	// If the error isnt unauthorized
	if responseJSON["error_description"] == "The specified code is not valid." {
		return true
	}

	return false

}
