// Copyright 2018 Google LLC All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package gcrane

import (
	"bytes"
	"context"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
	"time"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-containerregistry/pkg/internal/retry"
	"github.com/google/go-containerregistry/pkg/logs"
	"github.com/google/go-containerregistry/pkg/name"
	"github.com/google/go-containerregistry/pkg/v1/google"
	"github.com/google/go-containerregistry/pkg/v1/remote/transport"
	"github.com/google/go-containerregistry/pkg/v1/types"
)

func mustRepo(s string) name.Repository {
	repo, err := name.NewRepository(s, name.WeakValidation)
	if err != nil {
		panic(err)
	}
	return repo
}

func TestRename(t *testing.T) {
	c := copier{
		srcRepo: mustRepo("gcr.io/foo"),
		dstRepo: mustRepo("gcr.io/bar"),
	}

	got, err := c.rename(mustRepo("gcr.io/foo/sub/repo"))
	if err != nil {
		t.Fatalf("unexpected err: %v", err)
	}
	want := mustRepo("gcr.io/bar/sub/repo")

	if want.String() != got.String() {
		t.Errorf("%s != %s", want, got)
	}
}

func TestSubtractStringLists(t *testing.T) {
	cases := []struct {
		minuend    []string
		subtrahend []string
		result     []string
	}{{
		minuend:    []string{"a", "b", "c"},
		subtrahend: []string{"a"},
		result:     []string{"b", "c"},
	}, {
		minuend:    []string{"a", "a", "a"},
		subtrahend: []string{"a", "b"},
		result:     []string{},
	}, {
		minuend:    []string{},
		subtrahend: []string{"a", "b"},
		result:     []string{},
	}, {
		minuend:    []string{"a", "b"},
		subtrahend: []string{},
		result:     []string{"a", "b"},
	}}

	for _, tc := range cases {
		want, got := tc.result, subtractStringLists(tc.minuend, tc.subtrahend)
		if diff := cmp.Diff(want, got); diff != "" {
			t.Errorf("subtracting string lists: %v - %v: (-want +got)\n%s", tc.minuend, tc.subtrahend, diff)
		}
	}
}

func TestDiffImages(t *testing.T) {
	cases := []struct {
		want map[string]google.ManifestInfo
		have map[string]google.ManifestInfo
		need map[string]google.ManifestInfo
	}{{
		// Have everything we need.
		want: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "c"},
			},
		},
		have: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "c"},
			},
		},
		need: map[string]google.ManifestInfo{},
	}, {
		// Missing image a.
		want: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "c", "d"},
			},
		},
		have: map[string]google.ManifestInfo{},
		need: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "c", "d"},
			},
		},
	}, {
		// Missing tags "b" and "d"
		want: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "c", "d"},
			},
		},
		have: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"c"},
			},
		},
		need: map[string]google.ManifestInfo{
			"a": {
				Tags: []string{"b", "d"},
			},
		},
	}, {
		// Make sure all properties get copied over.
		want: map[string]google.ManifestInfo{
			"a": {
				Size:      123,
				MediaType: string(types.DockerManifestSchema2),
				Created:   time.Date(1992, time.January, 7, 6, 40, 00, 5e8, time.UTC),
				Uploaded:  time.Date(2018, time.November, 29, 4, 13, 30, 5e8, time.UTC),
				Tags:      []string{"b", "c", "d"},
			},
		},
		have: map[string]google.ManifestInfo{},
		need: map[string]google.ManifestInfo{
			"a": {
				Size:      123,
				MediaType: string(types.DockerManifestSchema2),
				Created:   time.Date(1992, time.January, 7, 6, 40, 00, 5e8, time.UTC),
				Uploaded:  time.Date(2018, time.November, 29, 4, 13, 30, 5e8, time.UTC),
				Tags:      []string{"b", "c", "d"},
			},
		},
	}}

	for _, tc := range cases {
		want, got := tc.need, diffImages(tc.want, tc.have)
		if diff := cmp.Diff(want, got); diff != "" {
			t.Errorf("diffing images: %v - %v: (-want +got)\n%s", tc.want, tc.have, diff)
		}
	}
}

// Test that our backoff works the way we expect.
func TestBackoff(t *testing.T) {
	backoff := GCRBackoff()

	if d := backoff.Step(); d > 10*time.Second {
		t.Errorf("Duration too long: %v", d)
	}
	if d := backoff.Step(); d > 100*time.Second {
		t.Errorf("Duration too long: %v", d)
	}
	if d := backoff.Step(); d > 1000*time.Second {
		t.Errorf("Duration too long: %v", d)
	}
	if s := backoff.Steps; s != 0 {
		t.Errorf("backoff.Steps should be 0, got %d", s)
	}
}

func TestErrors(t *testing.T) {
	if hasStatusCode(nil, http.StatusOK) {
		t.Fatal("nil error should not have any status code")
	}
	if !hasStatusCode(&transport.Error{StatusCode: http.StatusOK}, http.StatusOK) {
		t.Fatal("200 should be 200")
	}
	if hasStatusCode(&transport.Error{StatusCode: http.StatusOK}, http.StatusNotFound) {
		t.Fatal("200 should not be 404")
	}

	if isServerError(nil) {
		t.Fatal("nil should not be server error")
	}
	if isServerError(fmt.Errorf("i am a string")) {
		t.Fatal("string should not be server error")
	}
	if !isServerError(&transport.Error{StatusCode: http.StatusServiceUnavailable}) {
		t.Fatal("503 should be server error")
	}
	if isServerError(&transport.Error{StatusCode: http.StatusTooManyRequests}) {
		t.Fatal("429 should not be server error")
	}
}

func TestRetryErrors(t *testing.T) {
	// We log a warning during retries, so we can tell if somethign retried by checking logs.Warn.
	var b bytes.Buffer
	logs.Warn.SetOutput(&b)

	err := backoffErrors(retry.Backoff{
		Duration: 1 * time.Millisecond,
		Steps:    3,
	}, func() error {
		return &transport.Error{StatusCode: http.StatusTooManyRequests}
	})

	if err == nil {
		t.Fatal("backoffErrors should return internal err, got nil")
	}
	if te, ok := err.(*transport.Error); !ok {
		t.Fatalf("backoffErrors should return internal err, got different error: %v", err)
	} else if te.StatusCode != http.StatusTooManyRequests {
		t.Fatalf("backoffErrors should return internal err, got different status code: %v", te.StatusCode)
	}

	if b.Len() == 0 {
		t.Fatal("backoffErrors didn't log to logs.Warn")
	}
}

func TestBadInputs(t *testing.T) {
	t.Parallel()
	invalid := "@@@@@@"

	// Create a valid image reference that will fail with not found.
	s := httptest.NewServer(http.NotFoundHandler())
	u, err := url.Parse(s.URL)
	if err != nil {
		t.Fatal(err)
	}
	valid404 := fmt.Sprintf("%s/some/image", u.Host)

	ctx := context.Background()

	for _, tc := range []struct {
		desc string
		err  error
	}{
		{"Copy(invalid, invalid)", Copy(invalid, invalid)},
		{"Copy(404, invalid)", Copy(valid404, invalid)},
		{"Copy(404, 404)", Copy(valid404, valid404)},
		{"CopyRepository(invalid, invalid)", CopyRepository(ctx, invalid, invalid)},
		{"CopyRepository(404, invalid)", CopyRepository(ctx, valid404, invalid)},
		{"CopyRepository(404, 404)", CopyRepository(ctx, valid404, valid404, WithJobs(1))},
	} {
		if tc.err == nil {
			t.Errorf("%s: expected err, got nil", tc.desc)
		}
	}
}
