"use strict";
/**
 * @license
 * Copyright 2016 Palantir Technologies, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var tsutils_1 = require("tsutils");
var ts = require("typescript");
var Lint = require("../index");
var Rule = (function (_super) {
    tslib_1.__extends(Rule, _super);
    function Rule() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /* tslint:enable:object-literal-sort-keys */
    Rule.FAILURE_STRING = function (name, message) {
        return name + " is deprecated" + (message === "" ? "." : ": " + message.trim());
    };
    Rule.prototype.applyWithProgram = function (sourceFile, program) {
        return this.applyWithFunction(sourceFile, function (ctx) { return walk(ctx, program.getTypeChecker()); });
    };
    return Rule;
}(Lint.Rules.TypedRule));
/* tslint:disable:object-literal-sort-keys */
Rule.metadata = {
    ruleName: "deprecation",
    description: "Warns when deprecated APIs are used.",
    descriptionDetails: (_a = ["Any usage of an identifier\n            with the @deprecated JSDoc annotation will trigger a warning.\n            See http://usejsdoc.org/tags-deprecated.html"], _a.raw = ["Any usage of an identifier\n            with the @deprecated JSDoc annotation will trigger a warning.\n            See http://usejsdoc.org/tags-deprecated.html"], Lint.Utils.dedent(_a)),
    rationale: "Deprecated APIs should be avoided, and usage updated.",
    optionsDescription: "",
    options: null,
    optionExamples: [],
    type: "maintainability",
    typescriptOnly: false,
    requiresTypeInfo: true,
};
exports.Rule = Rule;
function walk(ctx, tc) {
    return ts.forEachChild(ctx.sourceFile, function cb(node) {
        if (tsutils_1.isIdentifier(node)) {
            if (!isDeclaration(node)) {
                var deprecation = getDeprecation(node, tc);
                if (deprecation !== undefined) {
                    ctx.addFailureAtNode(node, Rule.FAILURE_STRING(node.text, deprecation));
                }
            }
        }
        else {
            return ts.forEachChild(node, cb);
        }
    });
}
function isDeclaration(identifier) {
    var parent = identifier.parent;
    switch (parent.kind) {
        case ts.SyntaxKind.ClassDeclaration:
        case ts.SyntaxKind.ClassExpression:
        case ts.SyntaxKind.InterfaceDeclaration:
        case ts.SyntaxKind.TypeParameter:
        case ts.SyntaxKind.FunctionExpression:
        case ts.SyntaxKind.FunctionDeclaration:
        case ts.SyntaxKind.LabeledStatement:
        case ts.SyntaxKind.JsxAttribute:
        case ts.SyntaxKind.MethodDeclaration:
        case ts.SyntaxKind.MethodSignature:
        case ts.SyntaxKind.PropertySignature:
        case ts.SyntaxKind.TypeAliasDeclaration:
        case ts.SyntaxKind.GetAccessor:
        case ts.SyntaxKind.SetAccessor:
        case ts.SyntaxKind.EnumDeclaration:
            return true;
        case ts.SyntaxKind.VariableDeclaration:
        case ts.SyntaxKind.TypeAliasDeclaration:
        case ts.SyntaxKind.Parameter:
        case ts.SyntaxKind.ModuleDeclaration:
        case ts.SyntaxKind.PropertyDeclaration:
        case ts.SyntaxKind.PropertyAssignment:
        case ts.SyntaxKind.EnumMember:
            return parent.name === identifier;
        case ts.SyntaxKind.BindingElement:
        case ts.SyntaxKind.ExportSpecifier:
        case ts.SyntaxKind.ImportSpecifier:
            // return true for `b` in `import {a as b} from "foo"`
            return parent.name === identifier &&
                parent.propertyName !== undefined;
        default:
            return false;
    }
}
function getDeprecation(node, tc) {
    var symbol = tc.getSymbolAtLocation(node);
    if (symbol !== undefined && Lint.isSymbolFlagSet(symbol, ts.SymbolFlags.Alias)) {
        symbol = tc.getAliasedSymbol(symbol);
    }
    if (symbol !== undefined) {
        return getDeprecationValue(symbol);
    }
    return undefined;
}
function getDeprecationValue(symbol) {
    if (symbol.getJsDocTags !== undefined) {
        for (var _i = 0, _a = symbol.getJsDocTags(); _i < _a.length; _i++) {
            var tag = _a[_i];
            if (tag.name === "deprecated") {
                return tag.text;
            }
        }
        return undefined;
    }
    // for compatibility with typescript@<2.3.0
    return getDeprecationFromDeclarations(symbol.declarations);
}
function getDeprecationFromDeclarations(declarations) {
    if (declarations === undefined) {
        return undefined;
    }
    var declaration;
    for (var _i = 0, declarations_1 = declarations; _i < declarations_1.length; _i++) {
        declaration = declarations_1[_i];
        if (tsutils_1.isBindingElement(declaration)) {
            declaration = tsutils_1.getDeclarationOfBindingElement(declaration);
        }
        if (tsutils_1.isVariableDeclaration(declaration)) {
            declaration = declaration.parent;
        }
        if (tsutils_1.isVariableDeclarationList(declaration)) {
            declaration = declaration.parent;
        }
        for (var _a = 0, _b = declaration.getChildren(); _a < _b.length; _a++) {
            var child = _b[_a];
            if (child.kind !== ts.SyntaxKind.JSDocComment) {
                break;
            }
            if (child.tags === undefined) {
                continue;
            }
            for (var _c = 0, _d = child.tags; _c < _d.length; _c++) {
                var tag = _d[_c];
                if (tag.tagName.text === "deprecated") {
                    return tag.comment === undefined ? "" : tag.comment;
                }
            }
        }
    }
    return undefined;
}
var _a;
