/**
 * Service which provides helper methods for working with the vulnerability system.
  */
angular.module('quay').factory('VulnerabilityService', ['Config', 'ApiService', 'ImageMetadataService',
    function(Config, ApiService, ImageMetadataService) {

  var vulnService = {};
  vulnService.LEVELS = window.__vuln_priority;

  vulnService.getUnadjustedScoreOf = function(vuln) {
    var severity = vulnService.LEVELS[vuln['Severity']];
    return severity.score;
  };

  vulnService.getCVSSScoreOf = function(vuln) {
    if (vuln.Metadata && vuln.Metadata.NVD && vuln.Metadata.NVD.CVSSv2 && vuln.Metadata.NVD.CVSSv2.Score) {
      return vuln.Metadata.NVD.CVSSv2.Score;
    }

    return null;
  }

  vulnService.buildVulnerabilitiesInfo = function(manifest, resp) {
    var levels = vulnService.getLevels();

    var severityCountMap = {};
    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
    });

    var fixable = [];
    var vulnerabilities = [];

    var featuresInfo = vulnService.buildFeaturesInfo(manifest, resp);
    featuresInfo.features.forEach(function(feature) {
      if (feature.vulnerabilities) {
        vulnerabilities = vulnerabilities.concat(feature.vulnerabilities);
        fixable = fixable.concat(feature.fixable);

        feature.severityBreakdown.forEach(function(level) {
          severityCountMap[level['index']] += level['value'];
        });
      }
    });

    var severityBreakdown = [];
    levels.forEach(function(level) {
      if (severityCountMap[level['index']]) {
        severityBreakdown.push({
          'index': level['index'],
          'label': level['title'],
          'value': severityCountMap[level['index']],
          'color': level['color']
        });
      }
    });

    return {
      'vulnerabilities': vulnerabilities,
      'fixable': fixable,
      'severityBreakdown': severityBreakdown,
      'features': featuresInfo.features,
    }
  };

  vulnService.buildVulnerabilitiesInfoForFeature = function(manifest, feature) {
    var levels = vulnService.getLevels();

    var vulnerabilities = [];
    var fixable = [];

    var severityCountMap = {};
    var fixableCountMap = {};

    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
      fixableCountMap[level['index']] = 0;
    });

    var score = 0;
    var fixableScore = 0;

    var highestSeverityIndex = levels.length;

    if (feature.Vulnerabilities) {
      var addedByImageId = feature.AddedBy ? feature.AddedBy.split('.')[0] : null;

      feature.Vulnerabilities.forEach(function(vuln) {
        var severity = vulnService.LEVELS[vuln['Severity']];
        var cvssScore = vulnService.getCVSSScoreOf(vuln);
        var unadjustedScore = vulnService.getUnadjustedScoreOf(vuln);

        var currentScore = unadjustedScore;
        var scoreDivergence = null;

        // If the vulnerability has a CVSS score, ensure it is within 2 levels of the severity
        // score from the distro. If it is out of that band, then we have a score divergence
        // and use the distro's score directly.
        if (cvssScore != null) {
          if (cvssScore - unadjustedScore > 2) {
            scoreDivergence = 'adjusted-lower';
          } else if (unadjustedScore > cvssScore) {
            scoreDivergence = 'adjusted-higher';
          } else {
            currentScore = cvssScore;
          }
        }

        var exponentialScore = Math.pow(2, currentScore) + 0.1;
        var vuln_object = {
          'score': exponentialScore,
          'scoreDivergence': scoreDivergence,
          'severityInfo': severity,

          'cvssScore': cvssScore,
          'cvssColor': vulnService.getCVSSColor(cvssScore),

          'name': vuln.Name,
          'namespace': vuln.NamespaceName || vuln.Namespace,
          'description': vuln.Description,
          'link': vuln.Link,
          'severity': vuln.Severity,
          'metadata': vuln.Metadata,

          'featureName': feature.Name,
          'fixedInVersion': vuln.FixedBy,
          'introducedInVersion': feature.Version,

          'imageId': addedByImageId,
          'imageCommand': vulnService.imageComamandFor(manifest, addedByImageId),

          'expanded': false
        };

        // Save the highest vulnerability severity for this feature.
        highestSeverityIndex = Math.min(severity['index'], highestSeverityIndex);

        // Add the score and (if necessary) the fixable scores.
        score += exponentialScore;
        severityCountMap[severity['index']]++;
        vulnerabilities.push(vuln_object);

        if (vuln.FixedBy) {
          fixableCountMap[severity['index']]++;
          fixableScore += exponentialScore;
          fixable.push(vuln_object)
        }
      });
    }

    // Calculate the breakdown of the vulnerabilities by severity.
    var severityBreakdown = [];
    var fixableBreakdown = [];
    var leftoverBreakdown = [];

    levels.forEach(function(level) {
      if (severityCountMap[level['index']]) {
        severityBreakdown.push({
          'index': level['index'],
          'label': level['title'],
          'value': severityCountMap[level['index']],
          'color': level['color']
        });

        if (fixableCountMap[level['index']]) {
          fixableBreakdown.push({
            'index': level['index'],
            'label': level['title'],
            'value': fixableCountMap[level['index']],
            'color': level['color']
          });
        }

        var leftoverCount = severityCountMap[level['index']] - fixableCountMap[level['index']];
        if (leftoverCount) {
          leftoverBreakdown.push({
            'index': level['index'],
            'label': level['title'],
            'value': leftoverCount,
            'color': level['color']
          });
        }
      }
    });

    return {
      'vulnerabilities': vulnerabilities,
      'fixable': fixable,

      'severityBreakdown': severityBreakdown,
      'fixableBreakdown': fixableBreakdown,
      'leftoverBreakdown': leftoverBreakdown,

      'score': score,
      'fixableScore': fixableScore,

      'highestSeverity': levels[highestSeverityIndex],
    };
  };

  vulnService.imageComamandFor = function(manifest, addedByImageId) {
    if (!manifest || !addedByImageId) {
      return null;
    }
    return addedByImageId.includes(':')
      ? ImageMetadataService.getManifestCommand(manifest, addedByImageId)
      : ImageMetadataService.getImageCommand(manifest.image, addedByImageId);
  };

  vulnService.buildFeaturesInfo = function(manifest, resp) {
    var features = [];
    var severityCountMap = {};
    var highestFixableScore = 0;

    var levels = vulnService.getLevels();

    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
    });

    vulnService.forEachFeature(resp, function(feature) {
      // Calculate the scores and breakdowns for all the vulnerabilities under feature.
      var vulnerabilityInfo = vulnService.buildVulnerabilitiesInfoForFeature(manifest, feature);
      var addedByImageId = feature.AddedBy ? feature.AddedBy.split('.')[0] : null;

      var feature_obj = {
        'name': feature.Name,
        'namespace': feature.NamespaceName || feature.Namespace,
        'version': feature.Version,
        'addedBy': feature.AddedBy,
        'imageId': addedByImageId,
        'imageCommand': vulnService.imageComamandFor(manifest, addedByImageId),
        'vulnCount': vulnerabilityInfo.vulnerabilities.length,
        'severityBreakdown': vulnerabilityInfo.severityBreakdown,
        'fixableBreakdown': vulnerabilityInfo.fixableBreakdown,
        'leftoverBreakdown': vulnerabilityInfo.leftoverBreakdown,
        'score': vulnerabilityInfo.score,
        'fixableCount': vulnerabilityInfo.fixable.length,
        'leftoverCount': vulnerabilityInfo.vulnerabilities.length - vulnerabilityInfo.fixable.length,
        'fixableScore': vulnerabilityInfo.fixableScore,
        'leftoverScore': vulnerabilityInfo.score - vulnerabilityInfo.fixableScore,
        'primarySeverity': vulnerabilityInfo.severityBreakdown[0],
        'primaryLeftover': vulnerabilityInfo.leftoverBreakdown[0],
        'vulnerabilities': vulnerabilityInfo.vulnerabilities,
        'fixable': vulnerabilityInfo.fixable
      };

      if (vulnerabilityInfo.highestSeverity) {
        severityCountMap[vulnerabilityInfo.highestSeverity['index']]++;
      } else {
        // Ensures that features with no vulns are always at the bottom of the table in the
        // default sort by fixableScore.
        feature_obj['fixableScore'] = -1;
        feature_obj['leftoverScore'] = -1;
      }

      highestFixableScore = Math.max(highestFixableScore, vulnerabilityInfo.fixableScore);

      features.push(feature_obj);
    });

    // Calculate the breakdown of each severity level for the features.
    var totalCount = features.length;
    var severityBreakdown = [];

    levels.forEach(function(level) {
      if (!severityCountMap[level['index']]) {
        return;
      }

      totalCount -= severityCountMap[level['index']];
      severityBreakdown.push({
        'index': level['index'],
        'label': level['title'],
        'value': severityCountMap[level['index']],
        'color': level['color']
      });
    });

    if (totalCount > 0) {
      severityBreakdown.push({
        'index': levels.length,
        'label': 'None',
        'value': totalCount,
        'color': '#2FC98E'
      });
    }

    return {
      'features': features,
      'brokenFeaturesCount': features.length - totalCount,
      'fixableFeatureCount': features.filter(function(f) { return f.fixableScore > 0 }).length,
      'severityBreakdown': severityBreakdown,
      'highestFixableScore': highestFixableScore
    }
  };

  vulnService.loadImageVulnerabilities = function(repo, image_id, result, reject) {
    var params = {
      'imageid': image_id,
      'repository': repo.namespace + '/' + repo.name,
      'vulnerabilities': true,
    };

    ApiService.getRepoImageSecurity(null, params).then(result, reject);
  };

  vulnService.loadManifestVulnerabilities = function(repo, digest, result, reject) {
    var params = {
      'manifestref': digest,
      'repository': repo.namespace + '/' + repo.name,
      'vulnerabilities': true,
    };

    ApiService.getRepoManifestSecurity(null, params).then(result, reject);
  };

  vulnService.hasFeatures = function(resp) {
    return resp.data && resp.data.Layer && resp.data.Layer.Features && resp.data.Layer.Features.length;
  };

  vulnService.forEachFeature = function(resp, callback) {
    if (!vulnService.hasFeatures(resp)) {
      return;
    }

    resp.data.Layer.Features.forEach(callback);
  };

  vulnService.forEachVulnerability = function(resp, callback) {
    if (!vulnService.hasFeatures(resp)) {
      return;
    }

    vulnService.forEachFeature(resp, function(feature) {
      if (feature.Vulnerabilities) {
        feature.Vulnerabilities.forEach(callback);
      }
    });
  };

  var cvssSeverityMap = {};

  vulnService.getSeverityForCVSS = function(score) {
    if (cvssSeverityMap[score]) {
      return cvssSeverityMap[score];
    }

    var levels = vulnService.getLevels();
    for (var i = 0; i < levels.length; ++i) {
      if (score >= levels[i].score) {
        cvssSeverityMap[score] = levels[i];
        return levels[i];
      }
    }

    return vulnService.LEVELS['Unknown'];
  };

  vulnService.getCVSSColor = function(score) {
    if (score == null) {
      return null;
    }

    return vulnService.getSeverityForCVSS(score).color;
  };

  vulnService.getLevels = function() {
    var levels =  Object.keys(vulnService.LEVELS).map(function(key) {
      return vulnService.LEVELS[key];
    });
    return levels.sort(function(a, b) {
        return a.index - b.index;
    });
  };

  vulnService.parseVectorsString = function(vectorsString) {
    return vectorsString.split('/');
  };

  vulnService.getVectorTitle = function(vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    return vector.title;
  };

  vulnService.getVectorDescription = function(vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    return vector.description;
  };

  vulnService.getVectorClasses = function(option, vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    var classes = '';
    if (option.id == parts[1]) {
      classes += 'current-vector ';
    } else {
      classes += 'not-current-vector ';
    }

    classes += option.severity;
    return classes;
  };

  vulnService.getVectorOptions = function(vectorString) {
    var parts = vectorString.split(':');
    return vulnService.NVD_VECTORS[parts[0]].values;
  };

  vulnService.NVD_VECTORS = {
    'AV': {
      'title': 'Access Vector',
      'description': 'This metric reflects how the vulnerability is exploited. The more remote an attacker can be to attack a host, the greater the vulnerability score.',
      'values': [
        {
          'id': 'N',
          'title': 'Network',
          'description': 'A vulnerability exploitable with network access means the vulnerable software is bound to the network stack and the attacker does not require local network access or local access.  Such a vulnerability is often termed "remotely exploitable".  An example of a network attack is an RPC buffer overflow.',
          'severity': 'high'
        },
        {
          'id': 'A',
          'title': 'Adjacent Network',
          'description': 'A vulnerability exploitable with adjacent network access requires the attacker to have access to either the broadcast or collision domain of the vulnerable software.  Examples of local networks include local IP subnet, Bluetooth, IEEE 802.11, and local Ethernet segment.',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Local',
          'description': 'A vulnerability exploitable with only local access requires the attacker to have either physical access to the vulnerable system or a local (shell) account. Examples of locally exploitable vulnerabilities are peripheral attacks such as Firewire/USB DMA attacks, and local privilege escalations (e.g., sudo).',
          'severity': 'low'
        }
      ]
    },

    'AC': {
      'title': 'Access Complexity',
      'description': 'This metric measures the complexity of the attack required to exploit the vulnerability once an attacker has gained access to the target system. For example, consider a buffer overflow in an Internet service: once the target system is located, the attacker can launch an exploit at will.',
      'values': [
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Specialized access conditions or extenuating circumstances do not exist making this easy to exploit',
          'severity': 'high'
        },
        {
          'id': 'M',
          'title': 'Medium',
          'description': 'The access conditions are somewhat specialized making this somewhat difficult to exploit',
          'severity': 'medium'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Specialized access conditions exist making this harder to exploit',
          'severity': 'low'
        }
      ]
    },

    'Au': {
      'title': 'Authentication',
      'description': 'This metric measures the number of times an attacker must authenticate to a target in order to exploit a vulnerability. This metric does not gauge the strength or complexity of the authentication process, only that an attacker is required to provide credentials before an exploit may occur.   The fewer authentication instances that are required, the higher the vulnerability score.',
      'values': [
        {
          'id': 'N',
          'title': 'None',
          'description': 'Authentication is not required to exploit the vulnerability.',
          'severity': 'high'
        },
        {
          'id': 'S',
          'title': 'Single',
          'description': 'The vulnerability requires an attacker to be logged into the system (such as at a command line or via a desktop session or web interface).',
          'severity': 'medium'
        },
        {
          'id': 'M',
          'title': 'Multiple',
          'description': 'Exploiting the vulnerability requires that the attacker authenticate two or more times, even if the same credentials are used each time. An example is an attacker authenticating to an operating system in addition to providing credentials to access an application hosted on that system.',
          'severity': 'low'
        }
      ]
    },

    'C': {
      'title': 'Confidentiality Impact',
      'description': 'This metric measures the impact on confidentiality of a successfully exploited vulnerability. Confidentiality refers to limiting information access and disclosure to only authorized users, as well as preventing access by, or disclosure to, unauthorized ones. Increased confidentiality impact increases the vulnerability score.',
      'values': [
        {
          'id': 'C',
          'title': 'Complete',
          'description': 'There is total information disclosure, resulting in all system files being revealed. The attacker is able to read all of the system\'s data (memory, files, etc.)',
          'severity': 'high'
        },
        {
          'id': 'P',
          'title': 'Partial',
          'description': 'There is considerable informational disclosure. Access to some system files is possible, but the attacker does not have control over what is obtained, or the scope of the loss is constrained. An example is a vulnerability that divulges only certain tables in a database.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the confidentiality of the system.',
          'severity': 'low'
        }
      ]
    },

    'I': {
      'title': 'Integrity Impact',
      'description': 'This metric measures the impact to integrity of a successfully exploited vulnerability. Integrity refers to the trustworthiness and guaranteed veracity of information. Increased integrity impact increases the vulnerability score.',
      'values': [
        {
          'id': 'C',
          'title': 'Complete',
          'description': 'There is a total compromise of system integrity. There is a complete loss of system protection, resulting in the entire system being compromised. The attacker is able to modify any files on the target system',
          'severity': 'high'
        },
        {
          'id': 'P',
          'title': 'Partial',
          'description': 'Modification of some system files or information is possible, but the attacker does not have control over what can be modified, or the scope of what the attacker can affect is limited. For example, system or application files may be overwritten or modified, but either the attacker has no control over which files are affected or the attacker can modify files within only a limited context or scope.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the integrity of the system.',
          'severity': 'low'
        }
      ]
    },

    'A': {
      'title': 'Availability Impact',
      'description': 'This metric measures the impact to availability of a successfully exploited vulnerability. Availability refers to the accessibility of information resources. Attacks that consume network bandwidth, processor cycles, or disk space all impact the availability of a system. Increased availability impact increases the vulnerability score.',
      'values': [
        {
          'id': 'C',
          'title': 'Complete',
          'description': 'There is a total shutdown of the affected resource. The attacker can render the resource completely unavailable.',
          'severity': 'high'
        },
        {
          'id': 'P',
          'title': 'Partial',
          'description': 'There is reduced performance or interruptions in resource availability. An example is a network-based flood attack that permits a limited number of successful connections to an Internet service.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the availability of the system.',
          'severity': 'low'
        }
      ]
    }
  };

  return vulnService;
}]);

