// Copyright 2016 CoreOS, Inc
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"flag"
	"os"
	"os/signal"
	"syscall"

	log "github.com/sirupsen/logrus"

	"github.com/quay/jwtproxy"
	"github.com/quay/jwtproxy/config"

	_ "github.com/quay/jwtproxy/jwt/claims/static"
	_ "github.com/quay/jwtproxy/jwt/keyserver/keyregistry"
	_ "github.com/quay/jwtproxy/jwt/keyserver/keyregistry/keycache/memory"
	_ "github.com/quay/jwtproxy/jwt/keyserver/preshared"
	_ "github.com/quay/jwtproxy/jwt/noncestorage/local"
	_ "github.com/quay/jwtproxy/jwt/privatekey/autogenerated"
	_ "github.com/quay/jwtproxy/jwt/privatekey/preshared"
)

func main() {
	flag.CommandLine = flag.NewFlagSet(os.Args[0], flag.ExitOnError)
	flagConfigPath := flag.String("config", "", "Load configuration from the specified yaml file.")
	flagLogLevel := flag.String("log-level", "info", "Define the logging level.")
	flag.Parse()

	// Load configuration.
	config, err := config.Load(*flagConfigPath)
	if err != nil {
		flag.Usage()
		log.Fatalf("Failed to load configuration: %s", err)
	}

	// Initialize logging system.
	level, err := log.ParseLevel(*flagLogLevel)
	if err != nil {
		log.Fatalf("Failed to parse the log level: %s", err)
	}
	log.SetLevel(level)

	// Run proxies until SIGINT/SIGTERM is received and then shutdown gracefully.
	run(config)
}

func run(config *config.Config) {
	// Nothing to run? Abort.
	var verifierEnabled bool
	for _, verifierCfg := range config.VerifierProxies {
		if verifierCfg.Enabled {
			verifierEnabled = true
			break
		}
	}

	if !verifierEnabled && !config.SignerProxy.Enabled {
		log.Error("No proxy is enabled. Terminating.")
		return
	}

	// Create shutdown channel and make it listen to SIGINT and SIGTERM.
	shutdown := make(chan os.Signal)
	signal.Notify(shutdown, syscall.SIGINT, syscall.SIGTERM)

	// Run proxies.
	stopper, abort := jwtproxy.RunProxies(config)

	// Wait for stop signal.
	select {
	case <-shutdown:
		log.Info("Received stop signal. Stopping gracefully...")
	case aborted := <-abort:
		log.Error(aborted)
	}

	stopped := stopper.Stop()

	// Restore the original behavior in case we need to force shutdown.
	signal.Reset(syscall.SIGINT, syscall.SIGTERM)

	// Wait for everything to stop.
	<-stopped
}
