/**
 * Service which provides helper methods for working with the vulnerability system.
  */
angular.module('quay').factory('VulnerabilityService', ['Config', 'ApiService', 'ImageMetadataService',
    function(Config, ApiService, ImageMetadataService) {

  var vulnService = {};
  vulnService.LEVELS = window.__vuln_priority;

  vulnService.getUnadjustedScoreOf = function(vuln) {
    var severity = vulnService.LEVELS[vuln['Severity']];
    return severity.score;
  };

  vulnService.getCVSSScoreOf = function(vuln) {
    if (vuln.Metadata && vuln.Metadata.NVD && vuln.Metadata.NVD.CVSSv3 && vuln.Metadata.NVD.CVSSv3.Score) {
      return vuln.Metadata.NVD.CVSSv3.Score;
    }
    return null;
  }

  vulnService.buildVulnerabilitiesInfo = function(manifest, resp) {
    var levels = vulnService.getLevels();

    var severityCountMap = {};
    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
    });

    var fixable = [];
    var vulnerabilities = [];

    var featuresInfo = vulnService.buildFeaturesInfo(manifest, resp);
    featuresInfo.features.forEach(function(feature) {
      if (feature.vulnerabilities) {
        vulnerabilities = vulnerabilities.concat(feature.vulnerabilities);
        fixable = fixable.concat(feature.fixable);

        feature.severityBreakdown.forEach(function(level) {
          severityCountMap[level['index']] += level['value'];
        });
      }
    });

    var severityBreakdown = [];
    levels.forEach(function(level) {
      if (severityCountMap[level['index']]) {
        severityBreakdown.push({
          'index': level['index'],
          'label': level['title'],
          'value': severityCountMap[level['index']],
          'color': level['color']
        });
      }
    });

    return {
      'vulnerabilities': vulnerabilities,
      'fixable': fixable,
      'severityBreakdown': severityBreakdown,
      'features': featuresInfo.features,
    }
  };

  vulnService.buildVulnerabilitiesInfoForFeature = function(manifest, feature) {
    var levels = vulnService.getLevels();

    var vulnerabilities = [];
    var fixable = [];

    var severityCountMap = {};
    var fixableCountMap = {};

    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
      fixableCountMap[level['index']] = 0;
    });

    var score = 0;
    var fixableScore = 0;

    var highestSeverityIndex = levels.length;

    if (feature.Vulnerabilities) {
      var addedByImageId = feature.AddedBy ? feature.AddedBy.split('.')[0] : null;

      feature.Vulnerabilities.forEach(function(vuln) {
        var severity = vulnService.LEVELS[vuln['Severity']];
        var cvssScore = vulnService.getCVSSScoreOf(vuln);
        var unadjustedScore = vulnService.getUnadjustedScoreOf(vuln);

        var currentScore = unadjustedScore;
        var scoreDivergence = null;

        // If the vulnerability has a CVSS score, ensure it is within 2 levels of the severity
        // score from the distro. If it is out of that band, then we have a score divergence
        // and use the distro's score directly.
        if (cvssScore != null) {
          if (cvssScore - unadjustedScore > 2) {
            scoreDivergence = 'adjusted-lower';
          } else if (unadjustedScore > cvssScore) {
            scoreDivergence = 'adjusted-higher';
          } else {
            currentScore = cvssScore;
          }
        }

        var exponentialScore = Math.pow(2, currentScore) + 0.1;
        var vuln_object = {
          'score': exponentialScore,
          'scoreDivergence': scoreDivergence,
          'severityInfo': severity,

          'cvssScore': cvssScore,
          'cvssColor': vulnService.getCVSSColor(cvssScore),

          'name': vuln.Name,
          'namespace': vuln.NamespaceName || vuln.Namespace,
          'description': vuln.Description,
          'link': vuln.Link,
          'severity': vuln.Severity,
          'metadata': vuln.Metadata,

          'featureName': feature.Name,
          'fixedInVersion': vuln.FixedBy,
          'introducedInVersion': feature.Version,

          'imageId': addedByImageId,
          'imageCommand': vulnService.imageComamandFor(manifest, addedByImageId),

          'expanded': false
        };

        // Save the highest vulnerability severity for this feature.
        highestSeverityIndex = Math.min(severity['index'], highestSeverityIndex);

        // Add the score and (if necessary) the fixable scores.
        score += exponentialScore;
        severityCountMap[severity['index']]++;
        vulnerabilities.push(vuln_object);

        if (vuln.FixedBy) {
          fixableCountMap[severity['index']]++;
          fixableScore += exponentialScore;
          fixable.push(vuln_object)
        }
      });
    }

    // Calculate the breakdown of the vulnerabilities by severity.
    var severityBreakdown = [];
    var fixableBreakdown = [];
    var leftoverBreakdown = [];

    levels.forEach(function(level) {
      if (severityCountMap[level['index']]) {
        severityBreakdown.push({
          'index': level['index'],
          'label': level['title'],
          'value': severityCountMap[level['index']],
          'color': level['color']
        });

        if (fixableCountMap[level['index']]) {
          fixableBreakdown.push({
            'index': level['index'],
            'label': level['title'],
            'value': fixableCountMap[level['index']],
            'color': level['color']
          });
        }

        var leftoverCount = severityCountMap[level['index']] - fixableCountMap[level['index']];
        if (leftoverCount) {
          leftoverBreakdown.push({
            'index': level['index'],
            'label': level['title'],
            'value': leftoverCount,
            'color': level['color']
          });
        }
      }
    });

    return {
      'vulnerabilities': vulnerabilities,
      'fixable': fixable,

      'severityBreakdown': severityBreakdown,
      'fixableBreakdown': fixableBreakdown,
      'leftoverBreakdown': leftoverBreakdown,

      'score': score,
      'fixableScore': fixableScore,

      'highestSeverity': levels[highestSeverityIndex],
    };
  };

  vulnService.imageComamandFor = function(manifest, addedByImageId) {
    if (!manifest || !addedByImageId) {
      return null;
    }
    return ImageMetadataService.getManifestCommand(manifest, addedByImageId)
  };

  vulnService.buildFeaturesInfo = function(manifest, resp) {
    var features = [];
    var severityCountMap = {};
    var highestFixableScore = 0;

    var levels = vulnService.getLevels();

    levels.forEach(function(level) {
      severityCountMap[level['index']] = 0;
    });

    vulnService.forEachFeature(resp, function(feature) {
      // Calculate the scores and breakdowns for all the vulnerabilities under feature.
      var vulnerabilityInfo = vulnService.buildVulnerabilitiesInfoForFeature(manifest, feature);
      var addedByImageId = feature.AddedBy ? feature.AddedBy.split('.')[0] : null;

      var feature_obj = {
        'name': feature.Name,
        'namespace': feature.NamespaceName || feature.Namespace,
        'version': feature.Version,
        'addedBy': feature.AddedBy,
        'imageId': addedByImageId,
        'imageCommand': vulnService.imageComamandFor(manifest, addedByImageId),
        'vulnCount': vulnerabilityInfo.vulnerabilities.length,
        'severityBreakdown': vulnerabilityInfo.severityBreakdown,
        'fixableBreakdown': vulnerabilityInfo.fixableBreakdown,
        'leftoverBreakdown': vulnerabilityInfo.leftoverBreakdown,
        'score': vulnerabilityInfo.score,
        'fixableCount': vulnerabilityInfo.fixable.length,
        'leftoverCount': vulnerabilityInfo.vulnerabilities.length - vulnerabilityInfo.fixable.length,
        'fixableScore': vulnerabilityInfo.fixableScore,
        'leftoverScore': vulnerabilityInfo.score - vulnerabilityInfo.fixableScore,
        'primarySeverity': vulnerabilityInfo.severityBreakdown[0],
        'primaryLeftover': vulnerabilityInfo.leftoverBreakdown[0],
        'vulnerabilities': vulnerabilityInfo.vulnerabilities,
        'fixable': vulnerabilityInfo.fixable
      };

      if (vulnerabilityInfo.highestSeverity) {
        severityCountMap[vulnerabilityInfo.highestSeverity['index']]++;
      } else {
        // Ensures that features with no vulns are always at the bottom of the table in the
        // default sort by fixableScore.
        feature_obj['fixableScore'] = -1;
        feature_obj['leftoverScore'] = -1;
      }

      highestFixableScore = Math.max(highestFixableScore, vulnerabilityInfo.fixableScore);

      features.push(feature_obj);
    });

    // Calculate the breakdown of each severity level for the features.
    var totalCount = features.length;
    var severityBreakdown = [];

    levels.forEach(function(level) {
      if (!severityCountMap[level['index']]) {
        return;
      }

      totalCount -= severityCountMap[level['index']];
      severityBreakdown.push({
        'index': level['index'],
        'label': level['title'],
        'value': severityCountMap[level['index']],
        'color': level['color']
      });
    });

    if (totalCount > 0) {
      severityBreakdown.push({
        'index': levels.length,
        'label': 'None',
        'value': totalCount,
        'color': '#2FC98E'
      });
    }

    return {
      'features': features,
      'brokenFeaturesCount': features.length - totalCount,
      'fixableFeatureCount': features.filter(function(f) { return f.fixableScore > 0 }).length,
      'severityBreakdown': severityBreakdown,
      'highestFixableScore': highestFixableScore
    }
  };

  vulnService.loadImageVulnerabilities = function(repo, image_id, result, reject) {
    var params = {
      'imageid': image_id,
      'repository': repo.namespace + '/' + repo.name,
      'vulnerabilities': true,
    };

    ApiService.getRepoImageSecurity(null, params).then(result, reject);
  };

  vulnService.loadManifestVulnerabilities = function(repo, digest, result, reject) {
    var params = {
      'manifestref': digest,
      'repository': repo.namespace + '/' + repo.name,
      'vulnerabilities': true,
    };

    ApiService.getRepoManifestSecurity(null, params).then(result, reject);
  };

  vulnService.hasFeatures = function(resp) {
    return resp.data && resp.data.Layer && resp.data.Layer.Features && resp.data.Layer.Features.length;
  };

  vulnService.forEachFeature = function(resp, callback) {
    if (!vulnService.hasFeatures(resp)) {
      return;
    }

    resp.data.Layer.Features.forEach(callback);
  };

  vulnService.forEachVulnerability = function(resp, callback) {
    if (!vulnService.hasFeatures(resp)) {
      return;
    }

    vulnService.forEachFeature(resp, function(feature) {
      if (feature.Vulnerabilities) {
        feature.Vulnerabilities.forEach(callback);
      }
    });
  };

  var cvssSeverityMap = {};

  vulnService.getSeverityForCVSS = function(score) {
    if (cvssSeverityMap[score]) {
      return cvssSeverityMap[score];
    }

    var levels = vulnService.getLevels();
    for (var i = 0; i < levels.length; ++i) {
      if (score >= levels[i].score) {
        cvssSeverityMap[score] = levels[i];
        return levels[i];
      }
    }

    return vulnService.LEVELS['Unknown'];
  };

  vulnService.getCVSSColor = function(score) {
    if (score == null) {
      return null;
    }

    return vulnService.getSeverityForCVSS(score).color;
  };

  vulnService.getLevels = function() {
    var levels =  Object.keys(vulnService.LEVELS).map(function(key) {
      return vulnService.LEVELS[key];
    });
    return levels.sort(function(a, b) {
        return a.index - b.index;
    });
  };

  vulnService.parseVectorsString = function(vectorsString) {
    return vectorsString.split('/');
  };

  vulnService.getVectorTitle = function(vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    return vector.title;
  };

  vulnService.getVectorDescription = function(vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    return vector.description;
  };

  vulnService.getVectorClasses = function(option, vectorString) {
    var parts = vectorString.split(':');
    var vector = vulnService.NVD_VECTORS[parts[0]];
    if (!vector) {
      return '';
    }

    var classes = '';
    if (option.id == parts[1]) {
      classes += 'current-vector ';
    } else {
      classes += 'not-current-vector ';
    }

    classes += option.severity;
    return classes;
  };

  vulnService.getVectorOptions = function(vectorString) {
    var parts = vectorString.split(':');
    if (vulnService.NVD_VECTORS[parts[0]]) {
      return vulnService.NVD_VECTORS[parts[0]].values;
    }
    return null;
  };

  vulnService.NVD_VECTORS = {
    'AV': {
      'title': 'Attack Vector',
      'description': 'This metric reflects how the vulnerability is exploited. The more remote an attacker can be to attack a host, the greater the vulnerability score.',
      'values': [
        {
          'id': 'N',
          'title': 'Network',
          'description': 'A vulnerability exploitable with network access means the vulnerable software is bound to the network stack and the attacker does not require local network access or local access.  Such a vulnerability is often termed "remotely exploitable". For example, an attacker causing a denial of service (DoS) by sending a specially crafted TCP packet across a wide area network.',
          'severity': 'high'
        },
        {
          'id': 'A',
          'title': 'Adjacent Network',
          'description': 'A vulnerability exploitable with adjacent network access means the vulnerable software is bound to the network stack. The attack is limited at the protocol level to a logically adjacent topology. An attack can be launched from the same shared physical (e.g., Bluetooth or IEEE 802.11) or logical (e.g., local IP subnet) network, or from within a secure or otherwise limited administrative domain.',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Local',
          'description': 'A vulnerability exploitable with only local access requires the attacker to have local access to the target system (e.g., keyboard, console), or remotely (e.g., SSH); or rely on User Interaction by another person to perform actions to exploit the vulnerability (e.g., using social engineering techniques to trick a legitimate user into opening a malicious document).',
          'severity': 'low'
        },
        {
          'id': 'P',
          'title': 'Physical',
          'description': 'A vulnerability exploitable with Physical access requires the attacker to have physical access to the vulnerable system or a local (shell) account. Examples of locally exploitable vulnerabilities are peripheral attacks such as Firewire/USB DMA attacks.',
          'severity': 'low'
        },
      ]
    },

    'AC': {
      'title': 'Attack Complexity',
      'description': 'This metric describes the conditions beyond the attacker’s control that must exist in order to exploit the vulnerability. The Base Score is greatest for the least complex attacks.',
      'values': [
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Specialized access conditions or extenuating circumstances do not exist making this easy to exploit',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Specialized access conditions exist making this harder to exploit',
          'severity': 'low'
        }
      ]
    },

    'PR': {
      'title': 'Privileges Required',
      'description': 'This metric describes the level of privileges an attacker must possess before exploiting the vulnerability. If no privileges are required, the base score is greatest.',
      'values': [
        {
          'id': 'N',
          'title': 'None',
          'description': 'The attacker is unauthorized prior to attack.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'An attacker with Low privileges has the ability to access only non-sensitive resources.',
          'severity': 'medium'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'The attacker requires privileges that provide significant control(e.g., component-wide settings and files) over the vulnerable component.',
          'severity': 'low'
        }
      ]
    },

    'UI': {
      'title': 'User Interaction',
      'description': 'This metric captures the requirement for a human user, other than the attacker, to participate in the successful compromise of the vulnerable component. If no user interaction is required, the base score is greatest.',
      'values': [
        {
          'id': 'N',
          'title': 'None',
          'description': 'The system can be exploited without interaction from any user.',
          'severity': 'high'
        },
        {
          'id': 'R',
          'title': 'Required',
          'description': 'The system can be exploited with user interaction(e.g., installation of an application).',
          'severity': 'medium'
        }
      ]
    },

    'S': {
      'title': 'Scope',
      'description': 'This metric captures whether a vulnerability in one vulnerable component impacts resources in components beyond its security scope. The Base Score is greatest when a scope change occurs.',
      'values': [
        {
          'id': 'U',
          'title': 'Unchanged',
          'description': 'An exploited vulnerability can only affect resources managed by the same security authority. The vulnerable component and the impacted component are either the same, or both are managed by the same security authority.',
          'severity': 'low'
        },
        {
          'id': 'C',
          'title': 'Changed',
          'description': 'An exploited vulnerability can affect resources beyond the security scope managed by the security authority of the vulnerable component. The vulnerable component and the impacted component are different and managed by different security authorities.',
          'severity': 'high'
        }
      ]
    },

    'C': {
      'title': 'Confidentiality Impact',
      'description': 'This metric measures the impact on confidentiality of a successfully exploited vulnerability. Confidentiality refers to limiting information access and disclosure to only authorized users, as well as preventing access by, or disclosure to, unauthorized ones. Increased confidentiality impact increases the vulnerability score.',
      'values': [
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is a total loss of confidentiality, resulting in disclosing all resources within the impacted component to the attacker. For example, an attacker steals the administrator\'s password, or private encryption keys of a web server.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'There is some loss of confidentiality. Access to some restricted information is obtained, but the attacker does not have control over what information is obtained, or the amount or kind of loss is limited. The information disclosure does not cause a direct, serious loss to the impacted component.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the confidentiality of the system.',
          'severity': 'low'
        }
      ]
    },

    'I': {
      'title': 'Integrity Impact',
      'description': 'This metric measures the impact to integrity of a successfully exploited vulnerability. Integrity refers to the trustworthiness and guaranteed veracity of information. The vulnerability Score is greatest when the consequence to the impacted component is highest.',
      'values': [
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is a total compromise of system integrity. There is a complete loss of system protection, resulting in the entire system being compromised. For example, the attacker is able to modify any/all files on the target system',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Modification of some system files or information is possible, but the attacker does not have control over what can be modified, or the scope of what the attacker can affect is limited. For example, system or application files may be overwritten or modified, but either the attacker has no control over which files are affected or the attacker can modify files within only a limited context or scope.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the integrity of the system.',
          'severity': 'low'
        }
      ]
    },

    'A': {
      'title': 'Availability Impact',
      'description': 'This metric measures the impact to availability of a successfully exploited vulnerability. Availability refers to the accessibility of information resources. Attacks that consume network bandwidth, processor cycles, or disk space all impact the availability of a system. Increased availability impact increases the vulnerability score.',
      'values': [
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is a total shutdown of the affected resource. The attacker can render the resource completely unavailable.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'There is reduced performance or interruptions in resource availability. An example is a network-based flood attack that permits a limited number of successful connections to an Internet service.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the availability of the system.',
          'severity': 'low'
        }
      ]
    },

    'E': {
      'title': 'Exploit Code Maturity',
      'description': 'This metric measures the likelihood of the vulnerability being attacked, and is based on the current state of exploit techniques, code availability or active exploitation.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Temporal Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'No exploit is required and details are widely available. Exploit code works in every situation, or is actively being delivered via an autonomous agent (such as a worm or virus).',
          'severity': 'high'
        },
        {
          'id': 'F',
          'title': 'Functional',
          'description': 'Functional exploit code is available. The code works in most situations where the vulnerability exists.',
          'severity': 'medium'
        },
        {
          'id': 'P',
          'title': 'Proof-of-Concept',
          'description': 'An attack demonstration is not practical for most systems. The code or technique is not functional in all situations and may require substantial modification by a skilled attacker.',
          'severity': 'medium'
        },
        {
          'id': 'U',
          'title': 'Unproven',
          'description': 'No exploit code is available, or an exploit is theoretical.',
          'severity': 'low'
        }
      ]
    },

    'RL': {
      'title': 'Remediation Level',
      'description': 'A typical vulnerability is unpatched when initially published. Workarounds or hotfixes may offer interim remediation until an official patch or upgrade is issued. The less official and permanent a fix, the higher the vulnerability score.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Temporal Score.',
          'severity': 'high'
        },
        {
          'id': 'U',
          'title': 'Unavailable',
          'description': 'There is either no solution available or it is impossible to apply.',
          'severity': 'high'
        },
        {
          'id': 'W',
          'title': 'Workaround',
          'description': 'There is an unofficial, non-vendor solution available. Users of the affected technology may create a patch of their own or provide steps to work around or mitigate the vulnerability.',
          'severity': 'medium'
        },
        {
          'id': 'T',
          'title': 'Temporary Fix',
          'description': 'There is an official but temporary fix available. This includes instances where the vendor issues a temporary hotfix, tool, or workaround.',
          'severity': 'medium'
        },
        {
          'id': 'O',
          'title': 'Official Fix',
          'description': 'A complete vendor solution is available. Either the vendor has issued an official patch, or an upgrade is available.',
          'severity': 'low'
        }
      ]
    },

    'RC': {
      'title': 'Report Confidence',
      'description': 'This metric measures the degree of confidence in the existence of the vulnerability and the credibility of known technical details. The urgency of a vulnerability is higher when a vulnerability is known to exist with certainty. ',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Temporal Score.',
          'severity': 'high'
        },
        {
          'id': 'C',
          'title': 'Confirmed',
          'description': 'Detailed reports exist, or functional reproduction is possible (functional exploits may provide this).',
          'severity': 'high'
        },
        {
          'id': 'R',
          'title': 'Reasonable',
          'description': 'Significant details are published, but researchers do not have full confidence in the root cause. The bug is reproducible and at least one impact is able to be verified (proof-of-concept exploits may provide this).',
          'severity': 'medium'
        },
        {
          'id': 'U',
          'title': 'Unknown',
          'description': 'Reporters are uncertain of the true nature of the vulnerability, and there is little confidence in the validity of the reports.',
          'severity': 'low'
        }
      ]
    },

    'CR': {
      'title': 'Confidentiality Requirement',
      'description': 'This metrics enables customization of CVSS score depending on the importance of the affected IT asset to a user’s organization, measured in terms of Confidentiality.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Loss of Confidentiality can have a catastrophic adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'high'
        },
        {
          'id': 'M',
          'title': 'Medium',
          'description': 'Loss of Confidentiality can have a serious adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Loss of Confidentiality can have only a limited adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'low'
        }
      ]
    },

    'IR': {
      'title': 'Integrity Requirement',
      'description': 'This metrics enables customization of CVSS score depending on the importance of the affected IT asset to a user’s organization, measured in terms of Integrity.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Loss of Integrity can have a catastrophic adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'high'
        },
        {
          'id': 'M',
          'title': 'Medium',
          'description': 'Loss of Integrity can have a serious adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Loss of Integrity can have only a limited adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'low'
        }
      ]
    },

    'AR': {
      'title': 'Availability Requirement',
      'description': 'This metrics enables customization of CVSS score depending on the importance of the affected IT asset to a user’s organization, measured in terms of Availability.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Loss of Availability can have a catastrophic adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'high'
        },
        {
          'id': 'M',
          'title': 'Medium',
          'description': 'Loss of Availability can have a serious adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Loss of Availability can have only a limited adverse effect on the organization or individuals associated (e.g., employees, customers).',
          'severity': 'low'
        }
      ]
    },

    'MAV': {
      'title': 'Modified Attack Vector',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'N',
          'title': 'Network',
          'description': 'A vulnerability exploitable with network access means the vulnerable software is bound to the network stack and the attacker does not require local network access or local access.  Such a vulnerability is often termed "remotely exploitable".  An example of a network attack is an RPC buffer overflow.',
          'severity': 'high'
        },
        {
          'id': 'A',
          'title': 'Adjacent Network',
          'description': 'A vulnerability exploitable with adjacent network access requires the attacker to have access to either the broadcast or collision domain of the vulnerable software.  Examples of local networks include local IP subnet, Bluetooth, IEEE 802.11, and local Ethernet segment.',
          'severity': 'medium'
        },
        {
          'id': 'L',
          'title': 'Local',
          'description': 'A vulnerability exploitable with only local access requires the attacker to have local access to the target system (e.g., keyboard, console), or remotely (e.g., SSH); or rely on User Interaction by another person to perform actions to exploit the vulnerability (e.g., using social engineering techniques to trick a legitimate user into opening a malicious document).',
          'severity': 'low'
        },
        {
          'id': 'P',
          'title': 'Physical',
          'description': 'A vulnerability exploitable with Physical access requires the attacker to have physical access to the vulnerable system or a local (shell) account. Examples of locally exploitable vulnerabilities are peripheral attacks such as Firewire/USB DMA attacks.',
          'severity': '??'
        },
      ]
    },

    'MAC': {
      'title': 'Modified Attack Complexity',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Specialized access conditions or extenuating circumstances do not exist making this easy to exploit',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'Specialized access conditions exist making this harder to exploit',
          'severity': 'low'
        }
      ]
    },

    'MPR': {
      'title': 'Modified Privileges Required',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'The attacker is unauthorized prior to attack.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'An attacker with Low privileges has the ability to access only non-sensitive resources.',
          'severity': 'medium'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'The attacker requires privileges that provide significant control(e.g., component-wide settings and files) over the vulnerable component.',
          'severity': 'low'
        }
      ]
    },

    'MUI': {
      'title': 'Modified User Interaction',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'The system can be exploited without interaction from any user.',
          'severity': 'high'
        },
        {
          'id': 'R',
          'title': 'Required',
          'description': 'The system can be exploited with user interaction(e.g., installation of an application).',
          'severity': 'medium'
        }
      ]
    },

    'MS': {
      'title': 'Modified Scope',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'U',
          'title': 'Unchanged',
          'description': 'An exploited vulnerability can only affect resources managed by the same security authority. The vulnerable component and the impacted component are either the same, or both are managed by the same security authority.',
          'severity': 'low'
        },
        {
          'id': 'C',
          'title': 'Changed',
          'description': 'An exploited vulnerability can affect resources beyond the security scope managed by the security authority of the vulnerable component. The vulnerable component and the impacted component are different and managed by different security authorities.',
          'severity': 'high'
        }
      ]
    },

    'MC': {
      'title': 'Modified Confidentiality Impact',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is total information disclosure, resulting in all system files being revealed. The attacker is able to read all of the system\'s data (memory, files, etc.)',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'There is considerable informational disclosure. Access to some system files is possible, but the attacker does not have control over what is obtained, or the scope of the loss is constrained. An example is a vulnerability that divulges only certain tables in a database.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the confidentiality of the system.',
          'severity': 'low'
        }
      ]
    },

    'MI': {
      'title': 'Modified Integrity Impact',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is a total compromise of system integrity. There is a complete loss of system protection, resulting in the entire system being compromised. The attacker is able to modify any files on the target system',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'Modification of some system files or information is possible, but the attacker does not have control over what can be modified, or the scope of what the attacker can affect is limited. For example, system or application files may be overwritten or modified, but either the attacker has no control over which files are affected or the attacker can modify files within only a limited context or scope.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the integrity of the system.',
          'severity': 'low'
        }
      ]
    },

    'MA': {
      'title': 'Modified Availability Impact',
      'description': 'This metrics enables the override of base metrics based on specific characteristics of a user’s environment.',
      'values': [
        {
          'id': 'X',
          'title': 'Not Defined',
          'description': 'Assigning this value indicates there is insufficient information to choose one of the other values, and has no impact on the overall Environmental Score.',
          'severity': 'high'
        },
        {
          'id': 'H',
          'title': 'High',
          'description': 'There is a total shutdown of the affected resource. The attacker can render the resource completely unavailable.',
          'severity': 'high'
        },
        {
          'id': 'L',
          'title': 'Low',
          'description': 'There is reduced performance or interruptions in resource availability. An example is a network-based flood attack that permits a limited number of successful connections to an Internet service.',
          'severity': 'medium'
        },
        {
          'id': 'N',
          'title': 'None',
          'description': 'There is no impact to the availability of the system.',
          'severity': 'low'
        }
      ]
    },

  };

  return vulnService;
}]);

