"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeHTTPRequestHeaders = exports.makeTraceDetails = exports.plugin = void 0;
const graphql_1 = require("graphql");
const apollo_engine_reporting_protobuf_1 = require("apollo-engine-reporting-protobuf");
const treeBuilder_1 = require("./treeBuilder");
const clientNameHeaderKey = 'apollographql-client-name';
const clientReferenceIdHeaderKey = 'apollographql-client-reference-id';
const clientVersionHeaderKey = 'apollographql-client-version';
exports.plugin = (options = Object.create(null), addTrace, { startSchemaReporting, executableSchemaIdGenerator, schemaReport, }) => {
    const loggerForPlugin = options.logger || console;
    const generateClientInfo = options.generateClientInfo || defaultGenerateClientInfo;
    return {
        serverWillStart: function ({ schema }) {
            if (!schemaReport)
                return;
            startSchemaReporting({
                executableSchema: options.experimental_overrideReportedSchema || graphql_1.printSchema(schema),
                executableSchemaId: executableSchemaIdGenerator(options.experimental_overrideReportedSchema || schema),
            });
        },
        requestDidStart({ logger: requestLogger, metrics, schema, request: { http, variables }, }) {
            const logger = requestLogger || loggerForPlugin;
            const treeBuilder = new treeBuilder_1.EngineReportingTreeBuilder({
                rewriteError: options.rewriteError,
                logger,
            });
            treeBuilder.startTiming();
            metrics.startHrTime = treeBuilder.startHrTime;
            if (http) {
                treeBuilder.trace.http = new apollo_engine_reporting_protobuf_1.Trace.HTTP({
                    method: apollo_engine_reporting_protobuf_1.Trace.HTTP.Method[http.method] ||
                        apollo_engine_reporting_protobuf_1.Trace.HTTP.Method.UNKNOWN,
                    host: null,
                    path: null,
                });
                if (options.sendHeaders) {
                    makeHTTPRequestHeaders(treeBuilder.trace.http, http.headers, options.sendHeaders);
                }
            }
            let endDone = false;
            function didEnd(requestContext) {
                if (endDone)
                    return;
                endDone = true;
                treeBuilder.stopTiming();
                treeBuilder.trace.fullQueryCacheHit = !!metrics.responseCacheHit;
                treeBuilder.trace.forbiddenOperation = !!metrics.forbiddenOperation;
                treeBuilder.trace.registeredOperation = !!metrics.registeredOperation;
                const operationName = requestContext.operationName ||
                    requestContext.request.operationName ||
                    '';
                if (metrics.queryPlanTrace) {
                    treeBuilder.trace.queryPlan = metrics.queryPlanTrace;
                }
                addTrace({
                    operationName,
                    queryHash: requestContext.queryHash,
                    document: requestContext.document,
                    source: requestContext.source,
                    trace: treeBuilder.trace,
                    executableSchemaId: executableSchemaIdGenerator(options.experimental_overrideReportedSchema || schema),
                    logger,
                }).catch(logger.error);
            }
            let didResolveSource = false;
            return {
                didResolveSource(requestContext) {
                    didResolveSource = true;
                    if (metrics.persistedQueryHit) {
                        treeBuilder.trace.persistedQueryHit = true;
                    }
                    if (metrics.persistedQueryRegister) {
                        treeBuilder.trace.persistedQueryRegister = true;
                    }
                    if (variables) {
                        treeBuilder.trace.details = makeTraceDetails(variables, options.sendVariableValues, requestContext.source);
                    }
                    const clientInfo = generateClientInfo(requestContext);
                    if (clientInfo) {
                        const { clientName, clientVersion, clientReferenceId } = clientInfo;
                        treeBuilder.trace.clientVersion = clientVersion || '';
                        treeBuilder.trace.clientReferenceId = clientReferenceId || '';
                        treeBuilder.trace.clientName = clientName || '';
                    }
                },
                executionDidStart() {
                    return {
                        willResolveField({ info }) {
                            return treeBuilder.willResolveField(info);
                        },
                    };
                },
                willSendResponse(requestContext) {
                    didEnd(requestContext);
                },
                didEncounterErrors(requestContext) {
                    if (!didResolveSource)
                        return;
                    treeBuilder.didEncounterErrors(requestContext.errors);
                    didEnd(requestContext);
                },
            };
        },
    };
};
function defaultGenerateClientInfo({ request }) {
    if (request.http &&
        request.http.headers &&
        (request.http.headers.get(clientNameHeaderKey) ||
            request.http.headers.get(clientVersionHeaderKey) ||
            request.http.headers.get(clientReferenceIdHeaderKey))) {
        return {
            clientName: request.http.headers.get(clientNameHeaderKey),
            clientVersion: request.http.headers.get(clientVersionHeaderKey),
            clientReferenceId: request.http.headers.get(clientReferenceIdHeaderKey),
        };
    }
    else if (request.extensions && request.extensions.clientInfo) {
        return request.extensions.clientInfo;
    }
    else {
        return {};
    }
}
function makeTraceDetails(variables, sendVariableValues, operationString) {
    const details = new apollo_engine_reporting_protobuf_1.Trace.Details();
    const variablesToRecord = (() => {
        if (sendVariableValues && 'transform' in sendVariableValues) {
            const originalKeys = Object.keys(variables);
            try {
                const modifiedVariables = sendVariableValues.transform({
                    variables: variables,
                    operationString: operationString,
                });
                return cleanModifiedVariables(originalKeys, modifiedVariables);
            }
            catch (e) {
                return handleVariableValueTransformError(originalKeys);
            }
        }
        else {
            return variables;
        }
    })();
    Object.keys(variablesToRecord).forEach(name => {
        if (!sendVariableValues ||
            ('none' in sendVariableValues && sendVariableValues.none) ||
            ('all' in sendVariableValues && !sendVariableValues.all) ||
            ('exceptNames' in sendVariableValues &&
                sendVariableValues.exceptNames.includes(name)) ||
            ('onlyNames' in sendVariableValues &&
                !sendVariableValues.onlyNames.includes(name))) {
            details.variablesJson[name] = '';
        }
        else {
            try {
                details.variablesJson[name] =
                    typeof variablesToRecord[name] === 'undefined'
                        ? ''
                        : JSON.stringify(variablesToRecord[name]);
            }
            catch (e) {
                details.variablesJson[name] = JSON.stringify('[Unable to convert value to JSON]');
            }
        }
    });
    return details;
}
exports.makeTraceDetails = makeTraceDetails;
function handleVariableValueTransformError(variableNames) {
    const modifiedVariables = Object.create(null);
    variableNames.forEach(name => {
        modifiedVariables[name] = '[PREDICATE_FUNCTION_ERROR]';
    });
    return modifiedVariables;
}
function cleanModifiedVariables(originalKeys, modifiedVariables) {
    const cleanedVariables = Object.create(null);
    originalKeys.forEach(name => {
        cleanedVariables[name] = modifiedVariables[name];
    });
    return cleanedVariables;
}
function makeHTTPRequestHeaders(http, headers, sendHeaders) {
    if (!sendHeaders ||
        ('none' in sendHeaders && sendHeaders.none) ||
        ('all' in sendHeaders && !sendHeaders.all)) {
        return;
    }
    for (const [key, value] of headers) {
        const lowerCaseKey = key.toLowerCase();
        if (('exceptNames' in sendHeaders &&
            sendHeaders.exceptNames.some(exceptHeader => {
                return exceptHeader.toLowerCase() === lowerCaseKey;
            })) ||
            ('onlyNames' in sendHeaders &&
                !sendHeaders.onlyNames.some(header => {
                    return header.toLowerCase() === lowerCaseKey;
                }))) {
            continue;
        }
        switch (key) {
            case 'authorization':
            case 'cookie':
            case 'set-cookie':
                break;
            default:
                http.requestHeaders[key] = new apollo_engine_reporting_protobuf_1.Trace.HTTP.Values({
                    value: [value],
                });
        }
    }
}
exports.makeHTTPRequestHeaders = makeHTTPRequestHeaders;
//# sourceMappingURL=plugin.js.map