/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

const log4js = require('log4js'),
      logger = log4js.getLogger('server'),
      moment = require('moment'),
      mime = require('mime-types'),
      fs = require('fs'),
      helmet = require('helmet')

const log4jsConfig = process.env.LOG4JS_CONFIG ? JSON.parse(process.env.LOG4JS_CONFIG) : undefined
const logPath = fs.existsSync('/etc/config') ? '/etc/config/' : 'config/'
// to update the log4js.json update the icp-platform-ui-chart configmap
log4js.configure(log4jsConfig || logPath + 'log4js.json')

const xContentTypeOptions = 'X-Content-Type-Options'

logger.info(`[pid ${process.pid}] [env ${process.env.NODE_ENV}] started.`)

const express = require('express'),
      exphbs  = require('express-handlebars'),
      handlebarsHelpers = require('./lib/shared/handlebarsHelpers'),
      path = require('path'),
      appConfig = require('./config'),
      appUtil = require('./lib/server/app-util'),
      csurf = require('csurf')

//early initialization
require('node-i18n-util')
process.env.BABEL_ENV = 'server'
require('@babel/register')

const bodyParser = require('body-parser'),
      cookieParser = require('cookie-parser'),
      requestLogger = require('./middleware/request-logger'),
      controllers = require('./controllers')

const app = express()

const csrfMiddleware = csurf({
  cookie: {
    httpOnly: false,
    secure: true
  }
})

const hbs = exphbs.create({
  // Specify helpers which are only registered on this instance.
  helpers: {
    properties: handlebarsHelpers,
    json: function(context) {
      return JSON.stringify(context)
    }
  }
})
app.engine('handlebars', hbs.engine)
app.set('env', 'production')
app.set('views', __dirname + '/views')
app.set('view engine', 'handlebars')
app.set('view cache', true)

appUtil.app(app)

const CONTEXT_PATH = appConfig.get('headerContextPath'),
      STATIC_PATH = path.join(__dirname, 'public')

app.use(cookieParser(), csrfMiddleware,(req, res, next) => {
  if(!req.path.endsWith('.js') && !req.path.endsWith('.css')) {
    next()
    return
  }
  res.setHeader('Cache-Control', 'no-store')
  res.setHeader('Pragma', 'no-cache')
  res.setHeader(xContentTypeOptions, 'nosniff')
  res.append('Content-Encoding', 'gzip')
  const type = mime.lookup(path.join('public', req.path))
  if (typeof type != 'undefined') {
    const charset = mime && mime.charsets.lookup(type)
    res.append('Content-Type', type + (charset ? '; charset=' + charset : ''))
  }
  req.url = `${req.url}.gz`
  const accessToken = req.cookies['acm-access-token-cookie']
  if (req.headers.authorization) {
    req.headers.authorization = `Bearer ${accessToken}`
  }
  else {
    req.headers.Authorization = `Bearer ${accessToken}`
  }
  next()
})
app.use(CONTEXT_PATH, express.static(STATIC_PATH, {
  maxAge: process.env.NODE_ENV === 'development' ? 0 : 1000 * 60 * 60 * 24 * 365,
  setHeaders: (res, fp) => {
    if (fp.startsWith(`${STATIC_PATH}/nls`)) {
      res.setHeader('Cache-Control', 'max-age=0')
    } else {
      res.setHeader('Expires', moment().add(12, 'months').toDate())
    }
  }
}))



app.get(`${CONTEXT_PATH}/performance-now.js.map`, (req, res) => res.sendStatus(404))

app.use(cookieParser())
  .use(requestLogger)
  .use(bodyParser.json({ limit: '512kb' }))
  .use(bodyParser.urlencoded({ extended: false }))
  .use(controllers)

app.get('/favicon.ico', (req, res) => res.sendStatus(204))

app.locals.config = require('./lib/shared/config')
// this path only existing after npm build
// eslint-disable-next-line import/no-unresolved
app.locals.manifest = require('./public/webpack-assets.json')
let server
if (process.env.NODE_ENV === 'development') {
  app.use(helmet())
  const https = require('https')
  const privateKey  = fs.readFileSync('./sslcert/server.key', 'utf8')
  const certificate = fs.readFileSync('./sslcert/server.crt', 'utf8')
  const credentials = {key: privateKey, cert: certificate}
  server = https.createServer(credentials, app)
} else {
  const http = require('http')
  server = http.createServer(app)
  app.use(helmet({
    hidePoweredBy: true,
    frameguard: true,
    noSniff: true,
    xssFilter: true
  }))
  // Remove the X-Powered-By headers.
  app.disable('x-powered-by')
}

const port = process.env.PORT || appConfig.get('httpPort')

// start server
logger.info('Starting express server.')
server.listen(port, () => {
  logger.info(`console-header listening on ${process.env.NODE_ENV === 'development' ? 'https' : 'http'} port ${port}`)
  if (process.env.NODE_ENV === 'development') {
    logger.info(`console-header is available at: https://localhost:${port}${appConfig.get('headerContextPath')}`)
  }
})

process.on('SIGTERM', () => {
  logger.info('SIGTERM received.  Shutting down express server.')
  server.close(err => {
    if (err) {
      logger.error(err)
      process.exit(1)
    }
    logger.info('Server shutdown successfully.')
    process.exit(0)
  })
})
