/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

const _ = require('lodash'),
      ReactDOMServer = require('react-dom/server'),
      React = require('react'),
      express = require('express'),
      redux = require('redux'),
      context = require('../../lib/shared/context'),
      router = express.Router(),
      Provider = require('react-redux').Provider,
      StaticRouter = require('react-router-dom').StaticRouter,
      navUtil = require('../../lib/server/nav-util'),
      appUtil = require('../../lib/server/app-util'),
      i18n = require('node-i18n-util'),
      path = require('path'),
      HeaderContainer = require('../../src-web/containers/HeaderContainer').default,
      appClient = require('../../lib/server/application-client'),
      namespaceClient = require('../../lib/server/namespace-client'),
      uiConfigClient = require('../../lib/server/uiconfig-client'),
      oauthInfoClient = require('../../lib/server/oauth-info-client'),
      clusterClient = require('../../lib/server/cluster-client'),
      userAccessClient = require('../../lib/server/user-access-client'),
      config = require('../../config'),
      log4js = require('log4js'),
      logger = log4js.getLogger('app'),
      async = require('async')

import { user, loggedIn } from '../../src-web/reducers/user'
import { modal } from '../../src-web/reducers/modal'
import { uiconfig } from '../../src-web/reducers/uiconfig'
import { namespaces } from '../../src-web/reducers/namespaces'
import { nav } from '../../src-web/reducers/nav'
import { experience } from '../../src-web/reducers/experience'
import { oauthInfo } from '../../src-web/reducers/oauthInfo'

let loginAction, nsAction, navAction, uiConfigAction, oauthInfoAction

router.get('/api/v1/header', (req, res) => {
  const query = (req && req.query) ? req.query : {}
  const headers = (req && req.headers) ? req.headers : {}
  const serviceId = query.serviceId ? query.serviceId : null
  const dev = query.dev ? query.dev : null
  let authorizationToken = ''
  if (headers.authorization) {
    authorizationToken = headers.authorization
  }
  else if (headers.Authorization) {
    authorizationToken = headers.Authorization
  }
  // user parameter that defines the returning api groups
  let targetAPIGroups = []
  let rawDataFlag = false
  if (query.targetAPIGroups && Array.isArray(JSON.parse(query.targetAPIGroups))) {
    targetAPIGroups = _.uniqWith(JSON.parse(query.targetAPIGroups), _.isEqual)
    if (query.raw) {
      rawDataFlag = query.raw // user parameter if returning the raw access data
    }
  }
  // logger.info(`targetAPIGroups from ${serviceId} is : ${JSON.stringify(targetAPIGroups)}`)
  const store = redux.createStore(redux.combineReducers({
    user, loggedIn, modal, uiconfig, namespaces, nav, experience, oauthInfo
  }))
  // build call array for getting the whole header data
  const headerCalls = []
  // always run step 1 for common console header data
  headerCalls.push(getHeaderData(req, res))
  // only run step 2 when other micro service passing API for user permission checking
  if (authorizationToken) {
    if (targetAPIGroups.length > 0) {
      headerCalls.push(getUserAccessInfo(req, res, authorizationToken, targetAPIGroups, rawDataFlag))
    }
  } else {
    logger.info(`${serviceId} has no permission for checking user access on ${JSON.stringify(targetAPIGroups)}}`)
  }

  async.waterfall(
    headerCalls,
    (err, results) => { // deal with data from step 1 and step 2 then to other micro service
      // logger.info(`results into final step is : ${JSON.stringify(results)}`)
      if (err) {
        return res.status(err.statusCode || 500).send(err.details)
      }

      loginAction = loginAction === undefined ? require('../../src-web/actions/login') : loginAction
      store.dispatch(loginAction.receiveLoginSuccess(req.user))

      nsAction = nsAction === undefined ? require('../../src-web/actions/namespaces') : nsAction
      store.dispatch(nsAction.namespacesReceiveSuccess(results.namespaceReq))

      oauthInfoAction = oauthInfoAction === undefined
        ? require('../../src-web/actions/oauthInfo')
        : oauthInfoAction
      store.dispatch(oauthInfoAction.oauthReceiveSuccess(results.oauthInfoReq))

      const reqContext = context(req)
      //getConfig will return all nav items and header config
      const navConfig = navUtil.getConfig(req, serviceId, dev, reqContext.locale)
      const uiConfigReqTemp = results.uiConfigReq
      const designConfigReq = {
        header: navConfig.header,
        about: navConfig.about,
        disabledItems: navConfig.disableItems,
        appLinks: _.flatten([results.OpenshiftURLReq, results.consoleLinkReq])
      }
      uiConfigAction = uiConfigAction === undefined
        ? require('../../src-web/actions/uiconfig')
        : uiConfigAction
      store.dispatch(uiConfigAction.uiConfigReceiveSuccess({
        config: designConfigReq,
        ...uiConfigReqTemp,
        xsrfToken: req.csrfToken(),
        headerContext: process.env.headerContextPath || '/multicloud/header'
      }))

      const finalNavRoutes = navConfig.navItems
      const finalOtherNavRoutes = navConfig.otherNav

      navAction = navAction === undefined ? require('../../src-web/actions/nav') : navAction
      store.dispatch(navAction.navReceiveSuccess(finalNavRoutes, finalOtherNavRoutes))

      const props = {
        locale: reqContext.locale
      }

      const headerHtml = ReactDOMServer.renderToString(
        <Provider store={store}>
          <StaticRouter
            location={req.originalUrl}
            context={reqContext}>
            <HeaderContainer />
          </StaticRouter>
        </Provider>
      )

      const state = store.getState()
      const files = getHeaderFiles(req, reqContext.locale)
      const userAccess = results.userAccess
      res.send({ headerHtml, props, state, files, userAccess })
    })
})

// parallelly calling to get common console header data
function getHeaderData(req, res) {
  const OpenshiftURLReq = appClient.getOpenshiftConsoleURL.bind(appClient, req)
  const consoleLinkReq = appClient.getConsoleLinks.bind(appClient, req)
  const namespaceReq = namespaceClient.getUserNamespaces.bind(namespaceClient, req)
  const uiConfigReq = uiConfigClient.getUiConfig.bind(uiConfigClient, req)
  const oauthInfoReq = oauthInfoClient.getOauthInfo.bind(oauthInfoClient, req)
  return function(cb) {
    async.parallel(
      { OpenshiftURLReq, consoleLinkReq, namespaceReq, uiConfigReq, oauthInfoReq },
      (err, headerResults) => {
        if (err) {
          return res.status(err.statusCode || 500).send(err.details)
        }
        // logger.info(`common header data are : ${JSON.stringify(headerResults)}`)
        return cb(null, headerResults)
      })
  }
}

// parallelly calling to get user access info on each NS
// by using user namespaces passed from getHeader/getHeaderData
function getUserAccessInfo(req, res, authorizationToken, targetAPIGroups, rawDataFlag) {
  return function(results, cb) {
    const userAllNS = nsFormatter(results.namespaceReq)
    // logger.info(`user all NS got from getHeader/getHeaderData are : ${JSON.stringify(userAllNS)}`)
    const userAccessReq = []
    userAllNS.forEach((singleNS) => {// each element binds with one NS then parallelly call whole array
      userAccessReq.push(
        userAccessClient.getUserAccess.bind(
          userAccessClient,
          {
            authorizationToken,
            singleNS,
            targetAPIGroups,
            rawDataFlag
          }
        ))
    })
    async.parallel(userAccessReq, (err, userAccessResult) => {
      if (err) {
        return res.status(err.statusCode || 500).send(err.details)
      }
      // logger.info(`user access data got from step 2 is : ${JSON.stringify(userAccessResult)}`)
      results.userAccess = userAccessResult
      return cb(null, results)
    })
  }
}

function nsFormatter(multiNS) {
  let formattedNS = []
  if (Array.isArray(multiNS) && multiNS.length > 0) {
    formattedNS = multiNS.map((ns) => {
      if (ns.Name) {
        return ns.Name
      } else {
        return ns.name
      }})
  }
  // logger.info(`formattedNS is : ${JSON.stringify(formattedNS)}`)
  return formattedNS
}

function getHeaderFiles(req, locale) {
  const manifest = appUtil.app().locals.manifest
  return {
    'dll': {
      path: `${config.get('headerContextPath')}/${manifest.vendor.js}`
    },
    'js': {
      path: `${config.get('headerContextPath')}/${manifest.header.js}`
    },
    'css': {
      path: Array.isArray(manifest.header.css)
        ? manifest.header.css.length > 1 && `${config.get('headerContextPath')}/${manifest.header.css[1]}`
        : `${config.get('headerContextPath')}/${manifest.header.css}`
    },
    'nls': {
      path: `${config.get('headerContextPath')}/${resolvePropertiesFile('nls/platform-header.properties', locale)}`
    }
  }
}

function resolvePropertiesFile(href, lang) {
  const relativePath = `../../${href}`
  require(relativePath)

  const bundlePath = path.join(__dirname, relativePath),
        locale = i18n._resolveBundle(bundlePath, lang).locale,
        basePath = href.substring(0, href.indexOf('.properties'))

  href = basePath + (locale ? '_' + locale.replace('-', '_') : '') + '.js'
  return href
}

router.get('/api/v1/clusters', (req, res) => {
  clusterClient.getManagedClusters(req, (err, clusterResults) => {
    if (err) {
      if (err.message) {
        res.statusMessage = err.message
      }
      return res.status(err.statusCode || 500).send(err)
    }
    res.send(clusterResults)
  })
})

router.get('/api/logout/test', (req, res) =>{
  return res.send({result: 'OK'})
})

module.exports = router
