/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import { WelcomeStatic } from '../../src-web/containers/WelcomeStatic'

const thunkMiddleware = require('redux-thunk').default,
      redux = require('redux'),
      express = require('express'),
      router = express.Router({ mergeParams: true }),
      namespaceClient = require('../../lib/server/namespace-client'),
      headerClient = require('../../lib/server/header-client'),
      oauthInfoClient = require('../../lib/server/oauth-info-client'),
      async = require('async'),
      commonUtil = require('../../src-web/util/common'),
      securityMW = require('security-middleware'),
      appUtil = require('../../lib/server/app-util'),
      Provider = require('react-redux').Provider,
      StaticRouter = require('react-router-dom').StaticRouter,
      ReactDOMServer = require('react-dom/server'),
      config = require('../../config'),
      context = require('../../lib/shared/context'),
      React = require('react'),
      manifestCSShelper = require('../../lib/shared/manifestCSShelper')

const log4js = require('log4js'),
      logger = log4js.getLogger('app')

let Login, namespaces, reducers, nav, experience, oauthInfoAction //laziy initialize to reduce startup time seen on k8s

router.get('/logout', securityMW.logout)

router.get('*', (req, res) => {
  if (req.query && req.query.root === 'true') {
    return res.redirect('/multicloud/header')
  }

  reducers = reducers === undefined ? require('../../src-web/reducers') : reducers

  const store = redux.createStore(redux.combineReducers(reducers), redux.applyMiddleware(
    thunkMiddleware // lets us dispatch() functions
  ))

  const namespaceReq = namespaceClient.getUserNamespaces.bind(namespaceClient, req)
  const headerReq = headerClient.getHeader.bind(headerClient, req)
  const oauthInfoReq = oauthInfoClient.getOauthInfo.bind(oauthInfoClient, req)
  Login = Login === undefined ? require('../../src-web/actions/login') : Login

  async.parallel({ namespaceReq, headerReq, oauthInfoReq }, (err, results) => {
    if (err) {
      logger.error(err)
      return res.status(err.statusCode || 500).send(err.details)
    }

    store.dispatch(Login.receiveLoginSuccess(req.user))

    namespaces = namespaces === undefined ? require('../../src-web/actions/namespaces') : namespaces
    store.dispatch(namespaces.namespacesReceiveSuccess(results.namespaceReq))

    experience = experience === undefined ? require('../../src-web/actions/experience') : experience
    const params = commonUtil.parseQueryParams(req.url)

    store.dispatch(experience.updateHeaderExperience(params.useNav || ''))

    oauthInfoAction = oauthInfoAction === undefined ? require('../../src-web/actions/oauthInfo') : oauthInfoAction
    store.dispatch(oauthInfoAction.oauthReceiveSuccess(results.oauthInfoReq))

    const baseNavRoutes = results.headerReq.state.nav.navItems
    nav = nav === undefined ? require('../../src-web/actions/nav') : nav
    store.dispatch(nav.navReceiveSuccess(baseNavRoutes))

    const contextTemp = getContext(req)

    const { headerHtml: header, props: propsH, state: stateH, files: filesH } = results.headerReq
    const manifest = manifestCSShelper(appUtil.app().locals.manifest, 1)
    if (req.originalUrl === '/multicloud/header/error') {
      try {
        res.render('home', Object.assign({
          manifest,
          content: ReactDOMServer.renderToString(
            <Provider store={store}>
              <StaticRouter
                location={req.originalUrl}
                context={contextTemp}>
                <WelcomeStatic></WelcomeStatic>
              </StaticRouter>
            </Provider>
          ),
          headerContextPath: config.get('headerContextPath'),
          state: store.getState(),
          props: contextTemp,
          header: header,
          propsH: propsH,
          stateH: stateH,
          filesH: filesH
        }, contextTemp))
      } catch(err) {
        // eslint-disable-next-line no-console
        console.error(err)
        res.send(err)
      }
    }
  })
  if (req.originalUrl !== '/multicloud/welcome' && req.originalUrl !== '/multicloud/header/error') {
    return res.redirect('/multicloud/welcome')
  }
})

function getContext(req) {
  const reqContext = context(req)
  return {
    title: 'Red Hat Advanced Cluster Management for Kubernetes',
    context: reqContext
  }
}

module.exports = router
