/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

const client = require('../shared/client')
const config = require('../shared/config')
const _ = require('lodash')

const DEFAULT_OPTIONS = {
  credentials: 'same-origin',
  headers: {
    'X-Requested-With': 'XMLHttpRequest'
  }
}

module.exports.getHostUrl = function () {
  const port = window.location.port ? `:${window.location.port}` : ''
  return `${window.location.protocol}//${window.location.hostname}${port}`
}

module.exports.getBaseUrl = function () {
  let path = window.location.pathname
  path = path.split('/').slice(0, 4).join('/')
  return module.exports.getHostUrl() + path
}

module.exports.getContextRoot = function() {
  if (client) {
    const nav = document.getElementById('multicloud-headerContext')
    if (nav && nav.value) {
      return nav.value
    }
    return '/multicloud/header'
  }
  return config.headerContextPath
}

module.exports.getQueryString = function (field, url) {

  if (!client && !url) {
    return null
  }

  const href = url ? url : window.location.href,
        reg = new RegExp('[?&]' + field + '=([^&#]*)', 'i'),
        string = reg.exec(href)

  return string ? string[1] : null
}

module.exports.appendQueryString = function (url, queryVars) {

  if (!queryVars || Object.keys(queryVars).length === 0) {
    return url
  }

  const firstSeparator = (url.indexOf('?') === -1 ? '?' : '&')
  const queryStringParts = []
  for (const key in queryVars) {
    if (!module.exports.getQueryString(key, url) && queryVars[key]) {
      queryStringParts.push(key + '=' + queryVars[key])
    }
  }

  if (queryStringParts.length === 0) {
    return url
  }

  const queryString = queryStringParts.join('&')
  return url + firstSeparator + queryString
}

module.exports.fetch = function (url, successCB, errorCB, options) {

  options = options || DEFAULT_OPTIONS
  const token = document.getElementById('headerAccess') ? document.getElementById('headerAccess').value : ''
  const methods = [ 'POST', 'PUT', 'DELETE']
  if(_.indexOf(methods, options.method) > -1) {
    options.headers['XSRF-Token'] = token.toString('ascii')
  }

  return fetch(url, options)
    .then(checkStatus)
    .then(parseJSON)
    .then(json => successCB && successCB(json))
    .catch(ex => errorCB && errorCB({ error: ex }))
}

module.exports.fetchWithReduce = function (url, successCB, errorCB, options, params) {
  options = options || DEFAULT_OPTIONS
  fetch(url, options)
    .then(checkStatus)
    .then(parseJSON)
    .then(json => {
      if (params) {
        _.remove(json.items, x => {
          return (params.field).split('.').reduce(index, x) !== params.match
        })
      }
      successCB && successCB(json)
    })
    .catch(ex => errorCB && errorCB({ error: ex }))
}

function index(obj,i) {
  return obj[i]
}

function checkStatus(response) {
  if (response.status >= 200 && response.status < 300) {
    return response
  } else {
    const error = new Error(response.statusText)
    error.response = response
    throw error
  }
}

function parseJSON(response) {
  // '' and 'OK' is not a valid json response, need to check
  return response.text().then(text => (text && text.trim() !== 'OK') ? JSON.parse(text) : {})
}
