/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (C) Copyright IBM Corporation 2014, 2018 All Rights Reserved.
 * Copyright (c) 2020 Red Hat, Inc.
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *******************************************************************************/

class JsnParseException {

  constructor(message, parsedLine, snippet, parsedFile) {
    this.rawMessage = message
    this.parsedLine = (parsedLine !== undefined) ? parsedLine : -1
    this.snippet = (snippet !== undefined) ? snippet : null
    this.parsedFile = (parsedFile !== undefined) ? parsedFile : null

    this.updateRepr()

    this.message = message
  }

  isDefined(input) {
    return input !== undefined && input !== null
  }

  getSnippet() {
    return this.snippet
  }

  setSnippet(snippet) {
    this.snippet = snippet

    this.updateRepr()
  }

  getParsedFile() {
    return this.parsedFile
  }

  setParsedFile(parsedFile) {
    this.parsedFile = parsedFile

    this.updateRepr()
  }

  getParsedLine() {
    return this.parsedLine
  }

  setParsedLine(parsedLine) {
    this.parsedLine = parsedLine

    this.updateRepr()
  }

  updateRepr() {
    this.message = this.rawMessage

    let dot = false
    if ('.' === this.message.charAt(this.message.length - 1)) {
      this.message = this.message.substring(0, this.message.length - 1)
      dot = true
    }

    if (null !== this.parsedFile) {
      this.message += ` in ${JSON.stringify(this.parsedFile)}`
    }

    if (this.parsedLine >= 0) {
      this.message += ` at line ${this.parsedLine}`
    }

    if (this.snippet) {
      this.message += ` (near ${this.snippet})`
    }

    if (dot) {
      this.message += '.'
    }
  }
}


class JsonParser {

  constructor(offset, lined) {
    this.skipped= 0
    this.lines = []
    this.currentLineNb= -1
    this.currentLine= ''
    this.ignoreDups= false
    this.offset = (offset !== undefined) ? offset : 0
    this.lined = lined
  }

  parse(value, row, ignoreDups) {
    this.currentLineNb = -1
    this.currentLinePos = 0
    this.currentLine = ''
    this.ignoreDups = ignoreDups

    // pseudo objects
    const data = {$v:{}, $r:-1}

    // add line #'s to each line before we start messing around
    this.lines = value.split('\n')
    if (!this.lined) {
      const skipped = (row || 0) + this.skipped // add this.skipped in case leading comments are removed by cleanup
      for (let i = 0; i < this.lines.length; i++) {
        if (!this.lines[i].endsWith('\r')) {
          this.lines[i] += '\r'
        }
        this.lines[i] += ' #' + (i + skipped)
      }
      this.lined = true
    }

    // remove multiline comments (/*   fff   */)
    // replace tabs with spaces
    this.lines = this.lines.join('\n').replace(/\/\*[^]*?\*\//g, '').replace(/\t+/g, '  ').trim().split('\n')

    // parse it
    this.moveToNextLine()
    this.parseHelper(data, true)
    return data.$v
  }

  parseHelper(map, first) {
    let arr
    const tok = this.getNextToken()
    if (tok) {
      switch (tok.del) {
      case '{':
        this.getObject(map.$v, map)
        break
      case '[':
        arr = map.$v = []
        this.getList(arr)
        break
      default:
        if (first) {
          throw new JsnParseException('Must start with { or [.', 10, this.currentLine)
        }
        map.$v = tok.val
        break
      }
    }
  }

  getObject(map, parent) {
    let obj
    let tok
    do {
      // get the key
      let key
      tok = this.getNextToken()
      if (tok.del==='}') {
        break
      } else if (tok.isStr) {
        if (!!map[tok.val] && !this.ignoreDups) {
          throw new JsnParseException(`Duplicate key "${tok.val}"`, this.getRealCurrentLineNb() + 1, this.currentLine)
        }
        key = tok.val
      } else {
        throw new JsnParseException(`Expecting key or end brace before start of this line "${this.currentLineRaw}"`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }

      // init the object
      obj = this.getRealCurrentLineNb({})
      obj.$v = {}
      obj.$j = true
      obj.$k = key
      map[key] = obj

      // get the value
      tok = this.getNextToken()
      if (tok.del !== ':') {
        throw new JsnParseException(`Colon is missing after "${key}" key.`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }
      this.parseHelper(obj)

      // get the comma if this is a list
      tok = this.getNextToken()
    } while (tok.del===',')

    // eat ending '}'
    if (tok.del!=='}') {
      throw new JsnParseException(`Missing ending brace on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
    } else if (parent) {
      const lineNum = this.getRealCurrentLineNb()
      if (parent.$r !== lineNum) {
        parent.$_r = lineNum
      }
    }
  }

  getList(val) {
    let inc=0
    let obj
    let value
    let needComma = false
    let extraComma = false
    while (inc<1000000) {
      const tok = this.getNextToken()
      if (needComma && tok.del!==',' && tok.del!==']') {
        throw new JsnParseException(`Expecting comma or ] at end of this line
        "${this.currentLine.substr(0,this.currentLinePos)}".`,
        this.getRealCurrentLineNb() + 1, this.currentLine)
      }
      if (tok.val!==undefined) {
        val.push(tok.val)
        needComma = true
        extraComma = false
      } else {
        switch (tok.del) {
        case ',':
          needComma = false
          extraComma = true
          break
        case '{':
          obj = this.getRealCurrentLineNb({})
          value = obj.$v = {}
          this.getObject(value, obj)
          val.push(obj)
          needComma = true
          extraComma = false
          break
        case '[':
          val.push(this.getList([]))
          needComma = true
          extraComma = false
          break
        case ']':
          if (extraComma) {
            throw new JsnParseException(`List should not end in comma "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
          }
          return val
        default:
          throw new JsnParseException(`List syntax error on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
        }
      }
      inc++
    }
    return []
  }

  peekNextToken() {
    // save everything
    const saveCurrentLineNb = this.currentLineNb
    const saveCurrentLinePos = this.currentLinePos
    const saveLastLineRaw = this.lastLineRaw
    const saveLastLineNum = this.lastLineNum
    const saveCurrentLine = this.currentLine
    const saveCurrentLineRaw = this.currentLineRaw

    // get token
    const tok = this.getNextToken()

    // revert
    this.currentLinePos = saveCurrentLinePos
    this.currentLineNb = saveCurrentLineNb
    this.lastLineRaw = saveLastLineRaw
    this.lastLineNum = saveLastLineNum
    this.currentLine = saveCurrentLine
    this.currentLineRaw = saveCurrentLineRaw

    return tok
  }

  getNextToken() {
    let ch = this.getNextChar()

    // skip white space
    while (this.isWhitespace(ch)) {
      ch = this.getNextChar()
    }

    const tok = {}
    this.startTokenPos = this.currentLinePos-1
    if (!ch) {
      return null
    } else if (this.isLetter(ch)) {
      tok.val = this.getIdentifier()
      switch (tok.val) {
      case 'true':
        tok.val = true
        break
      case 'false':
        tok.val = false
        break
      default:
        tok.isId = true
        break
      }
    } else if (this.isDecimal(ch)) {
      tok.val = this.getNumber(ch)
    } else {
      switch (ch) {
      case '"':
        tok.val = this.getString()
        tok.isStr = true
        break

      case '#':
        this.moveToNextLine()
        return this.getNextToken()

      case '/':
        if (this.getNextChar()==='/') {
          this.moveToNextLine()
          return this.getNextToken()
        } else {
          throw new JsnParseException(`Comment missing / on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
        }

      case '[':
      case ']':
      case '{':
      case '}':
      case ',':
      case ':':
      case '+':
        tok.del = ch
        break
      case '-':
        if (this.isDecimal(this.peekNextChar())) {
          ch = this.getNextChar()
          tok.val = this.getNumber(ch)
        } else {
          tok.del = ch
        }
        break
      default:
        throw new JsnParseException(`Key value pair syntax error on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }
    }
    return tok
  }

  // getIdentifier gets an identifier and returns the literal string;
  getIdentifier() {
    let ch = this.getNextChar()
    while (!!ch && (this.isLetter(ch) || this.isDigit(ch) || ch === '-' || ch === '.')) {
      ch = this.getNextChar(true)
    }

    if (ch) {
      this.currentLinePos--
    }

    return this.currentLine.substring(this.startTokenPos, this.currentLinePos)
  }

  // getString gets a quoted string;
  getString() {
    let inc=0
    let braces = 0
    while (inc<1000000) {
      const ch = this.getNextChar(true)

      if (!ch) {
        throw new JsnParseException(`Missing closing double quote on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }

      if (ch === '"' && braces === 0) {
        break
      }

      if (braces === 0 && ch === '$' && this.peekNextChar() === '{') {
        braces++
        this.getNextChar()
      } else if (braces > 0 && ch === '{') {
        braces++
      }
      if (braces > 0 && ch === '}') {
        braces--
      }

      if (ch === '\\') {
        this.skipEscape()
      }
      inc++
    }
    return this.currentLine.substring(this.startTokenPos+1, this.currentLinePos-1)
  }

  skipEscape() {
    const ch = this.getNextChar() // read character after '/';
    if (['a', 'b', 'f', 'n', 'r', 't', 'v', '\\', '/', '"'].indexOf(ch) !== -1) {
      // normal
    } else if (['0', '1', '2', '3', '4', '5', '6', '7'].indexOf(ch) !== -1) {
      // octal
      this.skipDigits(ch, 8, 3)
    } else {
      // hex/universal char
      const len = ch==='x' ? 2 : (ch==='u'? 4 : (ch==='U'?8:0))
      if (len) {
        this.skipDigits(this.getNextChar(), 16, len)
      } else {
        throw new JsnParseException(`Illegal escape on this line "${this.currentLineRaw}".`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }
    }
  }

  skipDigits(ch, base, n) {
    const start = n
    while (n > 0 && this.digitVal(ch) < base) {
      ch = this.getNextChar()
      if (!ch) {
        break
      }
      n--
    }
    if (n > 0) {
      throw new JsnParseException(`Illegal digit on this line "${this.currentLineRaw }".`, this.getRealCurrentLineNb() + 1, this.currentLine)
    }

    if (n !== start) {
      this.currentLinePos--
    }
  }

  getNumber(ch) {
    if (ch === '0') {
      // check for hexadecimal, octal or float;
      ch = this.getNextChar()
      if (ch === 'x' || ch === 'X') {
        // hexadecimal;
        ch = this.getNextChar()
        let found = false
        while (this.isHexadecimal(ch)) {
          ch = this.getNextChar()
          found = true
        }

        if (!found) {
          throw new JsnParseException(`Illegal hex number on this line "${this.currentLineRaw }".`, this.getRealCurrentLineNb() + 1, this.currentLine)
        }

        if (ch) {
          this.currentLinePos--
        }
        return this.currentLine.substring(this.startTokenPos, this.currentLinePos)
      }

      // now it's either something like: 0421(octal) or 0.1231(float);
      let illegalOctal = false
      while (this.isDecimal(ch)) {
        ch = this.getNextChar()
        if (ch === '8' || ch === '9') {
          illegalOctal = true
        }
      }

      if (ch === 'e' || ch === 'E') {
        this.skipExponent(ch)
        return this.currentLine.substring(this.startTokenPos, this.currentLinePos)
      }

      if (ch === '.') {
        ch = this.skipFraction(ch)

        if (ch === 'e' || ch === 'E') {
          ch = this.getNextChar()
          this.skipExponent(ch)
        }
        return this.currentLine.substring(this.startTokenPos, this.currentLinePos)
      }

      if (illegalOctal) {
        throw new JsnParseException(`Illegal octal number on this line "${this.currentLineRaw }".`, this.getRealCurrentLineNb() + 1, this.currentLine)
      }

      if (ch) {
        this.currentLinePos--
      }
      const number = this.currentLine.substring(this.startTokenPos, this.currentLinePos)
      return number==='0'? 0 : number
    }

    this.skipMantissa(ch)
    ch = this.getNextChar()
    if (ch === 'e' || ch === 'E') {
      this.skipExponent(ch)
      return this.currentLine.substring(this.startTokenPos, this.currentLinePos)
    }

    if (ch === '.') {
      ch = this.skipFraction(ch)
      if (ch === 'e' || ch === 'E') {
        this.getNextChar()
      }
      return parseFloat(this.currentLine.substring(this.startTokenPos, this.currentLinePos))
    }

    if (ch) {
      this.currentLinePos--
    }
    return parseInt(this.currentLine.substring(this.startTokenPos, this.currentLinePos), 10)
  }

  skipMantissa(ch) {
    while (this.isDecimal(ch)) {
      ch = this.getNextChar()
    }
    if (ch) {
      this.currentLinePos--
    }
  }

  skipFraction(ch) {
    if (ch === '.') {
      ch = this.peekNextChar()
      this.skipMantissa(ch)
    }
  }

  skipExponent(ch) {
    if (ch === 'e' || ch === 'E') {
      ch = this.getNextChar()
      if (ch === '-' || ch === '+') {
        ch = this.getNextChar()
      }
      this.skipMantissa(ch)
    }
  }

  isLetter(ch) {
    return 'a' <= ch && ch <= 'z' || 'A' <= ch && ch <= 'Z' || ch === '_'
  }

  isDigit(ch) {
    return '0' <= ch && ch <= '9'
  }

  isDecimal(ch) {
    return '0' <= ch && ch <= '9'
  }

  isHexadecimal(ch) {
    return '0' <= ch && ch <= '9' || 'a' <= ch && ch <= 'f' || 'A' <= ch && ch <= 'F'
  }

  isWhitespace(ch) {
    return ch === ' ' || ch === '\t' || ch === '\n' || ch === '\r'
  }

  digitVal(ch) {
    if ('0' <= ch && ch <= '9') {
      const digitChar = ch - '0' + 0
      return parseInt(digitChar, 10)
    } else if ('a' <= ch && ch <= 'f') {
      return 'abcdef'.indexOf(ch) + 10
    } else if ('A' <= ch && ch <= 'F') {
      return 'ABCDEF'.indexOf(ch) + 10
    }
    return 16
  }

  getNextChar(notPastEOL) {
    if (this.currentLinePos>this.currentLineRaw.length-1) {
      if (notPastEOL) {
        return null
      } else {
        if (!this.moveToNextLine()) {
          return null
        }
        this.currentLinePos=0
      }
    }
    const nextChar = this.currentLineRaw.charAt(this.currentLinePos)
    this.currentLinePos++
    return nextChar
  }

  peekNextChar() {
    if (this.currentLinePos>this.currentLineRaw.length-1) {
      if (this.currentLineNb >= this.lines.length - 1) {
        return null
      } else {
        return this.lines[this.currentLineNb+1].charAt(0)
      }
    } else {
      return this.currentLineRaw.charAt(this.currentLinePos)
    }
  }

  moveToNextLine() {
    if (this.currentLineNb >= this.lines.length - 1) {
      return false
    }

    this.lastLineRaw = this.currentLineRaw
    this.lastLineNum = this.getRealCurrentLineNb()
    this.lastLinePos = this.currentLinePos
    this.currentLineNb++
    this.currentLine = this.currentLineRaw = this.lines[this.currentLineNb]
    const inx = this.currentLineRaw.lastIndexOf('#')
    if (inx !== -1) {
      this.currentLineRaw = this.currentLineRaw.substr(0, inx - 1)
    }
    this.currentLinePos = 0
    return true
  }

  moveToPreviousLine() {
    this.currentLineNb--
    this.currentLine = this.currentLineRaw = this.lines[this.currentLineNb]
    const inx = this.currentLineRaw.lastIndexOf('#')
    if (inx !== -1) {
      this.currentLineRaw = this.currentLineRaw.substr(0, inx - 1)
    }
    this.currentLinePos = 0
  }

  getRealCurrentLineNb(obj) {
    const inxNb = this.currentLine.lastIndexOf('#')
    if (inxNb !== -1) {
      const row = parseInt(this.currentLine.substr(inxNb + 1), 10)
      if (obj) {
        obj.$r = row
        const inxCmt = this.currentLine.lastIndexOf('#', inxNb - 1)
        if (inxCmt !== -1) {
          obj.$cmt = this.currentLine.substring(inxCmt + 1, inxNb - 1).trim()
        }
        return obj
      } else {
        return row
      }
    }
    return this.currentLineNb + this.offset
  }

  isCurrentLineEmpty() {
    return this.isCurrentLineBlank() || this.isCurrentLineComment()
  }

  isCurrentLineBlank() {
    return '' === this.trim(this.currentLine)
  }

  isCurrentLineComment() {
    //checking explicitly the first char of the trim is faster than loops or strpos
    const ltrimmedLine = this.currentLine.replace(/^ +/g, '').trim()
    return ltrimmedLine[0] === '#'
  }

  isObject(input) {
    return typeof(input) === 'object' && this.isDefined(input)
  }

  isEmpty(input) {
    return input === undefined || input === null || input === '' || input === 0 || input === '0' || input === false
  }

  isDefined(input) {
    return input !== undefined && input !== null
  }

  reverseArray(input /* Array */ ) {
    const result = []
    const len = input.length
    for (let i = len - 1; i >= 0; i--) {
      result.push(input[i])
    }

    return result
  }

  trim(str /* String */ ) {
    return (str + '').replace(/^ +/, '').replace(/ +$/, '')
  }
}

export default JsonParser
