/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2018. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import _ from 'lodash'
import http from './http-util'
import { RESOURCE_TYPES } from '../shared/constants'

const headerContextPath = http.getContextRoot()

const K8S_API_PATH = `${headerContextPath}/kubernetes`,
      UNIFIED_ROUTER_PATH = `${headerContextPath}/unified-router`,
      IMAGE_MANAGER_PATH = `${headerContextPath}/image-manager`,
      USER_MGMT_PATH = `${headerContextPath}/identity`,
      ACTION_API_URL = '/apis/action.open-cluster-management.io/v1beta1'

const DEFAULT_OPTIONS = {
  credentials: 'same-origin',
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
    'X-Requested-With': 'XMLHttpRequest'
  }
}

const getPostOptions = body => {
  return Object.assign({}, DEFAULT_OPTIONS, {
    method: 'POST',
    body: JSON.stringify(body)
  })
}
const getPutOptions = body => {
  return Object.assign({}, DEFAULT_OPTIONS, {
    method: 'PUT',
    body: JSON.stringify(body)
  })
}

const getDeleteOptions = () => {
  return Object.assign({}, DEFAULT_OPTIONS, {
    method: 'DELETE'
  })
}
const getParams = (type, data) => {
  let params = ''
  if (type) {
    params += `?${type}=`
    for (const key of Object.keys(data)) {
      params += `${key}=${data[key]},`
    }
    params = params.substring(0, params.length - 1)
  }
  return params
}

class KubernetesClient {
  get = {
    [RESOURCE_TYPES.DAEMON_SET.name]: this.getDaemonSets,
    [RESOURCE_TYPES.CONFIG_MAP.name]: this.getConfigMaps,
    [RESOURCE_TYPES.DEPLOYMENT.name]: this.getDeployments,
    [RESOURCE_TYPES.JOB.name]: this.getJobs,
    [RESOURCE_TYPES.CRON_JOB.name]: this.getCronJobs,
    [RESOURCE_TYPES.IMAGE.name]: this.getImages,
    [RESOURCE_TYPES.NODE.name]: this.getNodes,
    [RESOURCE_TYPES.PERSISTENT_VOLUME.name]: this.getPersistentVolumes,
    [RESOURCE_TYPES.PERSISTENT_VOLUME_CLAIM.name]: this.getPersistentVolumeClaims,
    [RESOURCE_TYPES.HORIZONTAL_POD_AUTOSCALER.name]: this.getPolicies,
    [RESOURCE_TYPES.SERVICE.name]: this.getServices,
    [RESOURCE_TYPES.INGRESS.name]: this.getIngress,
    [RESOURCE_TYPES.STATEFUL_SET.name]: this.getStatefulSets,
    [RESOURCE_TYPES.NETWORK_POLICY.name]: this.getNetworkPolicies,
    [RESOURCE_TYPES.NAMESPACE.name]: this.getNamespaces,
    [RESOURCE_TYPES.RESOURCE_QUOTA.name]: this.getQuotas,
    [RESOURCE_TYPES.POD_SECURITY_POLICY.name]: this.getPodSecurityPolicies,
    [RESOURCE_TYPES.REPLICA_SET.name]: this.getReplicaSets,
    [RESOURCE_TYPES.POD.name]: this.getPods,
    [RESOURCE_TYPES.EVENT.name]: this.getEvents,
    [RESOURCE_TYPES.LDAP.name]: this.getLdap,
    [RESOURCE_TYPES.TEAM.name]: this.getTeams,
    [RESOURCE_TYPES.TEAM_USER.name]: this.getTeamUsers,
    [RESOURCE_TYPES.TEAM_USERGROUP.name]: this.getTeamUsergroups,
    [RESOURCE_TYPES.TEAM_RESOURCE.name]: this.getTeamResources,
    [RESOURCE_TYPES.USER_RESOURCE.name]: this.getUserResources,
    [RESOURCE_TYPES.USER_TEAM.name]: this.getUserTeams,
    [RESOURCE_TYPES.USER.name]: this.getUsers,
    [RESOURCE_TYPES.SECRET.name]: this.getSecrets,
    [RESOURCE_TYPES.VA.name]: this.getUserNamespaces,
    [RESOURCE_TYPES.CLUSTER_SERVICE_BROKER.name]: this.getClusterServiceBrokers,
    [RESOURCE_TYPES.CLUSTER_SERVICE_CLASS.name]: this.getClusterServiceClasses,
    [RESOURCE_TYPES.CLUSTER_SERVICE_PLAN.name]: this.getClusterServicePlans,
    [RESOURCE_TYPES.SERVICE_INSTANCE.name]: this.getServiceInstances,
    [RESOURCE_TYPES.SERVICE_BINDING.name]: this.getServiceBindings,
    [RESOURCE_TYPES.IMAGE_POLICIES.name]: this.getImagePolicies,
    [RESOURCE_TYPES.CLUSTER_IMAGE_POLICY.name]: this.getClusterImagePolicies,
    [RESOURCE_TYPES.IMAGE_POLICY.name]: this.getNamespaceImagePolicies
  }

  post = this.postResources
  put = this.putResources
  del = this.deleteResources

  createAction(namespace, body, successCB, errorCB) {
    if(namespace) {
      const options = getPostOptions(body)
      return http.fetch(
        `${K8S_API_PATH}${ACTION_API_URL}/namespaces/${namespace}/managedclusteractions`,
        successCB,
        errorCB,
        options
      )
    }
    else {
      throw new Error('Namespace must be set for create action')
    }
  }

  getActions(namespace, successCB, errorCB) {
    if(namespace) {
      return http.fetch(
        `${K8S_API_PATH}${ACTION_API_URL}/namespaces/${namespace}/managedclusteractions`,
        successCB,
        errorCB,
        DEFAULT_OPTIONS
      )
    }
    else {
      throw new Error('Namespace must be set for get action')
    }
  }

  getVersion(successCB, errorCB) {
    return http.fetch(`${K8S_API_PATH}/version`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getDaemonSets(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.DAEMON_SET.api_prefix}/${RESOURCE_TYPES.DAEMON_SET.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}daemonsets${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getConfigMaps(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CONFIG_MAP.api_prefix}/${RESOURCE_TYPES.CONFIG_MAP.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}configmaps${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getDeployments(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.DEPLOYMENT.api_prefix}/${RESOURCE_TYPES.DEPLOYMENT.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}deployments${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getJobs(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.JOB.api_prefix}/${RESOURCE_TYPES.JOB.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}jobs${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getIngress(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.INGRESS.api_prefix}/${RESOURCE_TYPES.INGRESS.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    const postParams = params ? getParams('labelSelector', params) : ''
    return http.fetch(
      `${preFix}${ns}ingresses${postFix}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getPolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.HORIZONTAL_POD_AUTOSCALER.api_prefix}/${RESOURCE_TYPES.HORIZONTAL_POD_AUTOSCALER.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}horizontalpodautoscalers${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getNetworkPolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.NETWORK_POLICY.api_prefix}/${RESOURCE_TYPES.NETWORK_POLICY.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}networkpolicies${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getCronJobs(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CRON_JOB.api_prefix}/${RESOURCE_TYPES.CRON_JOB.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}cronjobs${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getImages(namespace, name, successCB, errorCB) {
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${IMAGE_MANAGER_PATH}/api/v1/repositories${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getNodes(namespace, name, successCB, errorCB) {
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${UNIFIED_ROUTER_PATH}/api/v1/nodedetail${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getPersistentVolumes(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.PERSISTENT_VOLUME.api_prefix}/${RESOURCE_TYPES.PERSISTENT_VOLUME.api_version}`
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}/persistentvolumes${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getPersistentVolumeClaims(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.PERSISTENT_VOLUME_CLAIM.api_prefix}/${RESOURCE_TYPES.PERSISTENT_VOLUME_CLAIM.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}persistentvolumeclaims${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getServices(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.SERVICE.api_prefix}/${RESOURCE_TYPES.SERVICE.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    const postParams = params ? getParams('labelSelector', params) : ''
    return http.fetch(
      `${preFix}${ns}services${postFix}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getStatefulSets(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.STATEFUL_SET.api_prefix}/${RESOURCE_TYPES.STATEFUL_SET.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}statefulsets${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getNamespaces(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.NAMESPACE.api_prefix}/${RESOURCE_TYPES.NAMESPACE.api_version}`
    return http.fetch(
      `${preFix}/users/${params}/namespaces`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getQuotas(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.RESOURCE_QUOTA.api_prefix}/${RESOURCE_TYPES.RESOURCE_QUOTA.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}resourcequotas${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getPodSecurityPolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.POD_SECURITY_POLICY.api_prefix}/${RESOURCE_TYPES.POD_SECURITY_POLICY.api_version}`
    return http.fetch(
      `${preFix}/podsecuritypolicies`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getReplicaSets(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.REPLICA_SET.api_prefix}/${RESOURCE_TYPES.REPLICA_SET.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    const postParams = params ? getParams('labelSelector', params) : ''
    return http.fetch(
      `${preFix}${ns}replicasets${postFix}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getPods(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.POD.api_prefix}/${RESOURCE_TYPES.POD.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    const postParams = params ? getParams('labelSelector', params) : ''
    return http.fetch(
      `${preFix}${ns}pods${postFix}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getEvents(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.EVENT.api_prefix}/${RESOURCE_TYPES.EVENT.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postParams = params ? getParams('fieldSelector', params) : ''
    return http.fetch(
      `${preFix}${ns}events${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getLdap(name, namespace, successCB, errorCB) {
    const preFix = `${USER_MGMT_PATH}/api/v1/directory/ldap`
    const postFix = name ? `/${name}` : '/list'
    return http.fetch(`${preFix}${postFix}`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getTeams(namespace, name, successCB, errorCB) {
    const preFix = `${USER_MGMT_PATH}/api/v1/teams`
    const postFix = name ? `/${name}` : ''
    return http.fetch(`${preFix}${postFix}`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getTeamUsers(name, namespace, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/teams/${name}/users`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getTeamUsergroups(name, namespace, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/teams/${name}/usergroups`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getUsers(name, namespace, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/usergroup/${name}/getUsers`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getTeamResources(name, namespace, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/teams/${name}/resources`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getUserResources(name, namespace, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/teamResources/${name}`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getUserTeams(namespace, name, successCB, errorCB, params) {
    const preFix = `${USER_MGMT_PATH}/api/v1/users/${params}/getTeams`
    const ns = namespace ? `?namespace=${namespace}` : ''
    return http.fetch(`${preFix}${ns}`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getUserNamespaces(user, successCB, errorCB) {
    return http.fetch(`${USER_MGMT_PATH}/api/v1/users/${user}/getUserNamespaces`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getSecrets(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.SECRET.api_prefix}/${RESOURCE_TYPES.SECRET.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}secrets${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getClusterServiceBrokers(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CLUSTER_SERVICE_BROKER.api_prefix}/${RESOURCE_TYPES.CLUSTER_SERVICE_BROKER.api_version}`
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}/clusterservicebrokers${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getServiceInstances(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.SERVICE_INSTANCE.api_prefix}/${RESOURCE_TYPES.SERVICE_INSTANCE.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}${ns}serviceinstances${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getClusterServiceClasses(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CLUSTER_SERVICE_CLASS.api_prefix}/${RESOURCE_TYPES.CLUSTER_SERVICE_CLASS.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postParams = params ? getParams('fieldSelector', params) : ''
    return http.fetch(
      `${preFix}/clusterserviceclasses${ns}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getClusterServicePlans(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CLUSTER_SERVICE_PLAN.api_prefix}/${RESOURCE_TYPES.CLUSTER_SERVICE_PLAN.api_version}`
    const postFix = name ? `/${name}` : ''
    const postParams = params ? getParams('fieldSelector', params) : ''
    return http.fetch(
      `${preFix}/clusterserviceplans${postFix}${postParams}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getServiceBindings(namespace, name, successCB, errorCB, params) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.SERVICE_BINDING.api_prefix}/${RESOURCE_TYPES.SERVICE_BINDING.api_version}`
    const ns = namespace ? `/namespaces/${namespace}/` : '/'
    const postFix = name ? `/${name}` : ''
    return http.fetchWithReduce(
      `${preFix}${ns}servicebindings${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS,
      params
    )
  }
  getImagePolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.IMAGE_POLICIES.api_prefix}/${RESOURCE_TYPES.IMAGE_POLICIES.api_version}`
    return http.fetch(`${preFix}/imagepolicies`, successCB, errorCB, DEFAULT_OPTIONS)
  }
  getClusterImagePolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.CLUSTER_IMAGE_POLICY.api_prefix}/${RESOURCE_TYPES.CLUSTER_IMAGE_POLICY.api_version}`
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}/clusterimagepolicies${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  getNamespaceImagePolicies(namespace, name, successCB, errorCB) {
    const preFix = `${K8S_API_PATH}/${RESOURCE_TYPES.IMAGE_POLICY.api_prefix}/${RESOURCE_TYPES.IMAGE_POLICY.api_version}`
    const postFix = name ? `/${name}` : ''
    return http.fetch(
      `${preFix}/namespaces/${namespace}/imagepolicies${postFix}`,
      successCB,
      errorCB,
      DEFAULT_OPTIONS
    )
  }
  postResources(namespace, body, successCB, errorCB) {
    const options = getPostOptions(body)
    return http.fetch(`${K8S_API_PATH}/`, successCB, errorCB, options)
  }
  putResources(namespace, body, successCB, errorCB) {
    const options = getPutOptions(body)
    const resourceType = _.snakeCase(body.kind).toUpperCase()
    const resource = getResourceType(resourceType)
    switch (resource.name) {
    case RESOURCE_TYPES.PERSISTENT_VOLUME.name:
    case RESOURCE_TYPES.POD_SECURITY_POLICY.name:
      return http.fetch(
        `${K8S_API_PATH}/${resource.api_prefix}/${body.apiVersion}/${resource.resource}/${body.metadata.name}`,
        successCB,
        errorCB,
        options
      )
    case RESOURCE_TYPES.IMAGE.name:
      return http.fetch(
        `${IMAGE_MANAGER_PATH}/api/v1/repositories/${encodeURIComponent(body.name)}`,
        successCB,
        errorCB,
        options
      )
    default:
      if(namespace) {
        return http.fetch(
          `${K8S_API_PATH}/${resource.api_prefix}/${body.apiVersion}/namespaces/${namespace}/${resource.resource}/${body.metadata.name}`,
          successCB,
          errorCB,
          options
        )
      }
      else {
        throw new Error(`Namespace must be set for put resource ${_.get(body, 'metadata.name', '')}`)
      }
    }
  }
  deleteResources(body, successCB, errorCB) {
    const options = getDeleteOptions()
    const resourceType = _.snakeCase(body.kind).toUpperCase()
    const resource = getResourceType(resourceType)
    switch (resource.name) {
    case RESOURCE_TYPES.POD_SECURITY_POLICY.name:
    case RESOURCE_TYPES.PERSISTENT_VOLUME.name:
    case RESOURCE_TYPES.CLUSTER_SERVICE_BROKER.name:
    case RESOURCE_TYPES.CLUSTER_IMAGE_POLICY.name:
      return http.fetch(
        `${K8S_API_PATH}/${resource.api_prefix}/${body.apiVersion}/${resource.resource}/${body.metadata.name}?orphanDependents=false`,
        successCB,
        errorCB,
        options
      )
    case RESOURCE_TYPES.IMAGE.name:
      return http.fetch(
        `${IMAGE_MANAGER_PATH}/api/v1/repositories/${encodeURIComponent(body.name)}`,
        successCB,
        errorCB,
        options
      )
    case RESOURCE_TYPES.NAMESPACE.name:
      return http.fetch(
        `${K8S_API_PATH}/${RESOURCE_TYPES.NAMESPACE.api_prefix}/${RESOURCE_TYPES.NAMESPACE.api_version}/namespaces/${body.metadata.name}/${encodeURIComponent(body.crn)}`,
        successCB,
        errorCB,
        options
      )
    default:
      return http.fetch(
        `${K8S_API_PATH}/${resource.api_prefix}/${body.apiVersion}/namespaces/${body.metadata.namespace}/${resource.resource}/${body.metadata.name}?orphanDependents=false`,
        successCB,
        errorCB,
        options
      )
    }

  }
}
function getResourceType(type) {
  const resource = RESOURCE_TYPES[type]
  if(!resource) {
    throw new Error(`Unsupported resource type '${type}' specified`)
  }
  return resource
}

export default new KubernetesClient()
