/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import jsYaml from 'js-yaml'
import YamlParser from './yaml-parser'
import JsonParser from './json-parser'
import _ from 'lodash'

export const parseYAML = (yaml, validator, locale) => {
  let absLine=0
  const parsed = {}
  const yamls = yaml.split(/^---$/gm)
  const exceptions = []
  // check for syntax errors
  try {
    yamls.forEach((snip)=>{
      const obj = jsYaml.safeLoad(snip)
      const key = _.get(obj, 'kind', 'unknown')
      let values = parsed[key]
      if (!values) {
        values = parsed[key] = []
      }
      const post = snip.endsWith('\n')
      snip = snip.trim()
      const $synced = new YamlParser().parse(snip, absLine)
      $synced.$r = absLine
      $synced.$l = snip.split('\n').length
      values.push({$raw: obj, $yml: snip, $synced})
      absLine += $synced.$l
      if (post) {
        absLine++
      }
    })
  } catch (e) {
    const {mark={}, reason, message} = e
    const {line=0, column=0} = mark
    exceptions.push({
      row: line+absLine,
      column,
      text: _.capitalize(reason||message),
      type: 'error',
    })
  }

  // validate
  if (validator && exceptions.length===0) {
    validator(parsed, exceptions, locale)
  }


  return {parsed, exceptions}
}

export const parseJSON = (json) => {
  let parsed = {}
  const exceptions = []
  // check for syntax errors
  try {
    json = json.trim()
    const $synced = new JsonParser().parse(json, 0)
    $synced.$r = 0
    $synced.$l = json.split('\n').length
    parsed ={$raw: JSON.parse(json), $json: json, $synced}
  } catch (e) {
    const {parsedLine, message} = e
    exceptions.push({
      row: parsedLine,
      column: 0,
      text: _.capitalize(message),
      type: 'error',
    })
  }
  return {parsed, exceptions}
}
