/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2018. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import msgs from '../../nls/platform.properties'

const request = require('./request'),
      config = require('../../config'),
      httpUtil = require('../server/http-util'),
      serviceAccount = require('./service-account'),
      cookieUtil = require('./cookie-util'),
      _ = require('lodash'),
      context = require('../../lib/shared/context')

exports.proxy = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL') + removeKubeKeyword(req.path)}`)
  doRequest(req, options, cb)
}

exports.get = (path, cb) => {
  const options = {
    url: `${config.get('API_SERVER_URL')}${removeKubeKeyword(path)}`,
    headers: {
      Cookie: httpUtil.getAuth()
    },
    json: true
  }
  request(options, null, [200, 201, 202], (err, res) => {
    if (err) {
      return cb(err, null)
    }
    cb(err, res.body)
  })
}

exports.getJSON = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL')}${removeKubeKeyword(req.path)}`)
  options.json = req.body
  doRequest(req, options, cb)
}

exports.getWithServiceCredentials = (path, cb) => {
  const options = {
    url: `${config.get('API_SERVER_URL')}${removeKubeKeyword(path)}`,
    headers: {
      Authorization: `Bearer ${serviceAccount.getServiceCredentials()}`
    },
    json: true,
    agentOptions: {
      ca: serviceAccount.getCACert()
    }
  }
  request(options, null, [200, 201, 202], (err, res) => {
    if (err) {
      return cb(err, null)
    }
    cb(err, res.body)
  })
}

exports.getWithReq = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL')}${removeKubeKeyword(req.path)}`)
  options.method = 'GET'
  doRequest(req, options, cb)
}

exports.post = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL')}${removeKubeKeyword(req.path)}`)
  options.method = 'POST'
  options.json = req.body
  doRequest(req, options, cb)
}

exports.put = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL')}${removeKubeKeyword(req.path)}`)
  options.method = 'PUT'
  options.json = req.body
  doRequest(req, options, cb)
}

exports.del = (req, cb) => {
  const options = httpUtil.getOptions(req, `${config.get('API_SERVER_URL')}${removeKubeKeyword(req.path)}`)
  options.method = 'DELETE'
  doRequest(req, options, cb)
}

// remove unnecessary kubernetes keyword for external request, which is passed from internal URL string
const removeKubeKeyword = (URLString) => {
  return URLString.replace('kubernetes/', '')
}

function doRequest(req, options, cb) {
  options.headers = {
    Authorization: `Bearer ${cookieUtil.getAccessToken(req)}`
  }
  request(options, null, [200, 201, 202], (err, res) => {
    if (err) {
      if (err.message) {
        err.message = err.message.replace(/\r?\n|\r/, '')
      }
      return cb(err, null)
    }
    cb(err, res.body)
  })
}

// getK8sPaths, getK8sConfig, postK8sResource used to POST k8s resource with introspection

// Fetch all k8s api paths and match with the api version supplied in k8s resource config to obtain the api prefix
exports.getK8sPaths = (req, cb) => {
  module.exports.getJSON(req, (err, body) => {
    if (err) {
      return cb(err)
    }

    const { apiVersion } = req.body
    const k8sPaths = body.paths
    const apiPath = k8sPaths.find(path => path.match(`/[0-9a-zA-z]*/?${apiVersion}`))
    return cb(null, req, removeKubeKeyword(apiPath))
  })
}

// Query the api path from getK8sPath and pull the data for the resource that is being created
exports.getK8sConfig = (req, apiPath, cb) => {
  module.exports.getJSON({ path: removeKubeKeyword(apiPath), ...req }, (err, body) => {
    if (err) {
      return cb(err)
    }

    const { kind } = req.body
    const k8sResourceList = body.resources
    const resourceType = k8sResourceList.find(resource => resource.kind === kind)
    if (resourceType === undefined) {
      return cb({
        message: msgs.get('modal.create.group.error', context(req).locale)
      })
    }
    return cb(null, req, apiPath, resourceType)
  })
}

// Build the URL with the data from getK8sPaths and getK8sConfig and POST
exports.postK8sResource = (req, apiPath, resourceType, cb) => {
  const { name, namespaced } = resourceType
  const namespace = _.get(req, 'body.metadata.namespace')
  if (namespaced && !namespace) {
    return cb({message: 'namespace must be set'})
  }
  const nsPrefixed = namespaced ? `namespaces/${namespace}/` : ''
  const path = `${removeKubeKeyword(apiPath)}/${nsPrefixed}${name}`
  return module.exports.post({ path, ...req }, (err, result) => cb(err, result))
}
