/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

const request = require('./request'),
      config = require('../../config'),
      log4js = require('log4js'),
      logger = log4js.getLogger('namespace-client')

import _ from 'lodash'

// get user access info on single namespace
exports.getUserAccess = (req, cb) => {
  const k8sAPI = '/apis/authorization.k8s.io/v1'
  const options = {
    method: 'POST',
    url: `${config.get('API_SERVER_URL')}${k8sAPI}/selfsubjectrulesreviews`,
    json: true,
    headers: {
      'Content-Type': 'application/json',
      'Accept': 'application/json',
      Authorization: req.authorizationToken
    },
    body: {
      apiVersion: 'authorization.k8s.io/v1',
      kind: 'SelfSubjectRulesReview',
      spec: {
        namespace: req.singleNS
      },
    }
  }

  request(options, null, [200,201], (err, res) => {
    if (err) {
      return cb(err, null)
    }
    cb(null, userAccessFormatter(req, res))
  }, logger)
}

// format/combine/simplify/deal with * for user access raw data on single NS
function userAccessFormatter(req, accessInfo) {
  const targetAPIGroups = new Set(req.targetAPIGroups)
  const singleNSAccess ={
    namespace: req.singleNS,
    rules: {},
  }
  // if user query raw = true, also return original raw access data on each namespace
  if (req.rawDataFlag) {
    singleNSAccess.rawData = accessInfo
  }
  const resourceRules = _.get(accessInfo,'body.status.resourceRules','')
  if (Array.isArray(resourceRules) && resourceRules.length > 0) {
    resourceRules.forEach((resourceRule) => {
      if (Array.isArray(resourceRule.apiGroups)) {
        const apiGroups = _.compact(resourceRule.apiGroups)
        resourceRule.apiGroups.length > 0 && apiGroups.forEach((api) => {
          // no matter if user want *, always return these special cases
          if (Array.isArray(resourceRule.resources) && (targetAPIGroups.has(api) || api === '*')) {
            const resources = _.compact(resourceRule.resources)
            const verbs = _.compact(resourceRule.verbs)
            resourceRule.resources.length > 0 && resources.forEach((resource) => {
              // use the combined api + resource as the unique key
              const mapKey = `${api}/${resource}`
              if (Object.prototype.hasOwnProperty.call(singleNSAccess.rules, mapKey)) {
                singleNSAccess.rules[mapKey] = _.union(singleNSAccess.rules[mapKey], verbs)
              }
              else {
                singleNSAccess.rules[mapKey] = verbs
              }
            })
          }
        })
      }
    })
  }
  return singleNSAccess
}
