/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
import k8sClient from '../../lib/client/k8s-client'
import _ from 'lodash'
import { namespacesReceiveSuccess, namespacesReceiveFailure } from './namespaces'
import {
  TABLE_PAGE_CHANGE, TABLE_SEARCH, TABLE_SORT, RESOURCE_RECEIVE_SUCCESS,
  REQUEST_STATUS, RESOURCE_RECEIVE_FAILURE, RESOURCE_REQUEST, RESOURCE_ADD,
  RESOURCE_MODIFY, RESOURCE_DELETE, SECONDARY_HEADER_UPDATE, TABLE_RESET,
  TABLE_FILTER, TABLE_SELECT, MODAL_UPDATE, POST_REQUEST, POST_RECEIVE_SUCCESS,
  POST_RECEIVE_FAILURE, PUT_REQUEST, PUT_RECEIVE_SUCCESS, PUT_RECEIVE_FAILURE,
  DEL_REQUEST, RESOURCE_UPDATE, SECONDARY_HEADER_UPDATE_CLASSES, TOAST_NOTIFICATION_UPDATE,
  DEL_RECEIVE_SUCCESS, DEL_RECEIVE_FAILURE, CLEAR_REQUEST_STATUS, RESOURCE_RESET,
} from './index'

export const changeTablePage = ({page, pageSize}, resourceType) => ({
  type: TABLE_PAGE_CHANGE,
  page,
  pageSize,
  resourceType
})

export const searchTable = (search, resourceType) => ({
  type: TABLE_SEARCH,
  search,
  resourceType
})

export const sortTable = (sortDirection, sortColumn, resourceType) => ({
  type: TABLE_SORT,
  sortDirection,
  sortColumn,
  resourceType
})

export const resetTable = resourceType => ({
  type: TABLE_RESET,
  resourceType
})

export const filterTable = (filterType, resourceType) => ({
  type: TABLE_FILTER,
  filterType,
  resourceType
})

export const modifyTable = (resource, key, resourceType) => ({
  type: TABLE_SELECT,
  resource,
  key,
  resourceType
})

export const receiveResourceSuccess = (response, resourceType) => ({
  type: RESOURCE_RECEIVE_SUCCESS,
  status: REQUEST_STATUS.DONE,
  items: response.items,
  resourceVersion: _.get(response, 'metadata.resourceVersion'), //only supported on k8s resoruces
  resourceType,
  metricsDisabled: response.metricsDisabled
})

export const receiveResourceError = (err, resourceType) => ({
  type: RESOURCE_RECEIVE_FAILURE,
  status: REQUEST_STATUS.ERROR,
  err,
  resourceType
})

export const requestResource = (resourceType) => ({
  type: RESOURCE_REQUEST,
  status: REQUEST_STATUS.IN_PROGRESS,
  resourceType
})

export const addResource = (item, resourceType) => ({
  type: RESOURCE_ADD,
  resourceType: item.kind || resourceType,
  item
})

export const modifyResource = (item, resourceType) => ({
  type: RESOURCE_MODIFY,
  resourceType: item.kind || resourceType,
  item
})

export const updateResource = (items, resourceType) => ({
  type: RESOURCE_UPDATE,
  resourceType,
  items
})

export const deleteResource = (item, resourceType) => ({
  type: RESOURCE_DELETE,
  resourceType:  item.kind || resourceType,
  item
})

export const fetchResources = (resourceType, namespace, params) => (dispatch => {
  dispatch(requestResource(resourceType))
  return k8sClient.get[resourceType.name](
    namespace,
    null,
    response => dispatch(receiveResourceSuccess(response, resourceType)),
    err => dispatch(receiveResourceError(err, resourceType)),
    params
  )
})

export const fetchResource = (resourceType, namespace, name, params) => (dispatch => {
  return k8sClient.get[resourceType.name](
    namespace,
    name,
    response => dispatch(modifyResource(response, resourceType)),
    err => dispatch(receiveResourceError(err, resourceType)),
    params
  )
})

export const updateSecondaryHeader = (title, tabs, breadcrumbItems, links, actions) => ({
  type: SECONDARY_HEADER_UPDATE,
  title,
  tabs,
  breadcrumbItems,
  links,
  actions
})
export const updateSecondaryHeaderClasses = (headerClasses) => ({
  type: SECONDARY_HEADER_UPDATE_CLASSES,
  headerClasses
})
export const updateModal = (data) => ({
  type: MODAL_UPDATE,
  data
})

export const postResource = (resourceType) => ({ // TODO: Consider renaming
  type: POST_REQUEST,
  postStatus: REQUEST_STATUS.IN_PROGRESS,
  resourceType
})

export const receivePostResource = (item, resourceType) => ({
  type: POST_RECEIVE_SUCCESS,
  postStatus: REQUEST_STATUS.DONE,
  resourceType: item.kind || resourceType,
  item
})

export const receivePostError = (err, resourceType) => ({
  type: POST_RECEIVE_FAILURE,
  postStatus: REQUEST_STATUS.ERROR,
  err,
  resourceType
})

export const putResource = (resourceType) => ({ // TODO: Consider renaming
  type: PUT_REQUEST,
  putStatus: REQUEST_STATUS.IN_PROGRESS,
  resourceType
})

export const receivePutResource = (item, resourceType) => ({
  type: PUT_RECEIVE_SUCCESS,
  putStatus: REQUEST_STATUS.DONE,
  resourceType: item.kind || resourceType,
  item
})

export const receivePutError = (err, resourceType) => ({
  type: PUT_RECEIVE_FAILURE,
  putStatus: REQUEST_STATUS.ERROR,
  err,
  resourceType
})

export const delResource = (resourceType) => ({ // TODO: Consider renaming
  type: DEL_REQUEST,
  delStatus: REQUEST_STATUS.IN_PROGRESS,
  resourceType
})

export const receiveDelResource = (item, resourceType) => ({
  type: DEL_RECEIVE_SUCCESS,
  delStatus: REQUEST_STATUS.DONE,
  resourceType: item.kind || resourceType,
  item
})

export const receiveDelError = (err, resourceType) => ({
  type: DEL_RECEIVE_FAILURE,
  delStatus: REQUEST_STATUS.ERROR,
  err,
  resourceType
})

export const clearRequestStatus = (resourceType) => ({
  type: CLEAR_REQUEST_STATUS,
  resourceType: resourceType
})

export const resetResource = (resourceType) => ({
  type: RESOURCE_RESET,
  resourceType: resourceType
})

export const createResource = (resourceType, namespace, body, params, role=null) => (dispatch => {
  dispatch(postResource(resourceType))
  return k8sClient.post(
    namespace,
    body,
    response => {
      if (response.error) {
        dispatch(receivePostError(response, resourceType))
      } else {
        dispatch(receivePostResource(response, resourceType))
      }
    },
    err => dispatch(receivePostError(err, resourceType)),
    role
  )
})

export const editResource = (resourceType, namespace, body) => (dispatch => {
  dispatch(putResource(resourceType))
  return k8sClient.put(
    namespace,
    body,
    response => dispatch(receivePutResource(response, resourceType)),
    err => dispatch(receivePutError(err, resourceType))
  )
})

export const removeResource = (resourceType, resource, params) => (dispatch => {
  dispatch(delResource(resourceType))
  if (resourceType.name === 'Namespace') {
    return k8sClient.del(
      resource,
      response => {
        dispatch(receiveDelResource(response, resourceType))
        return k8sClient.getUserNamespaces(
          params,
          res => dispatch(namespacesReceiveSuccess(res.items.filter(item => item.Name !== resource.metadata.name))),
          err => dispatch(namespacesReceiveFailure(err))
        )
      },
      err => dispatch(receiveDelError(err, resourceType))
    )
  } else {
    return k8sClient.del(
      resource,
      response => dispatch(receiveDelResource(response, resourceType)),
      err => dispatch(receiveDelError(err, resourceType))
    )
  }
})

export const setToastNotification = (showToast, toastData) => ({
  type: TOAST_NOTIFICATION_UPDATE,
  showToast,
  toastData
})
