/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************
/* Copyright (c) 2020 Red Hat, Inc. */

import React from 'react'
import { withRouter } from 'react-router-dom'
import { connect } from 'react-redux'
import msgs from '../../nls/platform-header.properties'
import resources from '../../lib/shared/resources'
import { updateModal } from '../actions/common'
import { RESOURCE_TYPES } from '../../lib/shared/constants'
import { getContextRoot } from '../../lib/client/http-util'
import PropTypes from 'prop-types'
import client from'../../lib/shared/client'
import {
  AdminDropdown,
  ApplicationLauncher,
  CreateResources,
  Hamburger,
  InfoDropdown,
  LaunchTerminal,
  LogoContainer,
  MinimizedHeaderDropdown,
  NavigationContainer,
  SearchLink
} from './HeaderItems'
import _ from 'lodash'

resources(() => {
  require('../../scss/header.scss')
})

class Header extends React.PureComponent {
  constructor(props) {
    super(props)
    this.state = {
      componentHasMounted: false,
      hideNavItems: false
    }
  }

  handleWindowResize = () => {
    this.setState({ hideNavItems: window.innerWidth < 992 })
  }

  componentDidMount() {
    this.handleWindowResize()
    window.addEventListener('resize', this.handleWindowResize)
    this.setState({ componentHasMounted : true })
  }

  componentWillUnmount() {
    window.removeEventListener('resize', this.handleWindowResize)
    this.setState({ componentHasMounted : false })
  }

  handleResourceModalOpen = (event) => {
    event.preventDefault()
    this.props.handleCreateResDropdownClick()
    this.props.updateModal(
      { open: true, type:'resource', resourceType: RESOURCE_TYPES.RESOURCES, action: 'post', editorMode: 'yaml',
        label: { primaryBtn: 'modal.button.create', label: 'nav.resource', heading: 'nav.createResource' },
        data: { apiVersion: RESOURCE_TYPES.DEPLOYMENT.api_version, kind: RESOURCE_TYPES.DEPLOYMENT.name }})
  }

  handleAboutWindow = (event) => {
    event.preventDefault()
    this.props.updateModal({open:true, type: 'about-window'})
  }

  handleSetHomepage = (event) => {
    event.preventDefault()
    this.props.handleUserDropdownClick()
  }

  renderHeaderItems() {
    const {
      user,
      appsDropdownOpen,
      terminalDropdownOpen,
      userDropdownOpen,
      infoDropdownOpen,
      minimizedDropdownOpen,
      handleAppsDropdownClick,
      handleInfoDropdownClick,
      handleTerminalDropdownClick,
      handleUserDropdownClick,
      handleMinimizedDropdownClick,
      docLink,
      support,
      handleLogout,
      history,
      experience,
      gettingStartedUrl,
      hasMCMSearch,
      allHeaderConfig,
      hasKUI,
      oauthInfo,
      appLinks
    } = this.props
    const alternateConfig = experience.whichNav
      ? _.find(allHeaderConfig, (item) => item.id === experience.whichNav)
      : null
    const disabledItems = alternateConfig && alternateConfig.header.disabledItem
    const docUrl = experience.whichNav ? _.get(alternateConfig, 'header.docUrlMapping', null) : docLink
    const supportUrl = experience.whichNav ? _.get(alternateConfig, 'header.supportUrl', null ): support
    const gettingStarted = experience.whichNav
      ? _.get(alternateConfig, 'header.gettingStartedUrl', null )
      : gettingStartedUrl
    const headerContextPath = getContextRoot()
    const whichNavParm = experience.whichNav ? `?useNav=${experience.whichNav}` : ''

    if (this.state.hideNavItems) {
      return (
        <NavigationContainer>
          {/* MinimizedHeaderDropdown is only visible when screen width is < 992px */}
          <MinimizedHeaderDropdown
            minimizedDropdownOpen={minimizedDropdownOpen}
            handleMinimizedDropdownClick={handleMinimizedDropdownClick}
            handleAboutWindow={this.handleAboutWindow.bind(this)}
            handleLogout={handleLogout}
            handleResourceModalOpen={this.handleResourceModalOpen.bind(this)}
            hasMCMSearch={hasMCMSearch}
            hasKUI={hasKUI}
            whichNavParm={whichNavParm}
            disabledItems={disabledItems}
            oauthInfo={oauthInfo}
            user={user}
            apps={appLinks}
            docLink={docUrl} />
        </NavigationContainer>
      )
    }
    return (
      <NavigationContainer>
        <ApplicationLauncher
          apps={appLinks}
          appsDropdownOpen={appsDropdownOpen}
          handleAppsDropdownClick={handleAppsDropdownClick} />
        {hasMCMSearch && <SearchLink />}
        {!(disabledItems && disabledItems.includes('createResource')) &&
          <CreateResources handleResourceModalOpen={this.handleResourceModalOpen.bind(this)} />
        }
        {hasKUI && <LaunchTerminal
          whichNavParm={whichNavParm}
          terminalDropdownOpen={terminalDropdownOpen}
          handleTerminalDropdownClick={handleTerminalDropdownClick} />}
        {(docUrl || supportUrl || gettingStarted) && <InfoDropdown
          infoDropdownOpen={infoDropdownOpen}
          handleInfoDropdownClick={handleInfoDropdownClick}
          handleAboutWindow={this.handleAboutWindow.bind(this)}
          docLink={docUrl}
          support={supportUrl}
          gettingStartedUrl={gettingStarted}
          headerContextPath={headerContextPath}
          pathname={history && history.location.pathname}
        />}
        <AdminDropdown
          userDropdownOpen={userDropdownOpen}
          client={client}
          homepage={null}
          user={user}
          handleUserDropdownClick={handleUserDropdownClick}
          handleLogout={handleLogout}
          oauthInfo={oauthInfo} />
      </NavigationContainer>
    )
  }

  render() {
    const {
      endpointAccess,
      leftNavOpen,
      handleMenuClick,
      logo,
      experience,
      altText,
      allHeaderConfig,
    } = this.props
    const locale = this.context.locale
    const alternateConfig = experience.whichNav
      ? _.find(allHeaderConfig, (item) => item.id === experience.whichNav)
      : null
    return (
      <div className='app-header-wrapper'>
        <header className='app-header' aria-label={msgs.get('header.label', locale)}>
          <div className='app-header__container secondary'>
            <Hamburger leftNavOpen={leftNavOpen} handleMenuClick={handleMenuClick} />
            <LogoContainer
              homepage={'/multicloud/welcome'}
              logo={experience.whichNav ? _.get(alternateConfig, 'header.logoUrl', logo) : logo}
              altText={experience.whichNav ? _.get(alternateConfig, 'header.logoAltText', altText) : altText}
            />
            <div className='cluster'></div>
            {this.state.componentHasMounted && this.renderHeaderItems()}
          </div>
          <input type='hidden' id='headerAccess' style={{display: 'none'}} value={endpointAccess} />
        </header>
      </div>
    )
  }
}

Header.contextTypes = {
  locale: PropTypes.string
}

Header.propTypes = {
  allHeaderConfig: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  altText: PropTypes.string,
  appLinks: PropTypes.array,
  appsDropdownOpen: PropTypes.bool,
  docLink: PropTypes.string,
  endpointAccess: PropTypes.string,
  experience: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  gettingStartedUrl: PropTypes.string,
  handleAppsDropdownClick: PropTypes.func,
  handleCreateResDropdownClick: PropTypes.func,
  handleInfoDropdownClick: PropTypes.func,
  handleLogout: PropTypes.func,
  handleMenuClick: PropTypes.func,
  handleMinimizedDropdownClick: PropTypes.func,
  handleTerminalDropdownClick: PropTypes.func,
  handleUserDropdownClick: PropTypes.func,
  hasKUI: PropTypes.bool,
  hasMCMSearch: PropTypes.bool,
  history: PropTypes.object,
  infoDropdownOpen: PropTypes.bool,
  leftNavOpen: PropTypes.bool,
  logo: PropTypes.string,
  minimizedDropdownOpen: PropTypes.bool,
  oauthInfo: PropTypes.object,
  support: PropTypes.string,
  terminalDropdownOpen: PropTypes.bool,
  updateModal: PropTypes.func,
  user: PropTypes.object,
  userDropdownOpen: PropTypes.bool,
}

const mapStateToProps = state => {
  return {
    user: state.user,
    // userPreferences: state.userpreferences,
    role: state.role && state.role.role,
    experience: state.experience,
    logo: _.get(state, 'uiconfig.config.header.logoUrl'),
    altText: _.get(state, 'uiconfig.config.header.logoAltText'),
    allHeaderConfig: _.get(state, 'nav.secondaryNavItems', []),
    appLinks: _.get(state, 'uiconfig.config.appLinks')
  }
}

const mapDispatchToProps = dispatch => {
  return {
    updateModal: (data) => dispatch(updateModal(data)),
    // setHomepage: (userPreferences) => dispatch(setUserPreferences(userPreferences))
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(Header))
