/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************
/* Copyright (c) 2020 Red Hat, Inc. */
import React from 'react'
import msgs from '../../nls/platform-header.properties'
import PropTypes from 'prop-types'

const headerMenuLabelStr = 'header.menu.label'
const navSearchStr = 'nav.search'
const navCreateResourceStr = 'nav.createResource'
const relStr = 'noopener noreferrer'

import {
  AboutIcon,
  ApplicationIcon,
  AppSwitcherIcon,
  CloseMenuIcon,
  ConfigureClientIcon,
  CreateResourceIcon,
  CurrentBrowserTabIcon,
  DocIcon,
  InfoDropdownIcon,
  LaunchIcon,
  LogoutIcon,
  MenuIcon,
  NewBrowserTabIcon,
  OverflowDotsIcon,
  SearchIcon,
  // SupportIcon,
  TerminalIcon,
} from './common/Icons'

export const Hamburger = ({ leftNavOpen, handleMenuClick }, { locale }) =>
  <div className={`app-menu-btn-container${leftNavOpen ? ' is-open' : ''}`}>
    <button className='hamburger-btn' id='hamburger' aria-label={msgs.get(headerMenuLabelStr, locale)} onClick={handleMenuClick}>
      {leftNavOpen ?
        <CloseMenuIcon title={msgs.get(headerMenuLabelStr, locale)} />
        : <MenuIcon title={msgs.get(headerMenuLabelStr, locale)} />}
    </button>
    <div className='multicloud-tooltip'>{msgs.get('tooltip.menu', locale)}</div>
  </div>

Hamburger.propTypes = {
  handleMenuClick: PropTypes.func,
  leftNavOpen: PropTypes.bool,
}

export const LogoContainer = ({  homepage, logo, altText }) =>
  <div className='logo-container'>
    {homepage
      ? <a href={homepage}>
        <img className='logo'
          src={logo}
          alt={altText}
        />
      </a>
      :
      <img className='logo'
        src={logo}
        alt={altText}
      />
    }
  </div>

LogoContainer.propTypes = {
  altText: PropTypes.string,
  homepage: PropTypes.string,
  logo: PropTypes.string,
}

export const ClusterIndicator = ({ clusterName }, { locale }) =>
  <div className='cluster'>
    <p className='cluster__indicator'>{msgs.get('header.cluster', locale)}</p>
    <p className='cluster__name'>{clusterName}</p>
    { clusterName && clusterName.length > 30 && <div className='cluster__tooltip'>{clusterName}</div> }
  </div>

ClusterIndicator.propTypes = {
  clusterName: PropTypes.string,
  length: PropTypes.number,
}

export const NavigationContainer = ({ children }, { locale }) =>
  <div className='navigation-container'>
    <nav aria-label={msgs.get('nav.label', locale)}>
      <ul>
        {children}
      </ul>
    </nav>
  </div>

NavigationContainer.propTypes = {
  children: PropTypes.oneOfType([PropTypes.node, PropTypes.array, PropTypes.object]),
}

// List Items for the NavigationContainer component
export const SearchLink = ({ locale }) =>
  <li id='acm-search' className='header-action'>
    <a href='/search'>
      <SearchIcon title={msgs.get(navSearchStr, locale)} />
    </a>
    <div className='multicloud-tooltip'>{msgs.get(navSearchStr, locale)}</div>
  </li>

SearchLink.propTypes = {
  locale: PropTypes.string,
}

export const MinimizedSearchLink = ({ locale }) =>
  <li id='acm-search' className='dropdown-menu-item'>
    <a href='/search' className='dropwdown-content-items'>
      <SearchIcon title={msgs.get(navSearchStr, locale)} darkFill={true} />
      <span>{msgs.get(navSearchStr, locale)}</span>
    </a>
  </li>

MinimizedSearchLink.propTypes = {
  locale: PropTypes.string,
}

export const CreateResources = ({ handleResourceModalOpen }, { locale }) =>
  <li id='acm-create-resource' className='header-action'>
    <a href='#' id='create-resource' onClick={handleResourceModalOpen}>
      <CreateResourceIcon title={msgs.get(navCreateResourceStr, locale)} />
    </a>
    <div className='multicloud-tooltip'>{msgs.get(navCreateResourceStr, locale)}</div>
  </li>

CreateResources.propTypes = {
  handleResourceModalOpen: PropTypes.func,
}

export const MinimizedCreateResources = ({ handleResourceModalOpen }, { locale }) =>
  <li id='acm-create-resource' className='dropdown-menu-item' >
    <a href='#' id='create-resource' onClick={handleResourceModalOpen} className='dropwdown-content-items'>
      <CreateResourceIcon title={msgs.get(navCreateResourceStr, locale)} darkFill={true} />
      <span>{msgs.get(navCreateResourceStr, locale)}</span>
    </a>
  </li>

MinimizedCreateResources.propTypes = {
  handleResourceModalOpen: PropTypes.func,
}


export const LaunchTerminal = ({ whichNavParm, terminalDropdownOpen, handleTerminalDropdownClick, locale }) =>
  <li id='acm-kui-dropdown' className={`header-action${terminalDropdownOpen ? ' is-open' : ''}`}>
    <div className={`dropdown-container dropdown-container-apps${terminalDropdownOpen ? ' is-open' : ''}`}>
      <div
        role='button'
        aria-label='terminal-dropdown'
        tabIndex='0'
        className='header-action-trigger'
        onClick={handleTerminalDropdownClick}
        onKeyPress={handleTerminalDropdownClick}
      >
        <TerminalIcon title={msgs.get('nav.kui', locale)} />
      </div>
      <div className='multicloud-tooltip'>{msgs.get('nav.kui', locale)}</div>
      <ul className='dropdown-content' id='terminal-dropdown-content'>
        <span className='dropdown-content-header'>{msgs.get('nav.kui', locale)}</span>
        <span className='tech-preview'>{msgs.get('header.tech.preview')}</span>
        <li key={'Open in current browser tab'} id='current-tab-terminal' className='dropdown-menu-item'>
          <a className='dropwdown-content-items' href={`/kui${whichNavParm}`} >
            <CurrentBrowserTabIcon />
            <span className='item-text'>{msgs.get('nav.kui.current.tab', locale)}</span>
          </a>
        </li>
        <li key={'Open in new browser tab'} id='new-tab-terminal' className='dropdown-menu-item'>
          <a
            className='dropwdown-content-items'
            href={`/kui${whichNavParm}`}
            target={' _blank'} rel={relStr}
            onClick={handleTerminalDropdownClick}
            onKeyPress={handleTerminalDropdownClick}
          >
            <NewBrowserTabIcon />
            <span className='item-text'>{msgs.get('nav.kui.new.tab', locale)}</span>
          </a>
        </li>
      </ul>
    </div>
  </li>

LaunchTerminal.propTypes = {
  handleTerminalDropdownClick: PropTypes.func,
  locale: PropTypes.string,
  terminalDropdownOpen: PropTypes.bool,
  whichNavParm: PropTypes.string,
}


export const MinimizedLaunchTerminal = ({ whichNavParm, locale }) =>
  <li id='launch-terminal' className='dropdown-menu-item' >
    <a href={`/kui${whichNavParm}`} className='dropwdown-content-items'>
      <TerminalIcon title={msgs.get('nav.kui', locale)} darkFill={true} />
      <span>{msgs.get('nav.kui', locale)}</span>
      <span className='tech-preview'>{msgs.get('header.tech.preview')}</span>
    </a>
  </li>

MinimizedLaunchTerminal.propTypes = {
  locale: PropTypes.string,
  whichNavParm: PropTypes.string,
}

export const DocLink = ({ docLink }, { locale }) =>
  <li id='acm-doc' className='dropdown-menu-item'>
    <a className='dropwdown-content-items' href={docLink} rel="noopener noreferrer" target='_blank' >
      <DocIcon />
      <span>{msgs.get('nav.docs', locale)}</span>
    </a>
  </li>

DocLink.propTypes = {
  docLink: PropTypes.string,
}

// Disabled until the appropriate link is available
// export const SupportLink = ({ support }, { locale }) =>
//   <li id='icp-support' className='dropdown-menu-item'>
//     <a href={support ? support : 'http://ibm.biz/icpsupport'} target='support' aria-describedby='launchWindow'>
//       <SupportIcon />
//       <span>{msgs.get('nav.support', locale)}</span>
//     </a>
//   </li>

export const AboutButton = ({ handleAboutWindow }, { locale }) =>
  <li id='acm-about' className='dropdown-menu-item'>
    <a className='dropwdown-content-items' href='#' onClick={handleAboutWindow}>
      <AboutIcon />
      <span>{msgs.get('dropdown.user.about', locale)}</span>
    </a>
  </li>

AboutButton.propTypes = {
  handleAboutWindow: PropTypes.func,
}

export const ConfigureClient = ({ oauthInfo, locale }) =>
  <li id='configure-client' className='dropdown-menu-item'>
    {/** launch to OCPs token page */}
    <a className='dropwdown-content-items' href={`${oauthInfo.token_endpoint}/request`} target={'_blank'} rel={'noopener noreferrer'}>
      <ConfigureClientIcon />
      <span>{msgs.get('dropdown.user.configure', locale)}</span>
      <span className='launch-icon'>
        <LaunchIcon idPostfix='configure-client' />
      </span>
    </a>
  </li>

ConfigureClient.propTypes = {
  locale: PropTypes.string,
  oauthInfo: PropTypes.object,
}

export const Logout = ({ handleLogout }, { locale }) =>
  <li id='acm-logout' className='dropdown-menu-item'>
    <a className='dropwdown-content-items' href='#' id='logout' onClick={handleLogout}>
      <LogoutIcon />
      <span>{msgs.get('dropdown.user.logout', locale)}</span>
    </a>
  </li>

Logout.propTypes = {
  handleLogout: PropTypes.func,
}

export const AdminDropdown = ({ userDropdownOpen, user, handleUserDropdownClick, handleLogout, oauthInfo }, { locale }) =>
  <li id='acm-user-dropdown' className={`header-action${userDropdownOpen ? ' is-open' : ''}`}>
    <div className={'dropdown-container dropdown-container-admin ' + (userDropdownOpen ? 'is-open' : '')}>
      <div
        role='button'
        aria-label='acm-user-dropdown'
        tabIndex='0'
        className='header-action-trigger'
        onClick={handleUserDropdownClick}
        onKeyPress={handleUserDropdownClick}>
        <div className='header-user-info-dropdown'>
          {user.username || ''}
        </div>
        <svg className='header-user-info-dropdown_icon' viewBox='0 0 32 32' role='img' tabIndex='0' title='dropdown-icon'>
          <path d="M23 12L16 20 9 12 23 12z" />
        </svg>
      </div>
      <div className='multicloud-tooltip'>{msgs.get('nav.user', locale)}</div>
      <ul className='dropdown-content' id='acm-user-dropdown-content'>
        <ConfigureClient oauthInfo={oauthInfo} />
        <Logout handleLogout={handleLogout} />
      </ul>
    </div>
  </li>

AdminDropdown.propTypes = {
  handleLogout: PropTypes.func,
  handleUserDropdownClick: PropTypes.func,
  oauthInfo: PropTypes.object,
  user: PropTypes.object,
  userDropdownOpen: PropTypes.bool,
}

export const InfoDropdown = ({ infoDropdownOpen, handleInfoDropdownClick, handleAboutWindow, docLink /*, support*/}, { locale }) =>
  <li id='acm-info-dropdown' className={`header-action${infoDropdownOpen ? ' is-open' : ''}`}>
    <div className={`dropdown-container dropdown-container-info${infoDropdownOpen ? ' is-open' : ''}`}>
      <div
        role='button'
        aria-label='info-dropdown'
        tabIndex='0'
        className='header-action-trigger'
        onClick={handleInfoDropdownClick}
        onKeyPress={handleInfoDropdownClick}>
        <InfoDropdownIcon handleInfoDropdownClick={handleInfoDropdownClick} />
      </div>
      <div className='multicloud-tooltip'>{msgs.get('nav.info', locale)}</div>
      <ul className='dropdown-content' id='info-dropdown-content'>
        {docLink && <DocLink docLink={docLink} />}
        {/* {support && <SupportLink support={support} />} */}
        <AboutButton handleAboutWindow={handleAboutWindow} />
      </ul>
    </div>
  </li>

InfoDropdown.propTypes = {
  docLink: PropTypes.string,
  handleAboutWindow: PropTypes.func,
  handleInfoDropdownClick: PropTypes.func,
  infoDropdownOpen: PropTypes.bool,
}

function formatApps(apps) {
  return (
    apps && apps.map(app => {
      return (
        <li key={app.name} id='applications' className='dropdown-menu-item'>
          <a className='dropwdown-content-items' href={app.url} target={' _blank'} rel={relStr}>
            <ApplicationIcon title={app.name + ' icon'} source={app.icon || ''} />
            <span className='item-text'>{app.name}</span>
            <span className='launch-icon'>
              <LaunchIcon idPostfix={app.name} />
            </span>
          </a>
        </li>
      )
    })
  )}

export const ApplicationLauncher = ({ appsDropdownOpen, apps, handleAppsDropdownClick }, { locale }) =>
  <li id='acm-apps-dropdown' className={`header-action${appsDropdownOpen ? ' is-open' : ''}`}>
    <div className={`dropdown-container dropdown-container-apps${appsDropdownOpen ? ' is-open' : ''}`}>
      <div
        role='button'
        aria-label='apps-dropdown'
        tabIndex='0'
        className='header-action-trigger'
        onClick={handleAppsDropdownClick}
        onKeyPress={handleAppsDropdownClick}>
        <AppSwitcherIcon handleAppsDropdownClick={handleAppsDropdownClick} />
      </div>
      <div className='multicloud-tooltip'>{msgs.get('nav.apps', locale)}</div>
      <ul className='dropdown-content' id='apps-dropdown-content'>
        <span className='dropdown-content-header'>{'Red Hat Applications'}</span>
        {formatApps(apps)}
      </ul>
    </div>
  </li>

ApplicationLauncher.propTypes = {
  apps: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  appsDropdownOpen: PropTypes.bool,
  handleAppsDropdownClick: PropTypes.func,
}

export const MinimizedHeaderDropdown = ({
  minimizedDropdownOpen,
  handleMinimizedDropdownClick,
  handleAboutWindow,
  oauthInfo,
  handleLogout,
  handleResourceModalOpen,
  apps,
  user,
  hasMCMSearch,
  hasKUI,
  whichNavParm,
  disabledItems,
  docLink
}, { locale }) =>
  <li id='acm-minimized-dropdown' className={`header-action${minimizedDropdownOpen ? ' is-open' : ''}`}>
    <div className={`dropdown-container dropdown-container-apps${minimizedDropdownOpen ? ' is-open' : ''}`}>
      <div
        role='button'
        aria-label='apps-dropdown'
        tabIndex='0'
        className='header-action-trigger'
        onClick={handleMinimizedDropdownClick}
        onKeyPress={handleMinimizedDropdownClick}>
        <OverflowDotsIcon title={'menuIcon'} />
      </div>
      <div className='multicloud-tooltip'>{msgs.get('nav.minimized.items', locale)}</div>
      <ul className='dropdown-content' id='header-dropdown-content'>
        <span className='dropdown-content-header'>{'Red Hat Applications'}</span>
        {formatApps(apps)}
        <div className='section-divider' />
        {hasMCMSearch && <MinimizedSearchLink />}
        {!(disabledItems && disabledItems.includes('createResource')) &&
          <MinimizedCreateResources handleResourceModalOpen={handleResourceModalOpen} />
        }
        {hasKUI && <MinimizedLaunchTerminal whichNavParm={whichNavParm} />}
        <AboutButton handleAboutWindow={handleAboutWindow} />
        {docLink && <DocLink docLink={docLink} />}
        {/* {support && <SupportLink support={support} />} */}
        <div className='section-divider' />
        <span className='dropdown-content-header'>{user.username || ''}</span>
        <ConfigureClient oauthInfo={oauthInfo} />
        <Logout handleLogout={handleLogout} />
      </ul>
    </div>
  </li>

MinimizedHeaderDropdown.propTypes = {
  apps: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  disabledItems: PropTypes.object,
  docLink: PropTypes.string,
  handleAboutWindow: PropTypes.func,
  handleLogout: PropTypes.func,
  handleMinimizedDropdownClick: PropTypes.func,
  handleResourceModalOpen: PropTypes.func,
  hasKUI: PropTypes.bool,
  hasMCMSearch: PropTypes.bool,
  minimizedDropdownOpen: PropTypes.bool,
  oauthInfo: PropTypes.object,
  user: PropTypes.object,
  whichNavParm: PropTypes.string,
}
