/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import msgs from '../../nls/platform-header.properties'
import resources from '../../lib/shared/resources'
import _ from 'lodash'
import PropTypes from 'prop-types'
import { CSSTransition } from 'react-transition-group'
import handleKeyboardEvent from '../util/accessibility'
import AdaptiveLink from '../components/common/AdaptiveLink'
import { connect } from 'react-redux'
import { withRouter } from 'react-router-dom'

resources(() => {
  require('../../scss/left-nav.scss')
})

class LeftNav extends React.Component {
  constructor(props) {
    super(props)
    this.renderRoutes = this.renderRoutes.bind(this)
    this.renderSubRoutes = this.renderSubRoutes.bind(this)
    this.handleKeyDown = this.handleKeyDown.bind(this)
    this.handleSecondaryNavClose = this.handleSecondaryNavClose.bind(this)
  }

  render() {
    const { open, hoveredItem } = this.props
    return (
      <div className='left-nav-wrapper'>
        <CSSTransition
          classNames='transition'
          in={open}
          timeout={300}
          mountOnEnter={true}
          unmountOnExit={true}
          onEntered={() => this.leftNav.focus()}
          onExited={() => this.props.headerWrapper.focus()}
        >
          {/* eslint-disable-next-line jsx-a11y/no-noninteractive-element-interactions */}
          <nav
            ref={nav => this.leftNav = nav}
            id='left-nav'
            className='left-nav'
            aria-label={msgs.get('header.menu.bar.label', this.context.locale)}
            tabIndex='-1'
            onKeyDown={this.handleKeyDown}
          >
            <ul role='menubar'>
              {this.renderRoutes()}
            </ul>
          </nav>
        </CSSTransition>
        <CSSTransition classNames='transition' in={!!hoveredItem} timeout={300} mountOnEnter={true} unmountOnExit={true}>
          {/* eslint-disable-next-line jsx-a11y/no-noninteractive-element-interactions */}
          <nav
            ref={nav => this.secondaryNav = nav}
            id='secondary-nav'
            className='secondary-nav'
            aria-label={msgs.get('header.menu.bar.label', this.context.locale)}
            tabIndex='-1'
            onMouseLeave={this.handleSecondaryNavClose}
            onKeyDown={this.handleKeyDown}
          >
            {hoveredItem && hoveredItem.subItems && hoveredItem.subItems.length > 0 &&
              <ul role='menubar' className='left-nav-subitem-list'>
                <li className='secondary-nav-header' label={hoveredItem.label} title={hoveredItem.label}>
                  {/* <div
                  className='secondary-nav-header-icon'>{hoveredItem.iconUrl && <img alt={`${hoveredItem.label} icon`}
                  src={hoveredItem.iconUrl} />}</div> */}
                  {hoveredItem.label}
                </li>
                {this.renderSubRoutes()}
              </ul>
            }
          </nav>
        </CSSTransition>
      </div>
    )
  }

  renderRoutes() {
    const { navRoutes, secondaryNavRoutes, whichNav, hoveredItem, selectedItem } = this.props
    const { locale } = this.context
    let currentRoutes = navRoutes
    if(whichNav){
      //this is extra error handling in case the user enters the wrong name in useNav
      const routesTemp = _.find(secondaryNavRoutes, (item) => item.id === whichNav)
      currentRoutes = routesTemp ? routesTemp.navItems : navRoutes
    }
    const routes = currentRoutes.map(route => {
      const selected = selectedItem && selectedItem.id === route.id
      if (!route.disabled && route.subItems && !route.subItems.find(subroute => !subroute.disabled)) {
        return undefined
      }
      if (!route.disabled && route.subItems) {
        const open = hoveredItem && hoveredItem.id === route.id
        return (
          <li
            key={`${route.id}-${route.index}`}
            aria-label={route.label}
            id={route.id}
            className={(open ? 'open' : '') + (selected ? ' selected' : '')}
          >
            {/* <div
            className='left-nav-item-icon'>{route.iconUrl && <img alt={`${route.label} icon`}
            src={route.iconUrl} />}</div> */}
            {/* eslint-disable-next-line jsx-a11y/click-events-have-key-events */}
            <div
              className={'left-nav-item primary-nav-item ' + (open ? 'open' : '')}
              label={route.label}
              onMouseOver={this.handleMouseOver.bind(this, route)}
              onMouseOut={this.handleMouseOut.bind(this, route)}
              onFocus={this.handleMouseOver.bind(this, route)}
              onBlur={this.handleMouseOut.bind(this, route)}
              title={route.label}
              role='menuitem'
              tabIndex='0'>
              {route.label}
            </div>
            <div className='left-nav-item-chevron'>
              <svg viewBox='0 0 32 32' role='img' title={msgs.get('svg.description.menu', locale)} tabIndex='0'>
                <polygon points="22,16 12,26 10.6,24.6 19.2,16 10.6,7.4 12,6" />
                <rect width="32" height="32" fill='none' />
              </svg>
            </div>
          </li>
        )
      } else {
        return (
          !route.disabled && <li
            className={'left-nav-item primary-nav-item' + (selected ? ' selected' : '')}
            title={route.label}
            key={`${route.id}-${route.index}`}>
            {/* <div
            className='left-nav-item-icon'>{route.iconUrl && <img alt={`${route.label} icon`}
            src={route.iconUrl} />}</div> */}
            <AdaptiveLink target={route.target} id={route.id} external={route.external} to={route.url} label={route.label} />
          </li>
        )
      }
    })
    return routes.filter(route => route)
  }

  renderSubRoutes() {
    return this.props.hoveredItem.subItems.map(subRoute => (
      !subRoute.disabled && <li
        className='secondary-nav-item'
        title={subRoute.label}
        key={`${subRoute.id}-${subRoute.index}`}>
        <AdaptiveLink
          target={subRoute.target}
          external={subRoute.external}
          key={`${subRoute.id}-${subRoute.index}`}
          to={subRoute.url}
          label={subRoute.label}
          id={subRoute.id}>
        </AdaptiveLink>
      </li>
    )).filter(route => route)
  }

  handleKeyDown(e) {
    const primary = Array.from(this.leftNav.querySelectorAll('div.primary-nav-item, .primary-nav-item a')),
          secondary = Array.from(this.secondaryNav ? this.secondaryNav.querySelectorAll('ul a') : []),
          opened = _.findIndex(primary, item => item.className.includes('open'))

    const data = { primary, secondary, opened, handleMenuClick: this.props.handleMenuClick,
      leftNav: this.leftNav, secondaryNav: this.secondaryNav }
    handleKeyboardEvent(e, data, this.leftNav.contains(e.target) ? 'left-nav' : 'secondary-nav')
  }

  handleMouseOver(item) {
    const { hoveredItem, handleNavItemHover } = this.props
    if (!hoveredItem || hoveredItem.id !== item.id) {
      handleNavItemHover(item)
    }
  }

  handleMouseOut(item, event) {
    const { hoveredItem } = this.props
    if (hoveredItem && hoveredItem.id === item.id && this.secondaryNav && !this.secondaryNav.contains(event.relatedTarget)) {
      this.handleSecondaryNavClose(event)
    }
  }

  handleSecondaryNavClose(event) {
    const { handleNavItemHover} = this.props
    if (!event.relatedTarget || !event.relatedTarget.classList || !event.relatedTarget.classList.contains('primary-nav-item')) {
      handleNavItemHover(null)
    }
  }
}

LeftNav.contextTypes = {
  locale: PropTypes.string
}

LeftNav.propTypes = {
  handleMenuClick: PropTypes.func,
  handleNavItemHover: PropTypes.func,
  headerWrapper: PropTypes.object,
  hoveredItem: PropTypes.object,
  navRoutes: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  open: PropTypes.bool,
  secondaryNavRoutes: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  selectedItem: PropTypes.object,
  whichNav: PropTypes.string,
}

const mapStateToProps = (state) => {
  return {
    user: state.user
  }
}

export default withRouter(connect(mapStateToProps)(LeftNav))
