/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc.*/
'use strict'

import React from 'react'
import { ComposedModal, ModalHeader, ModalBody, Loading } from 'carbon-components-react'
import config from '../../../lib/shared/config'
import msgs from '../../../nls/platform-header.properties'
import resources from '../../../lib/shared/resources'
import { updateModal } from '../../actions/common'
import { connect } from 'react-redux'
import PropTypes from 'prop-types'
import ACMVersionClient from '../../../lib/client/acm-version'
import _ from 'lodash'
import _uniqueId from 'lodash/uniqueId'

resources(() => {
  require('../../../scss/about.scss')
})

class AboutModal extends React.PureComponent {
  static defaultProps = {
    useRemoteVersion: true
  }

  constructor(props) {
    super(props)
  }
  state = { mounted: false, version: undefined, showDetails: undefined }

  componentDidMount() {
    this.setState({ mounted: true, showDetails: true }) //initialize with copyright open
    ACMVersionClient.getACMVersion(
      (res) => {
        this.setState({ remoteVersion: _.get(res, 'status.currentVersion', msgs.get('unavailable', this.context.locale)) })
      },
      () => {
        this.setState({ remoteVersion: msgs.get('unavailable', this.context.locale) })
      }
    )
    setTimeout(() => { // cannot set aria-label prop to X button, fails to find selector without this setTimeout
      const closeBtn = document && document.querySelector('.close-about')
      closeBtn && closeBtn.setAttribute('aria-label', msgs.get('modal.button.close', this.context.locale))
    }, 500)
  }
  render() {
    const { open, locale, handleClose, aboutVersion, copyright, useRemoteVersion } = this.props
    const { mounted, remoteVersion } = this.state
    const fullVersion = `${useRemoteVersion ? remoteVersion : aboutVersion}`
    const headerCP = config.headerContextPath ? config.headerContextPath : '/multicloud/header'
    return mounted &&
      <ComposedModal
        id='acm-about-dialog'
        className='acm-about-dialog'
        aria-label={msgs.get('dropdown.user.about', locale)}
        open={open}
        onClose={handleClose}>
        <ModalHeader
          aria-label={msgs.get('modal.about.header', locale)}
          role='region'
          closeClassName='close-about'
          iconDescription={msgs.get('modal.button.close', locale)} />
        <ModalBody role='region' aria-label={msgs.get('modal.about.body', locale)}>
          <div className='about-wrapper'>
            <div className='version-details'>
              <span className='version-details__label'>{msgs.get('product.version.text', locale)}</span>
              <span className='version-details__no'>
                { fullVersion === 'undefined' ? <Loading small withOverlay={false} /> : fullVersion }
              </span>
            </div>
            <img
              className='acm-logo'
              src={`${headerCP}/graphics/RHACM-Logo.svg`}
              alt={msgs.get('about.modal.text.logo', locale)}
            />
            <span className='spacer' />
            <div className='copyright'>
              {copyright.split('\n').map((cp) => {
                return <p key={_uniqueId('key')}>{cp}</p>
              })}
            </div>
          </div>
          <img
            className='about-img'
            src={`${headerCP}/graphics/About-Modal-Logo.svg`}
            alt={msgs.get('about.modal.text.logo', locale)}
          />
        </ModalBody>
      </ComposedModal>
  }

  toggleCopyright(event) {
    event.preventDefault()
    this.setState(prevState => ({ showDetails: !prevState.showDetails }))
  }
}

AboutModal.contextTypes = {
  locale: PropTypes.string
}

AboutModal.propTypes = {
  aboutVersion: PropTypes.string,
  copyright: PropTypes.string,
  handleClose: PropTypes.func,
  locale: PropTypes.string,
  open: PropTypes.bool,
  useRemoteVersion: PropTypes.string,
}

const mapStateToProps = state => {
  return{
    open: state.modal.open,
    licenses: _.get(state, 'uiconfig.config.about.licenses', []),
    copyright: _.get(state, 'uiconfig.config.about.copyright', ''),
    edition: _.get(state, 'uiconfig.config.about.edition'),
    aboutVersion: _.get(state, 'uiconfig.config.about.version', ''),
    whichNav: _.get(state, 'experience.whichNav', ''),
    allHeaderConfig: _.get(state, 'nav.secondaryNavItems', [])
  }
}

const mapDispatchToProps = dispatch => {
  return {
    handleClose: () => dispatch(updateModal({open: false, type: ''}))
  }
}

export default connect(mapStateToProps, mapDispatchToProps)(AboutModal)
