/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */
'use strict'

import React from 'react'
import _ from 'lodash'
import {
  Loading,
  InlineNotification,
  Dropdown,
  MultiSelect,
  ComposedModal,
  ModalHeader,
  ModalBody,
  ModalFooter
} from 'carbon-components-react'
import msgs from '../../../nls/platform-header.properties'
import resources from '../../../lib/shared/resources'
import {
  createResource,
  clearRequestStatus,
  editResource,
  receivePostError,
  updateModal
} from '../../actions/common'
import { connect } from 'react-redux'
import { REQUEST_STATUS } from '../../actions/index'
import jsYaml from 'js-yaml'
import PlatformApiClient from '../../../lib/client/platform-api-client'
import KubernetesClient from '../../../lib/client/k8s-client'
import PropTypes from 'prop-types'
import _uniqueId from 'lodash/uniqueId'
import loadable from '@loadable/component'

const CONSTANTS = require('../../../lib/shared/constants')
const modelCreateLocalClusterString= 'modal.create-resource.local-cluster'

// can't be hydrated--needs babel-loader to import monaco
export const YamlEditor = loadable(() => import(/* webpackChunkName: "YamlEditor" */ '../common/YamlEditor'))

resources(() => {
  require('../../../scss/create-resource.scss')
})

class ResourceModal extends React.PureComponent {

  constructor(props){
    super(props)
    const successCb = result => {
      const goodClusters = result.filter(item =>
        _.get(item, 'status.conditions', []).find(condition => {
          return condition.type === 'ManagedClusterConditionAvailable' && condition.status === 'True'
        }))
      const clusters = goodClusters.map(clusterInfo => {
        return {
          clusterName: clusterInfo.metadata.name,
          clusterNamespace: clusterInfo.metadata.name,
        }
      })
      const localCluster = 'local-cluster'
      if (clusters && !clusters.find(c => c.clusterName === localCluster)) {
        clusters.push({
          clusterName: localCluster,
          clusterNamespace: localCluster,
        })
      }
      clusters.length > 0 && this.setState({ clusterList: clusters })
      this.setState({ requestStatus: '' })
    }
    const errorCb = err => {
      const localCluster = 'local-cluster'
      this.showError(`${this.getLocaleMsg('modal.create-resource.managed.clusters.request.error')} ${err && err.error ? err.error : ''}`)
      this.setState({
        requestStatus: '',
        clusterList: [{
          clusterName: localCluster,
          clusterNamespace: localCluster,
        }]
      })
    }
    PlatformApiClient.getManagedClusters(successCb, errorCb)
  }

  state = {
    reqErrorMsg: [],
    editorMode: this.props.editorMode,
    selectedClusters: [],
    resourceReqCount: 0,
    requestStatus: '',
    localhostIsManaged: false
  }

  modes = {
    yaml: 'yaml',
    json: 'json'
  }

  flattenMsg = msg => {
    return _.uniqWith(_.flattenDeep(msg), _.isEqual)
  }

  showError = msg => {
    if (Array.isArray(msg)){
      this.setState(prevState => ({ reqErrorMsg: this.flattenMsg([prevState.reqErrorMsg, msg]) }))
    } else {
      this.setState(prevState => ({ reqErrorMsg: this.flattenMsg([prevState.reqErrorMsg, [msg]]) }))
    }
  }

  getLocaleMsg = (key, arr = []) =>
    arr.length === 0
      ? msgs.get(key, this.context.locale)
      : msgs.get(key, arr, this.context.locale)

  addResourceReqCount = () => this.setState(prevState => ({ resourceReqCount: prevState.resourceReqCount + 1}))

  subtractResourceReqCount = () => this.setState(prevState => ({ resourceReqCount: prevState.resourceReqCount - 1 }))

  updateResourceReqCount = status => {
    this.subtractResourceReqCount()
    this.state.resourceReqCount <= 0 && this.setState({ requestStatus: status })
  }

  validateResourceConfig = resource => resource.kind && resource.metadata && resource.metadata.name

  actionStatusHeartBeat(actionName, clusterNamespace) {
    const { heartBeat } = this.state
    const successCb = res => {
      const action = res.items.find(actionResource => actionResource.metadata.name === actionName)
      const completed = _.get(action, 'status.conditions[0].type') === CONSTANTS.ACTION_STATUS.COMPLETED
      const actionSuccess = _.get(action, 'status.conditions[0].status') === 'True'
      if (completed && actionSuccess) {
        this.updateResourceReqCount(CONSTANTS.ACTION_STATUS.COMPLETED)
        clearInterval(heartBeat)
        this.state.reqErrorMsg.length === 0 && this.state.resourceReqCount <= 0 && this.state.requestStatus === CONSTANTS.ACTION_STATUS.COMPLETED && this.handleClose()
      } else if (completed && !actionSuccess) {
        this.updateResourceReqCount(CONSTANTS.ACTION_STATUS.FAILED)
        this.showError(`${actionName} on ${_.get(action, 'metadata.namespace', '-')} : ${_.get(action, 'status.conditions[0].message', '-')}`)
        clearInterval(heartBeat)
      }
    }
    const errorCb = err => {
      console.error(err) //eslint-disable-line no-console
      this.updateResourceReqCount(CONSTANTS.ACTION_STATUS.FAILED)
      this.showError(this.getLocaleMsg('modal.create-resource.check.request.error'))
    }
    KubernetesClient.getActions(clusterNamespace, successCb, errorCb)
  }

  createResourceOnSelectedClusters = (resource, successCb, errorCb) => {
    const { selectedClusters, localhostIsManaged } = this.state
    selectedClusters.forEach(cluster => {
      if ( !localhostIsManaged && cluster.clusterName === this.getLocaleMsg(modelCreateLocalClusterString)){
        this.createResourceLocally()
      } else {
        const now = new Date()
        const body = {
          apiVersion: 'action.open-cluster-management.io/v1beta1',
          kind: 'ManagedClusterAction',
          metadata: {
            name: `${resource.metadata.name}-${now.valueOf()}`,
            namespace: cluster.clusterNameSpace,
          },
          spec: {
            actionType: 'Create',
            kube: {
              resource: resource.kind.toLowerCase(),
              namespace: resource.metadata.namespace,
              template: resource,
            },
          },
        }
        this.setState({ requestStatus: CONSTANTS.ACTION_STATUS.ACTIVE })
        this.addResourceReqCount()
        KubernetesClient.createAction(cluster.clusterNamespace, body, successCb, errorCb)
      }
    })
  }

  createResourceLocally = () => {
    const { resourceType, receivePostError:receivePostErrorLocal, createResource:createResourceLocal, user } = this.props
    const { data, reqErrorMsg } = this.state
    let resourcesLocal, namespace
    try {
      resourcesLocal = _.compact(jsYaml.safeLoadAll(data))
      resourcesLocal.forEach(resource => {
        if (resource.metadata && resource.metadata.namespace) {
          namespace = resource.metadata.namespace
        }

        if (resource.kind === CONSTANTS.RESOURCE_TYPES.NAMESPACE.name) {
          createResourceLocal(resourceType, namespace, resource, user)
        } else {
          createResourceLocal(resourceType, namespace, resource)
        }
      })
    } catch(e) {
      receivePostErrorLocal(resourceType, {error: { message: e.message }})
      this.setState({reqErrorMsg: this.flattenMsg([...reqErrorMsg, e.message])})
    }
  }

  handleSubmit = () => {
    this.props.clearRequestStatus()
    this.setState({ reqErrorMsg: [] }, () => {
      const { selectedClusters, clusterList } = this.state

      if(clusterList && selectedClusters && selectedClusters.length > 0) {
        try {
          const rawData = jsYaml.safeLoadAll(this.state.data)
          const data = _.compact(rawData)
          const resource = data[0]
          const successCb = res => {
            const heartBeat = setInterval(this.actionStatusHeartBeat.bind(this, res.metadata.name, res.metadata.namespace), 2000)
            this.setState({ heartBeat })
          }
          const errorCb = err => {
            this.showError(this.getLocaleMsg('modal.create-resource.submit.request.error'))
            this.showError(err.error && err.error.message)
            this.updateResourceReqCount(CONSTANTS.ACTION_STATUS.FAILED)
          }
          this.validateResourceConfig(resource)
            ? this.createResourceOnSelectedClusters(resource, successCb, errorCb)
            : this.showError(this.getLocaleMsg('modal.create-resource.validate.failed'))
        } catch (e) {
          this.showError(this.getLocaleMsg('modal.create-resource.yaml.failed'))
        }
      }
      else {
        this.createResourceLocally()
      }
    })
  }

  handleClose = () => {
    this.setState({reqErrorMsg: []})
    this.setState({ currPath: window.location.pathname })
    this.props.handleClose()
  }

  escapeEditor = e => {
    e.persist()
    const button = document.querySelector('.bx--btn--secondary')
    e.shiftKey && e.ctrlKey && e.which === 81 && button.focus()
  }

  onChange = value => {
    this.setState({data: value})
  }

  UNSAFE_componentWillReceiveProps(nextProps) {
    if (nextProps.data && this.props.data !== nextProps.data) {
      this.changeDataFormat(nextProps.editorMode, nextProps.data)
    }
    if (nextProps.reqStatus && nextProps.reqStatus === REQUEST_STATUS.ERROR) {
      this.setState(prevState => ({ reqErrorMsg: this.flattenMsg([prevState.reqErrorMsg, nextProps.reqErrorMsg]) }))
    }
    if (nextProps.reqCount === 0 && !nextProps.reqErrCount) {
      this.handleClose()
    }
  }

  componentDidMount() {
    this.resourceModal.focus()
    this.changeDataFormat(this.props.editorMode, this.props.data)
    this.setState({ currPath: window.location.pathname })
    this.checkIfPathChanged()
    window.addEventListener('resize',  this.layoutEditors.bind(this))
  }

  setContainerRef = container => {
    this.containerRef = container
    this.layoutEditors()
  }

  setEditor = (editor) => {
    this.editor=editor
    this.layoutEditors()
  }

  layoutEditors() {
    if (this.containerRef && this.editor) {
      const rect = this.containerRef.getBoundingClientRect()
      const width = rect.width
      const height = rect.height
      this.editor.layout({width, height})
    }
  }

  handleNotificationClosed = () => this.setState({reqErrorMsg: []})

  checkIfPathChanged() {
    const closeIfPathChanged = () => {
      if (this.state.currPath !== window.location.pathname) {
        this.handleClose()
        clearInterval(this.state.heartBeatForModal)
      }
    }
    const heartBeatForModal = setInterval(closeIfPathChanged, 1000)
    this.setState({ heartBeatForModal })
  }

  handleChangeSelectedClusters = e => {
    this.setState({selectedClusters: e.selectedItems})
  }

  handleChangeConfigMode = e => {
    const mode = e.selectedItem || this.modes.yaml
    try {
      this.changeDataFormat(mode, _.compact(jsYaml.safeLoadAll(this.state.data))[0])
      this.setState({
        editorMode: mode
      })
    } catch(e) {
      this.setState(prevState => ({ reqErrorMsg: [prevState.reqErrorMsg, e.message] }))
    }

    setTimeout(() => { // cannot set aria-label prop to X button or editor, fails to find selector without this setTimeout
      const closeBtn = document && document.querySelector('#resource-modal-container .bx--modal-close')
      closeBtn && closeBtn.setAttribute('aria-label', msgs.get('modal.button.close', this.context.locale))

      const editor = document && document.querySelector('#resource-modal-editor .ace_text-input')
      editor && editor.setAttribute('aria-label', msgs.get('a11y.editor.label', this.context.locale))
    }, 500)
  }

  changeDataFormat = (mode, data) => {
    this.setState({
      data: mode === this.modes.json ? JSON.stringify(data, null, 2) : jsYaml.dump(data)
    })
  }

  render() {
    const { reqCount, open, label, locale } = this.props
    const { editorMode, clusterList, reqErrorMsg, selectedClusters, requestStatus } = this.state
    const uniqueReqErrorMsg =  this.flattenMsg(reqErrorMsg)
    let clusterDropdownLabel = ''
    if (Array.isArray(selectedClusters) && selectedClusters.length > 0) {
      selectedClusters.forEach((selectedCluster, index) => {
        const selectedClusterName = selectedCluster.clusterName
          ? selectedCluster.clusterName
          : selectedCluster.clusterNameSpace
        clusterDropdownLabel = (index === 0)
          ? selectedClusterName
          : `${clusterDropdownLabel}, ${selectedClusterName}`
      })
    }
    else {
      clusterDropdownLabel = clusterList
        ? msgs.get('modal.create-resource.select.label', locale)
        : msgs.get('modal.create-resource.loading.label', locale)
    }
    return (
      // eslint-disable-next-line jsx-a11y/no-noninteractive-element-interactions
      <div id='resource-modal-container'
        ref={div => this.resourceModal = div}
        tabIndex='-1'
        role='region'
        onKeyDown={this.escapeEditor}
        aria-label={msgs.get('a11y.editor.escape', locale)}>
        {((reqCount && reqCount > 0) || (requestStatus === CONSTANTS.ACTION_STATUS.ACTIVE)) && <Loading withOverlay={true} />}
        <ComposedModal
          id='resource-modal'
          className='create-resource-modal'
          open={open}
          role='region'
          onClose={() => true}
          aria-label={msgs.get(label.heading, locale)}>
          <ModalHeader title={msgs.get(label.heading, locale)} />
          <ModalBody>
            {uniqueReqErrorMsg && uniqueReqErrorMsg.length > 0 && uniqueReqErrorMsg.map((err) =>
              err && err.length > 0 && <InlineNotification
                key={_uniqueId('key')}
                kind='error'
                title=''
                subtitle={err}
                onCloseButtonClick={this.handleNotificationClosed}
                iconDescription={msgs.get('svg.description.error', locale)} />)}
            <pre>
              <p className='bx--modal-content__text'>
                <strong>{msgs.get('modal.create-resource.select.title.main', locale)} | </strong>
                {msgs.get('modal.create-resource.select.title.description', locale)}
              </p>
            </pre>
            {clusterList && <MultiSelect
              light
              label={clusterDropdownLabel}
              items={clusterList || [{clusterName: 'Loading clusters'}]}
              itemToString={cluster => (cluster ? cluster.clusterName : '')}
              onChange={this.handleChangeSelectedClusters}
              id='create-resource-select'
            />}
            <pre>
              <p className='bx--modal-content__text'>
                <strong>{msgs.get('modal.create-resource.format.title.main', locale)} | </strong>
                {msgs.get('modal.create-resource.format.title.description', locale)}
              </p>
            </pre>
            <Dropdown
              id={'resource-modal-dropdown'}
              label={editorMode.toUpperCase()}
              items={_.values(this.modes)}
              itemToString={item => (item ? item.toUpperCase() : '')}
              onChange={this.handleChangeConfigMode}
            />
            <div className='yamlEditorContainerContainer' ref={this.setContainerRef} >
              <YamlEditor
                width={'50vw'}
                height={'40vh'}
                readOnly={false}
                mode={editorMode}
                handleParsingError={()=>{}}
                setEditor={this.setEditor}
                onYamlChange={this.onChange}
                yaml={this.state && this.state.data}
              />
            </div>
          </ModalBody>
          <ModalFooter
            primaryButtonText={msgs.get(label.primaryBtn, locale)}
            primaryButtonDisabled={clusterList && selectedClusters.length === 0}
            secondaryButtonText={msgs.get('modal.button.cancel', locale)}
            onRequestClose={this.handleClose}
            onRequestSubmit={this.handleSubmit} />
        </ComposedModal>
      </div>
    )
  }
}

const mapStateToProps = state => {
  return {
    ...state.modal,
    namespace: state.namespaces && state.namespaces.defaultNamespace,
    user: state.user
  }
}

const mapDispatchToProps = dispatch => {
  return {
    createResource: (resourceType, namespace, data, user) => {
      dispatch(createResource(resourceType, namespace, data, user))
    },
    putResource: (resourceType, namespace, data) => {
      dispatch(editResource(resourceType, namespace, data))
    },
    handleClose: () => {
      dispatch(clearRequestStatus())
      dispatch(updateModal({open: false, type: 'resource'}))
    },
    receivePostError: (resourceType, err) => {
      dispatch(receivePostError(err, resourceType))
    },
    clearRequestStatus: () => dispatch(clearRequestStatus())
  }
}

ResourceModal.propTypes = {
  clearRequestStatus: PropTypes.func,
  createResource: PropTypes.func,
  data: PropTypes.object,
  editorMode: PropTypes.string,
  handleClose: PropTypes.func,
  label: PropTypes.object,
  locale: PropTypes.string,
  open: PropTypes.bool,
  receivePostError: PropTypes.func,
  reqCount: PropTypes.number,
  reqErrCount: PropTypes.number,
  reqErrorMsg: PropTypes.oneOfType([PropTypes.string, PropTypes.object]),
  reqStatus: PropTypes.string,
  resourceType: PropTypes.object,
  user: PropTypes.object,
}

export default connect(mapStateToProps, mapDispatchToProps)(ResourceModal)
