/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2017, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************
/* Copyright (c) 2020 Red Hat, Inc. */

import React from 'react'
import ReactDOM from 'react-dom'
import PropTypes from 'prop-types'
import { connect } from 'react-redux'
import { withRouter } from 'react-router-dom'
import IdleTimer from 'react-idle-timer'
import Modal from '../components/common/Modal'
import resources from '../../lib/shared/resources'
import Header from '../components/Header'
import LeftNav from '../components/LeftNav'
import client from '../../lib/shared/client'
import { isAuthenticated } from '../reducers/user'
import { switchNamespace } from '../actions/namespaces'
import { getDocLink } from '../../lib/shared/doc-util'
import { updateNav } from '../actions/nav'
import { updateHeaderExperience } from '../actions/experience'
import { parseQueryParams } from '../util/common'
import _ from 'lodash'
import loginClient from '../../lib/client/login-client'
import config from '../../lib/shared/config'

resources(() => {
  require('../../scss/common.scss')
})

class HeaderContainer extends React.Component {

  constructor(props) {
    super(props)

    if (client && this.props.serviceId !== 'platform-ui') {
      document.addEventListener('namespace-change', e => this.props.switchNamespace(e.detail.selectedNamespaces), true)
    }
    this.idleTimer = null
    this.handleMenuClick = this.handleMenuClick.bind(this)
    this.handleNavItemHover = this.handleNavItemHover.bind(this)
    this.handleAppsDropdownClick = this.handleAppsDropdownClick.bind(this)
    this.handleTerminalDropdownClick = this.handleTerminalDropdownClick.bind(this)
    this.handleUserDropdownClick = this.handleUserDropdownClick.bind(this)
    this.handleInfoDropdownClick = this.handleInfoDropdownClick.bind(this)
    this.handleMinimizedDropdownClick = this.handleMinimizedDropdownClick.bind(this)
    this.handleMouseClick = this.handleMouseClick.bind(this)
    this.handleLogout = this.handleLogout.bind(this)
    this.handleHeaderExperience = this.handleHeaderExperience.bind(this)
    this.handleCreateResDropdownClick = this.handleCreateResDropdownClick.bind(this)

    this.state = {
      leftNavOpen: false,
      leftNavSelectedItem: this.getCurrentNavItem(),
      leftNavHoveredItem: null,
      appsDropdownOpen: false,
      terminalDropdownOpen: false,
      userDropdownOpen: false,
      infoDropdownOpen: false,
      minimizedDropdownOpen: false,
      docLink: undefined,
      hasMCMSearch: true,
      hasKUI: true
    }
  }

  componentDidMount() {
    this.handleHeaderExperience()
    this.setState({
      docLink: getDocLink(window.location && window.location.pathname, this.props.doc)
    })

    this.documentBody = document.querySelector('body')
  }

  componentDidUpdate(prevProps) {
    if (prevProps.location.pathname !== this.props.location.pathname) {
      this.setState({
        leftNavOpen: false,
        leftNavHoveredItem: null,
        leftNavSelectedItem: this.getCurrentNavItem()
      })
    }
  }

  handleChartsUpdateEvent(event) {
    const { updateNav:updateNavLocal, role, namespaces } = this.props
    updateNavLocal(event, { currentRole: role, namespaces })
  }

  handleHeaderExperience() { //which experience should we override
    const { updateHeaderExperience:updateHeaderExperienceLocal } = this.props
    if (window && window.location && window.location.search !== '') {
      const params = parseQueryParams(window.location.search)
      if (params.useNav) {
        return updateHeaderExperienceLocal(params.useNav)
      }
    }
    return undefined
  }

  render() {
    const { endpointAccess, history, navRoutes, locale, logo, support, clusterName, secondaryNavRoutes,
      whichNav, activeAccountId, docUrl, gettingStartedUrl, headerContext, oauthInfo } = this.props
    // log out after 2hrs if not configured
    const idleTime = _.get(config, 'idleTime', 7200000)
    return (
      <div>
        <div
          ref={ref => this.headerWrapper = ref}
          className={'icp-header-wrapper' + (this.state.leftNavOpen ? ' nav-open' : '')}
          id='header-container'
          tabIndex='-1'
        >
          {config.env === 'production' &&
          <IdleTimer
            ref={ref => { this.idleTimer = ref }}
            timeout={idleTime}
            onIdle={this.handleLogout}
            debounce={250}
          />}
          <Header handleMenuClick={this.handleMenuClick}
            handleLogout={this.handleLogout}
            leftNavOpen={this.state.leftNavOpen}
            appsDropdownOpen={this.state.appsDropdownOpen}
            terminalDropdownOpen={this.state.terminalDropdownOpen}
            userDropdownOpen={this.state.userDropdownOpen}
            infoDropdownOpen={this.state.infoDropdownOpen}
            minimizedDropdownOpen={this.state.minimizedDropdownOpen}
            docLink={docUrl || this.state.docLink}
            handleAppsDropdownClick={this.handleAppsDropdownClick}
            handleTerminalDropdownClick={this.handleTerminalDropdownClick.bind(this)}
            handleUserDropdownClick={this.handleUserDropdownClick}
            handleInfoDropdownClick={this.handleInfoDropdownClick}
            handleMinimizedDropdownClick={this.handleMinimizedDropdownClick}
            handleCreateResDropdownClick={this.handleCreateResDropdownClick}
            hasMCMSearch={this.state.hasMCMSearch}
            logo={logo}
            navRoutes={navRoutes}
            support={support}
            gettingStartedUrl={gettingStartedUrl}
            clusterName={clusterName}
            endpointAccess={endpointAccess}
            activeAccountId={activeAccountId}
            hasKUI={this.state.hasKUI}
            oauthInfo={oauthInfo} />
          <LeftNav location={history.location}
            open={this.state.leftNavOpen}
            selectedItem={this.state.leftNavSelectedItem}
            hoveredItem={this.state.leftNavHoveredItem}
            handleNavItemHover={this.handleNavItemHover}
            handleMenuClick={this.handleMenuClick}
            headerWrapper={this.headerWrapper}
            whichNav={whichNav}
            navRoutes={navRoutes}
            secondaryNavRoutes={secondaryNavRoutes} />
        </div>
        <div className='left-nav-overlay'></div>
        {this.documentBody ? ReactDOM.createPortal(<Modal locale={locale} />, this.documentBody) : null}
        <input type='hidden' id='multicloud-headerContext' value={headerContext} />
      </div>
    )
  }

  handleMouseClick(event) {
    if (this.headerWrapper && !this.headerWrapper.contains(event.target)) {
      this.setState({
        leftNavOpen: false,
        leftNavHoveredItem: null,
        appsDropdownOpen: false,
        terminalDropdownOpen: false,
        userDropdownOpen: false,
        infoDropdownOpen: false,
        minimizedDropdownOpen: false
      })
      document.removeEventListener('click', this.handleMouseClick)
    } else if (this.state.minimizedDropdownOpen) {
      this.setState({
        minimizedDropdownOpen: false
      })
      document.removeEventListener('click', this.handleMouseClick)
    }
  }

  getCurrentNavItem() {
    const { location, navRoutes } = this.props
    const currentNavItem = navRoutes.filter(route => {
      if(location && typeof location.pathname === 'string') {
        if (route.id !== 'dashboard' && location.pathname.indexOf(route.url) !== -1) {
          return route
        } else if (route.subItems) {
          const matchedSubRoute = route.subItems.filter(subRoute => location.pathname.indexOf(subRoute.url) !== -1)
          if (matchedSubRoute.length > 0) {
            return route
          }
        } else if (route.url === location.pathname) {
          return route
        }
      }
      return undefined
    })
    // We may have multiple matches in the left nav that prefix the current url.
    // The one one with the largest length is going to be the one with the highest specificity so we return that one
    if (currentNavItem && currentNavItem.length > 1) {
      return currentNavItem.reduce((acc, navItem) => {
        if (navItem.subItems) {
          const matchedSubRoute = navItem.subItems.reduce(
            (accTemp, subItem) => subItem.url.length > accTemp.url.length ? subItem : accTemp
          )
          if (matchedSubRoute) {
            return navItem
          }
        }
        return navItem.url.length > acc.url.length ? navItem : acc
      })
    }
    return (currentNavItem && currentNavItem.length) >= 1 ? currentNavItem[0] : null
  }

  handleMenuClick() {
    this.setState(prevState => ({
      leftNavOpen: !prevState.leftNavOpen,
      leftNavHoveredItem: null
    }), () => {
      if (this.state.leftNavOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleCreateResDropdownClick() {
    this.setState({
      appsDropdownOpen: false,
      infoDropdownOpen: false,
      minimizedDropdownOpen: false,
      terminalDropdownOpen: false,
      userDropdownOpen: false
    })
  }

  handleAppsDropdownClick() {
    this.setState(prevState => ({
      appsDropdownOpen: !prevState.appsDropdownOpen,
      infoDropdownOpen: false,
      minimizedDropdownOpen: false,
      terminalDropdownOpen: false,
      userDropdownOpen: false
    }), () => {
      if (this.state.appsDropdownOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleInfoDropdownClick() {
    this.setState(prevState => ({
      appsDropdownOpen: false,
      infoDropdownOpen: !prevState.infoDropdownOpen,
      minimizedDropdownOpen: false,
      terminalDropdownOpen: false,
      userDropdownOpen: false
    }), () => {
      if (this.state.infoDropdownOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleMinimizedDropdownClick() {
    this.setState(prevState => ({
      appsDropdownOpen: false,
      infoDropdownOpen: false,
      minimizedDropdownOpen: !prevState.minimizedDropdownOpen,
      terminalDropdownOpen: false,
      userDropdownOpen: false
    }), () => {
      if (this.state.minimizedDropdownOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleTerminalDropdownClick() {
    this.setState(prevState => ({
      appsDropdownOpen: false,
      infoDropdownOpen: false,
      minimizedDropdownOpen: false,
      terminalDropdownOpen: !prevState.terminalDropdownOpen,
      userDropdownOpen: false
    }), () => {
      if (this.state.terminalDropdownOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleUserDropdownClick() {
    this.setState(prevState => ({
      appsDropdownOpen: false,
      infoDropdownOpen: false,
      minimizedDropdownOpen: false,
      terminalDropdownOpen: false,
      userDropdownOpen: !prevState.userDropdownOpen
    }), () => {
      if (this.state.userDropdownOpen) {
        document.addEventListener('click', this.handleMouseClick)
      } else {
        document.removeEventListener('click', this.handleMouseClick)
      }
    })
  }

  handleNavItemHover(item) {
    this.setState({
      leftNavHoveredItem: item
    })
  }

  handleLogout = (event) => {
    if (event) {
      event.preventDefault()
    }
    const headerCP = config.headerContextPath ? config.headerContextPath : '/multicloud/header'
    const path = location.protocol + '//' + location.host + `${headerCP}/logout`
    loginClient.logout(path, (res) => {
      const onLogout = (delay=0) => {
        return setTimeout(() => {
          location.reload(true)
        }, delay)
      }
      if (res.admin) {
        const form = document.createElement('form')
        form.target = 'hidden-form'
        form.method = 'POST'
        form.action = res.logoutPath
        const iframe = document.createElement('iframe')
        iframe.style = 'display:none'
        iframe.name = 'hidden-form'
        iframe.onload = onLogout(500)
        document.body.appendChild(iframe)
        document.body.appendChild(form)
        form.submit()
      } else {
        onLogout()
      }
    }, () => {
      // eslint-disable-next-line no-console
      console.error('Error: invalid response returned from logout API')
    })
  }
}

HeaderContainer.propTypes = {
  activeAccountId: PropTypes.string,
  clusterName: PropTypes.string,
  doc: PropTypes.string,
  docUrl: PropTypes.string,
  endpointAccess: PropTypes.string,
  gettingStartedUrl: PropTypes.string,
  headerContext: PropTypes.string,
  history: PropTypes.object,
  locale: PropTypes.string,
  location: PropTypes.object,
  logo: PropTypes.object,
  namespaces: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  navRoutes: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  oauthInfo: PropTypes.object,
  role: PropTypes.string,
  secondaryNavRoutes: PropTypes.oneOfType([PropTypes.array, PropTypes.object]),
  serviceId: PropTypes.string,
  support: PropTypes.string,
  switchNamespace: PropTypes.func,
  updateHeaderExperience: PropTypes.func,
  updateNav: PropTypes.func,
  whichNav: PropTypes.string,
}

const mapStateToProps = (state) => {
  return {
    user: state.user,
    authenticated: isAuthenticated(state),
    navRoutes: _.get(state, 'nav.navItems'),
    secondaryNavRoutes: state.nav && state.nav.secondaryNavItems,
    role: state.role && state.role.role,
    namespaces: state.namespaces && state.namespaces.namespaces,
    logo: state.logo && state.logo.url,
    doc: state.doc && state.doc.urls,
    support: _.get(state, 'uiconfig.config.header.supportUrl', (state.support && state.support.url)),
    interval: state.uiconfig && state.uiconfig.interval,
    clusterName: state.uiconfig && state.uiconfig.clusterName,
    whichNav: state.experience.whichNav,
    endpointAccess: state.uiconfig && state.uiconfig.xsrfToken,
    headerContext: _.get(state, 'uiconfig.headerContext', ''),
    activeAccountId: state.userpreferences && state.userpreferences.userPreferences.activeAccountId,
    docUrl: _.get(state, 'uiconfig.config.header.docUrlMapping', null),
    gettingStartedUrl: _.get(state, 'uiconfig.config.header.gettingStartedUrl', null),
    oauthInfo: state.oauthInfo
  }
}

const mapDispatchToProps = dispatch => {
  return {
    switchNamespace: namespace => dispatch(switchNamespace(namespace)),
    updateNav: (navItem, role, namespaces) => dispatch(updateNav(navItem, role, namespaces)),
    updateHeaderExperience: whichNav => dispatch(updateHeaderExperience(whichNav))
  }
}

export default withRouter(connect(mapStateToProps, mapDispatchToProps)(HeaderContainer))
