/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
'use strict'

import { modal } from '../../../src-web/reducers/modal'
import * as Actions from '../../../src-web/actions/index'
import { RESOURCE_TYPES } from '../../../lib/shared/constants'

describe('Modal reducer', () => {
  describe('#modal', () => {
    it('should return the initial state', () => {
      expect(modal(undefined, {})).toEqual({})
    })
    it('should handle MODAL_UPDATE', () => {
      const modalData = {
        action: 'put',
        data: {},
        editorMode: 'json',
        label: {},
        resourceType: RESOURCE_TYPES.DEPLOYMENT,
        type: 'resource'
      }
      const action = {
        type: Actions.MODAL_UPDATE,
        data: modalData
      }
      expect(modal({}, action)).toEqual(modalData)
    })
    it('should handle CLEAR_REQUEST_STATUS', () => {
      const modalErrorData = {
        reqCount: 0,
        reqErrCount: 1,
        reqErrorMsg: 'Internal server error',
        reqStatus: Actions.REQUEST_STATUS.ERROR
      }
      const action = {
        type: Actions.CLEAR_REQUEST_STATUS,
        data: modalErrorData
      }
      expect(modal({}, action)).toEqual({reqStatus: undefined, reqErrorMsg: undefined, reqCount: undefined, reqErrCount: undefined})
    })
    it('should handle POST_REQUEST', () => {
      const action = { type: Actions.POST_REQUEST }
      expect(modal({}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.IN_PROGRESS, reqCount: 1 })
    })
    it('should handle PUT_REQUEST', () => {
      const action = { type: Actions.PUT_REQUEST }
      expect(modal({}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.IN_PROGRESS, reqCount: 1 })
    })
    it('should handle DEL_REQUEST', () => {
      const action = { type: Actions.DEL_REQUEST }
      expect(modal({}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.IN_PROGRESS, reqCount: 1 })
    })
    it('should handle POST_RECEIVE_SUCCESS', () => {
      const action = { type: Actions.POST_RECEIVE_SUCCESS }
      expect(modal({reqCount: 1}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.DONE, reqCount: 0 })
    })
    it('should handle PUT_RECEIVE_SUCCESS', () => {
      const action = { type: Actions.PUT_RECEIVE_SUCCESS }
      expect(modal({reqCount: 1}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.DONE, reqCount: 0 })
    })
    it('should handle DEL_RECEIVE_SUCCESS', () => {
      const action = { type: Actions.DEL_RECEIVE_SUCCESS }
      expect(modal({reqCount: 1}, action)).toEqual({ reqStatus: Actions.REQUEST_STATUS.DONE, reqCount: 0 })
    })
    it('should handle POST_RECEIVE_FAILURE', () => {
      const action = {
        type: Actions.POST_RECEIVE_FAILURE,
        err: { error: { message: 'Internal server error'} }
      }
      expect(modal({ reqCount: 1, reqErrCount: 1}, action)).toEqual({ reqCount: 0, reqErrCount: 2, reqStatus: Actions.REQUEST_STATUS.ERROR, reqErrorMsg: action.err.error.message })
    })
    it('should handle POST_RECEIVE_FAILURE when reqCount and reqErrCount is not defined', () => {
      const action = {
        type: Actions.POST_RECEIVE_FAILURE,
        err: { error: { message: 'Internal server error'} }
      }
      expect(modal({}, action)).toEqual({ reqCount: 0, reqErrCount: 1, reqStatus: Actions.REQUEST_STATUS.ERROR, reqErrorMsg: action.err.error.message })
    })
    it('should handle PUT_RECEIVE_FAILURE', () => {
      const action = {
        type: Actions.PUT_RECEIVE_FAILURE,
        err: { error: { message: 'Internal server error'} }
      }
      expect(modal({ reqCount: 1, reqErrCount: 1}, action)).toEqual({ reqCount: 0, reqErrCount: 2, reqStatus: Actions.REQUEST_STATUS.ERROR, reqErrorMsg: action.err.error.message })
    })
    it('should handle DEL_RECEIVE_FAILURE', () => {
      const action = {
        type: Actions.DEL_RECEIVE_FAILURE,
        err: { error: { message: 'Internal server error'} }
      }
      expect(modal({ reqCount: 1, reqErrCount: 1}, action)).toEqual({ reqCount: 0, reqErrCount: 2, reqStatus: Actions.REQUEST_STATUS.ERROR, reqErrorMsg: action.err.error.message })
    })
  })
})
