/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2020. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 * Copyright (c) 2020 Red Hat, Inc.
 *******************************************************************************/
/* Copyright (c) 2020 Red Hat, Inc. */

const path = require('path'),
      config = require('./config'),
      webpack = require('webpack'),
      AssetsPlugin = require('assets-webpack-plugin'),
      CompressionPlugin = require('compression-webpack-plugin'),
      CopyPlugin = require('copy-webpack-plugin'),
      MonacoWebpackPlugin = require('monaco-editor-webpack-plugin'),
      MiniCssExtractPlugin = require('mini-css-extract-plugin'),
      TerserPlugin = require('terser-webpack-plugin'),
      WebpackMd5Hash = require('webpack-md5-hash')

const NO_OP = () => {/*This is intentional*/ },
      PRODUCTION = process.env.BUILD_ENV ? /production/.test(process.env.BUILD_ENV) : false

process.env.BABEL_ENV = process.env.BABEL_ENV ? process.env.BABEL_ENV : 'client'

const prodExternals = {}

const overpassTest = /overpass-.*\.(woff2?|ttf|eot|otf)(\?.*$|$)/

module.exports = {
  context: __dirname,
  devtool: PRODUCTION ? 'source-map' : 'cheap-module-eval-source-map',
  stats: { children: false },
  entry: {
    'common': ['./scss/common.scss'],
    'main': ['./src-web/index.js'],
    'header': ['./src-web/index.header.js']
  },

  externals: Object.assign(PRODUCTION ? prodExternals : {}, {
    // replace require-server with empty function on client
    './require-server': 'var function(){}'
  }),

  module: {
    rules: [
      {
        test: /\.js$/,
        enforce: 'pre',
        loader: 'eslint-loader',
        options: {
          quiet: true
        }
      },
      {
        // Transpile React JSX to ES5
        test: [/\.jsx$/, /\.js$/],
        exclude: /node_modules|\.scss/,
        loader: 'babel-loader?cacheDirectory',
      },
      {
        test: [/\.scss$/],
        use: [
          MiniCssExtractPlugin.loader,
          {
            loader: 'css-loader?sourceMap', // translates CSS into CommonJS
            options: {
              // minimize: PRODUCTION ? true : false
            },
          },
          {
            loader: 'postcss-loader?sourceMap', // compiles postcss to CSS
            options: {
              plugins: function () {
                return [
                  require('autoprefixer')
                ]
              },
            },
          },
          {
            loader: 'sass-loader?sourceMap', // compiles Sass to CSS
            options: {
              prependData: '$font-path: "'+ config.get('headerContextPath') + '/fonts";',
            },
          },
        ],
      },
      {
        test: /\.css$/,
        include: path.resolve(__dirname, './node_modules/monaco-editor'),
        use: ['style-loader', 'css-loader'],
      },
      {
        test: /\.properties$/,
        loader: 'properties-loader'
      },
      {
        test: /\.svg$/,
        use: [
          'svg-sprite-loader'
        ]
      },
      {
        test: [/\.handlebars$/, /\.hbs$/],
        loader: 'handlebars-loader',
        query: {
          helperDirs: [
            path.resolve(__dirname, './templates/helpers')
          ],
          precompileOptions: {
            knownHelpersOnly: false
          }
        }
      },
      {
        test: /\.(png|jpg)$/,
        use: [
          'svg-sprite-loader?symbolId=icon-[name]',
          'image2svg-loader',
        ],
      },
      {
        test: /\.(woff2?|ttf|eot|otf)(\?.*$|$)/,
        exclude: overpassTest,
        loader: 'file-loader',
        options: {
          name: 'assets/[name].[ext]',
        },
      },
      {
        // Resolve to an empty module for overpass fonts included in SASS files.
        // This way file-loader won't parse them. Make sure this is BELOW the
        // file-loader rule.
        test: overpassTest,
        loader: 'null-loader',
      },
    ],
    noParse: [
      // don't parse minified bundles (vendor libs) for faster builds
      /\.min\.js$/
    ]
  },

  node: {
    cluster: 'empty',
    net: 'empty',
    tls: 'empty',
    fs: 'empty'
  },

  output: {
    //needs to be hash for production (vs chunckhash) in order to cache bust references to chunks
    chunkFilename: PRODUCTION ? 'js/[name].[chunkhash].min.js' : 'js/[name].min.js',
    filename: PRODUCTION ? 'js/[name].[hash].min.js' : 'js/[name].min.js',
    jsonpFunction: 'webpackJsonpFunctionHeader',
    path: __dirname + '/public',
    publicPath: config.get('headerContextPath').replace(/\/?$/, '/')
  },

  plugins: [
    new webpack.DefinePlugin({
      'process.env': {
        NODE_ENV: JSON.stringify(PRODUCTION ? 'production' : 'development'),
      },
      CONSOLE_CONTEXT_URL: JSON.stringify(config.get('headerContextPath'))
    }),
    new webpack.DllReferencePlugin({
      context: __dirname,
      // eslint-disable-next-line import/no-unresolved
      manifest: require('./dll/vendor-manifest.json')
    }),
    new MiniCssExtractPlugin({
      filename: 'api/css/login.[contenthash].css'
    }),
    new MiniCssExtractPlugin({
      filename: PRODUCTION ? 'css/[name].[contenthash].css' : 'css/[name].css',
      allChunks: true
    }),
    PRODUCTION ? new TerserPlugin({
      sourceMap: true
    }) : NO_OP,
    new webpack.LoaderOptionsPlugin({
      options: {
        eslint: {
          configFile: './.eslintrc.json',
          quiet: true
        }
      }
    }),
    new webpack.LoaderOptionsPlugin({
      options: {
        context: __dirname
      }
    }),
    new CompressionPlugin({
      filename: '[path].gz[query]',
      algorithm: 'gzip',
      test: /\.js$|\.css$/,
      minRatio: 1
    }),
    new MonacoWebpackPlugin({
      languages: ['yaml', 'json'],
    }),
    new AssetsPlugin({
      path: path.join(__dirname, 'public'),
      fullPath: false,
      prettyPrint: true,
      update: true
    }),
    PRODUCTION ? new webpack.HashedModuleIdsPlugin() : new webpack.NamedModulesPlugin(),
    new WebpackMd5Hash(),

    new CopyPlugin({
      patterns: [
        { from: 'node_modules/carbon-icons/dist/carbon-icons.svg', to: 'graphics' },
        { from: 'graphics', to: 'graphics' },
        { from: 'fonts', to: 'fonts' },
      ],
      options: {
        concurrency: 100,
      },
    })
  ],

  resolve: {
    alias: {
      handlebars: 'handlebars/dist/handlebars.min.js'
    }
  },

  resolveLoader: {
    modules: [
      path.join(__dirname, 'node_modules'),
      // properties-loader
      path.join(__dirname, 'node_modules/node-i18n-util/lib')
    ]
  }
}
