"use strict";
/// <reference lib="esnext.asynciterable" />
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __await = (this && this.__await) || function (v) { return this instanceof __await ? (this.v = v, this) : new __await(v); }
var __asyncGenerator = (this && this.__asyncGenerator) || function (thisArg, _arguments, generator) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var g = generator.apply(thisArg, _arguments || []), i, q = [];
    return i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i;
    function verb(n) { if (g[n]) i[n] = function (v) { return new Promise(function (a, b) { q.push([n, v, a, b]) > 1 || resume(n, v); }); }; }
    function resume(n, v) { try { step(g[n](v)); } catch (e) { settle(q[0][3], e); } }
    function step(r) { r.value instanceof __await ? Promise.resolve(r.value.v).then(fulfill, reject) : settle(q[0][2], r); }
    function fulfill(value) { resume("next", value); }
    function reject(value) { resume("throw", value); }
    function settle(f, v) { if (f(v), q.shift(), q.length) resume(q[0][0], q[0][1]); }
};
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebClientEvent = exports.WebClient = void 0;
// polyfill for async iterable. see: https://stackoverflow.com/a/43694282/305340
// can be removed once node v10 is the minimum target (node v8 and v9 require --harmony_async_iteration flag)
if (Symbol['asyncIterator'] === undefined) {
    (Symbol['asyncIterator']) = Symbol.for('asyncIterator');
}
const querystring_1 = require("querystring");
const path_1 = require("path");
const is_stream_1 = __importDefault(require("is-stream"));
const p_queue_1 = __importDefault(require("p-queue")); // tslint:disable-line:import-name
const p_retry_1 = __importStar(require("p-retry"));
const axios_1 = __importDefault(require("axios"));
const form_data_1 = __importDefault(require("form-data")); // tslint:disable-line:import-name
const is_electron_1 = __importDefault(require("is-electron"));
const methods_1 = require("./methods");
const instrument_1 = require("./instrument");
const errors_1 = require("./errors");
const logger_1 = require("./logger");
const retry_policies_1 = __importDefault(require("./retry-policies"));
const helpers_1 = require("./helpers");
/**
 * A client for Slack's Web API
 *
 * This client provides an alias for each {@link https://api.slack.com/methods|Web API method}. Each method is
 * a convenience wrapper for calling the {@link WebClient#apiCall} method using the method name as the first parameter.
 */
class WebClient extends methods_1.Methods {
    /**
     * @param token - An API token to authenticate/authorize with Slack (usually start with `xoxp`, `xoxb`)
     */
    constructor(token, { slackApiUrl = 'https://slack.com/api/', logger = undefined, logLevel = undefined, maxRequestConcurrency = 3, retryConfig = retry_policies_1.default.tenRetriesInAboutThirtyMinutes, agent = undefined, tls = undefined, timeout = 0, rejectRateLimitedCalls = false, headers = {}, teamId = undefined, } = {}) {
        super();
        this.token = token;
        this.slackApiUrl = slackApiUrl;
        this.retryConfig = retryConfig;
        this.requestQueue = new p_queue_1.default({ concurrency: maxRequestConcurrency });
        // NOTE: may want to filter the keys to only those acceptable for TLS options
        this.tlsConfig = tls !== undefined ? tls : {};
        this.rejectRateLimitedCalls = rejectRateLimitedCalls;
        this.teamId = teamId;
        // Logging
        if (typeof logger !== 'undefined') {
            this.logger = logger;
            if (typeof logLevel !== 'undefined') {
                this.logger.debug('The logLevel given to WebClient was ignored as you also gave logger');
            }
        }
        else {
            this.logger = logger_1.getLogger(WebClient.loggerName, logLevel !== null && logLevel !== void 0 ? logLevel : logger_1.LogLevel.INFO, logger);
        }
        this.axios = axios_1.default.create({
            timeout,
            baseURL: slackApiUrl,
            headers: is_electron_1.default() ? headers : Object.assign({ 'User-Agent': instrument_1.getUserAgent() }, headers),
            httpAgent: agent,
            httpsAgent: agent,
            transformRequest: [this.serializeApiCallOptions.bind(this)],
            validateStatus: () => true,
            maxRedirects: 0,
            // disabling axios' automatic proxy support:
            // axios would read from envvars to configure a proxy automatically, but it doesn't support TLS destinations.
            // for compatibility with https://api.slack.com, and for a larger set of possible proxies (SOCKS or other
            // protocols), users of this package should use the `agent` option to configure a proxy.
            proxy: false,
        });
        // serializeApiCallOptions will always determine the appropriate content-type
        delete this.axios.defaults.headers.post['Content-Type'];
        this.logger.debug('initialized');
    }
    /**
     * Generic method for calling a Web API method
     *
     * @param method - the Web API method to call {@link https://api.slack.com/methods}
     * @param options - options
     */
    async apiCall(method, options) {
        this.logger.debug(`apiCall('${method}') start`);
        warnDeprecations(method, this.logger);
        warnIfFallbackIsMissing(method, this.logger, options);
        warnIfThreadTsIsNotString(method, this.logger, options);
        if (typeof options === 'string' || typeof options === 'number' || typeof options === 'boolean') {
            throw new TypeError(`Expected an options argument but instead received a ${typeof options}`);
        }
        const response = await this.makeRequest(method, Object.assign({
            token: this.token,
            team_id: this.teamId,
        }, options));
        const result = this.buildResult(response);
        // log warnings in response metadata
        if (result.response_metadata !== undefined && result.response_metadata.warnings !== undefined) {
            result.response_metadata.warnings.forEach(this.logger.warn.bind(this.logger));
        }
        // log warnings and errors in response metadata messages
        // related to https://api.slack.com/changelog/2016-09-28-response-metadata-is-on-the-way
        if (result.response_metadata !== undefined && result.response_metadata.messages !== undefined) {
            result.response_metadata.messages.forEach((msg) => {
                const errReg = /\[ERROR\](.*)/;
                const warnReg = /\[WARN\](.*)/;
                if (errReg.test(msg)) {
                    const errMatch = msg.match(errReg);
                    if (errMatch != null) {
                        this.logger.error(errMatch[1].trim());
                    }
                }
                else if (warnReg.test(msg)) {
                    const warnMatch = msg.match(warnReg);
                    if (warnMatch != null) {
                        this.logger.warn(warnMatch[1].trim());
                    }
                }
            });
        }
        if (!result.ok) {
            throw errors_1.platformErrorFromResult(result);
        }
        return result;
    }
    paginate(method, options, shouldStop, reduce) {
        if (!methods_1.cursorPaginationEnabledMethods.has(method)) {
            this.logger.warn(`paginate() called with method ${method}, which is not known to be cursor pagination enabled.`);
        }
        const pageSize = (() => {
            if (options !== undefined && typeof options.limit === 'number') {
                const limit = options.limit;
                delete options.limit;
                return limit;
            }
            return defaultPageSize;
        })();
        function generatePages() {
            return __asyncGenerator(this, arguments, function* generatePages_1() {
                // when result is undefined, that signals that the first of potentially many calls has not yet been made
                let result = undefined;
                // paginationOptions stores pagination options not already stored in the options argument
                let paginationOptions = {
                    limit: pageSize,
                };
                if (options !== undefined && options.cursor !== undefined) {
                    paginationOptions.cursor = options.cursor;
                }
                // NOTE: test for the situation where you're resuming a pagination using and existing cursor
                while (result === undefined || paginationOptions !== undefined) {
                    result = yield __await(this.apiCall(method, Object.assign(options !== undefined ? options : {}, paginationOptions)));
                    yield yield __await(result);
                    paginationOptions = paginationOptionsForNextPage(result, pageSize);
                }
            });
        }
        if (shouldStop === undefined) {
            return generatePages.call(this);
        }
        const pageReducer = (reduce !== undefined) ? reduce : noopPageReducer;
        let index = 0;
        return (async () => {
            // Unroll the first iteration of the iterator
            // This is done primarily because in order to satisfy the type system, we need a variable that is typed as A
            // (shown as accumulator before), but before the first iteration all we have is a variable typed A | undefined.
            // Unrolling the first iteration allows us to deal with undefined as a special case.
            var e_1, _a;
            const pageIterator = generatePages.call(this);
            const firstIteratorResult = await pageIterator.next(undefined);
            // Assumption: there will always be at least one result in a paginated API request
            // if (firstIteratorResult.done) { return; }
            const firstPage = firstIteratorResult.value;
            let accumulator = pageReducer(undefined, firstPage, index);
            index += 1;
            if (shouldStop(firstPage)) {
                return accumulator;
            }
            try {
                // Continue iteration
                for (var pageIterator_1 = __asyncValues(pageIterator), pageIterator_1_1; pageIterator_1_1 = await pageIterator_1.next(), !pageIterator_1_1.done;) {
                    const page = pageIterator_1_1.value;
                    accumulator = pageReducer(accumulator, page, index);
                    if (shouldStop(page)) {
                        return accumulator;
                    }
                    index += 1;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (pageIterator_1_1 && !pageIterator_1_1.done && (_a = pageIterator_1.return)) await _a.call(pageIterator_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
            return accumulator;
        })();
    }
    /**
     * Low-level function to make a single API request. handles queuing, retries, and http-level errors
     */
    async makeRequest(url, body, headers = {}) {
        // TODO: better input types - remove any
        const task = () => this.requestQueue.add(async () => {
            this.logger.debug('will perform http request');
            try {
                const response = await this.axios.post(url, body, Object.assign({
                    headers,
                }, this.tlsConfig));
                this.logger.debug('http response received');
                if (response.status === 429) {
                    const retrySec = parseRetryHeaders(response);
                    if (retrySec !== undefined) {
                        this.emit(WebClientEvent.RATE_LIMITED, retrySec);
                        if (this.rejectRateLimitedCalls) {
                            throw new p_retry_1.AbortError(errors_1.rateLimitedErrorWithDelay(retrySec));
                        }
                        this.logger.info(`API Call failed due to rate limiting. Will retry in ${retrySec} seconds.`);
                        // pause the request queue and then delay the rejection by the amount of time in the retry header
                        this.requestQueue.pause();
                        // NOTE: if there was a way to introspect the current RetryOperation and know what the next timeout
                        // would be, then we could subtract that time from the following delay, knowing that it the next
                        // attempt still wouldn't occur until after the rate-limit header has specified. an even better
                        // solution would be to subtract the time from only the timeout of this next attempt of the
                        // RetryOperation. this would result in the staying paused for the entire duration specified in the
                        // header, yet this operation not having to pay the timeout cost in addition to that.
                        await helpers_1.delay(retrySec * 1000);
                        // resume the request queue and throw a non-abort error to signal a retry
                        this.requestQueue.start();
                        throw Error('A rate limit was exceeded.');
                    }
                    else {
                        // TODO: turn this into some CodedError
                        throw new p_retry_1.AbortError(new Error('Retry header did not contain a valid timeout.'));
                    }
                }
                // Slack's Web API doesn't use meaningful status codes besides 429 and 200
                if (response.status !== 200) {
                    throw errors_1.httpErrorFromResponse(response);
                }
                return response;
            }
            catch (error) {
                this.logger.warn('http request failed', error.message);
                if (error.request) {
                    throw errors_1.requestErrorWithOriginal(error);
                }
                throw error;
            }
        });
        return p_retry_1.default(task, this.retryConfig);
    }
    /**
     * Transforms options (a simple key-value object) into an acceptable value for a body. This can be either
     * a string, used when posting with a content-type of url-encoded. Or, it can be a readable stream, used
     * when the options contain a binary (a stream or a buffer) and the upload should be done with content-type
     * multipart/form-data.
     *
     * @param options - arguments for the Web API method
     * @param headers - a mutable object representing the HTTP headers for the outgoing request
     */
    serializeApiCallOptions(options, headers) {
        // The following operation both flattens complex objects into a JSON-encoded strings and searches the values for
        // binary content
        let containsBinaryData = false;
        const flattened = Object.entries(options)
            .map(([key, value]) => {
            if (value === undefined || value === null) {
                return [];
            }
            let serializedValue = value;
            if (Buffer.isBuffer(value) || is_stream_1.default(value)) {
                containsBinaryData = true;
            }
            else if (typeof value !== 'string' && typeof value !== 'number' && typeof value !== 'boolean') {
                // if value is anything other than string, number, boolean, binary data, a Stream, or a Buffer, then encode it
                // as a JSON string.
                serializedValue = JSON.stringify(value);
            }
            return [key, serializedValue];
        });
        // A body with binary content should be serialized as multipart/form-data
        if (containsBinaryData) {
            this.logger.debug('request arguments contain binary data');
            const form = flattened.reduce((form, [key, value]) => {
                if (Buffer.isBuffer(value) || is_stream_1.default(value)) {
                    const options = {};
                    options.filename = (() => {
                        // attempt to find filename from `value`. adapted from:
                        // tslint:disable-next-line:max-line-length
                        // https://github.com/form-data/form-data/blob/028c21e0f93c5fefa46a7bbf1ba753e4f627ab7a/lib/form_data.js#L227-L230
                        // formidable and the browser add a name property
                        // fs- and request- streams have path property
                        const streamOrBuffer = value;
                        if (typeof streamOrBuffer.name === 'string') {
                            return path_1.basename(streamOrBuffer.name);
                        }
                        if (typeof streamOrBuffer.path === 'string') {
                            return path_1.basename(streamOrBuffer.path);
                        }
                        return defaultFilename;
                    })();
                    form.append(key, value, options);
                }
                else if (key !== undefined && value !== undefined) {
                    form.append(key, value);
                }
                return form;
            }, new form_data_1.default());
            // Copying FormData-generated headers into headers param
            // not reassigning to headers param since it is passed by reference and behaves as an inout param
            for (const [header, value] of Object.entries(form.getHeaders())) {
                headers[header] = value;
            }
            return form;
        }
        // Otherwise, a simple key-value object is returned
        headers['Content-Type'] = 'application/x-www-form-urlencoded';
        const initialValue = {};
        return querystring_1.stringify(flattened.reduce((accumulator, [key, value]) => {
            if (key !== undefined && value !== undefined) {
                accumulator[key] = value;
            }
            return accumulator;
        }, initialValue));
    }
    /**
     * Processes an HTTP response into a WebAPICallResult by performing JSON parsing on the body and merging relevant
     * HTTP headers into the object.
     * @param response - an http response
     */
    buildResult(response) {
        const data = response.data;
        if (data.response_metadata === undefined) {
            data.response_metadata = {};
        }
        // add scopes metadata from headers
        if (response.headers['x-oauth-scopes'] !== undefined) {
            data.response_metadata.scopes = response.headers['x-oauth-scopes'].trim().split(/\s*,\s*/);
        }
        if (response.headers['x-accepted-oauth-scopes'] !== undefined) {
            data.response_metadata.acceptedScopes =
                response.headers['x-accepted-oauth-scopes'].trim().split(/\s*,\s*/);
        }
        // add retry metadata from headers
        const retrySec = parseRetryHeaders(response);
        if (retrySec !== undefined) {
            data.response_metadata.retryAfter = retrySec;
        }
        return data;
    }
}
exports.WebClient = WebClient;
/**
 * The name used to prefix all logging generated from this object
 */
WebClient.loggerName = 'WebClient';
exports.default = WebClient;
var WebClientEvent;
(function (WebClientEvent) {
    WebClientEvent["RATE_LIMITED"] = "rate_limited";
})(WebClientEvent = exports.WebClientEvent || (exports.WebClientEvent = {}));
/*
 * Helpers
 */
const defaultFilename = 'Untitled';
const defaultPageSize = 200;
const noopPageReducer = () => undefined;
/**
 * Determines an appropriate set of cursor pagination options for the next request to a paginated API method.
 * @param previousResult - the result of the last request, where the next cursor might be found.
 * @param pageSize - the maximum number of additional items to fetch in the next request.
 */
function paginationOptionsForNextPage(previousResult, pageSize) {
    if (previousResult !== undefined &&
        previousResult.response_metadata !== undefined &&
        previousResult.response_metadata.next_cursor !== undefined &&
        previousResult.response_metadata.next_cursor !== '') {
        return {
            limit: pageSize,
            cursor: previousResult.response_metadata.next_cursor,
        };
    }
    return;
}
/**
 * Extract the amount of time (in seconds) the platform has recommended this client wait before sending another request
 * from a rate-limited HTTP response (statusCode = 429).
 */
function parseRetryHeaders(response) {
    if (response.headers['retry-after'] !== undefined) {
        const retryAfter = parseInt(response.headers['retry-after'], 10);
        if (!Number.isNaN(retryAfter)) {
            return retryAfter;
        }
    }
    return undefined;
}
/**
 * Log a warning when using a deprecated method
 * @param method api method being called
 * @param logger instance of web clients logger
 */
function warnDeprecations(method, logger) {
    const deprecatedConversationsMethods = ['channels.', 'groups.', 'im.', 'mpim.'];
    const deprecatedMethods = ['admin.conversations.whitelist.'];
    const isDeprecatedConversations = deprecatedConversationsMethods.some((depMethod) => {
        const re = new RegExp(`^${depMethod}`);
        return re.test(method);
    });
    const isDeprecated = deprecatedMethods.some((depMethod) => {
        const re = new RegExp(`^${depMethod}`);
        return re.test(method);
    });
    if (isDeprecatedConversations) {
        logger.warn(`${method} is deprecated. Please use the Conversations API instead. For more info, go to https://api.slack.com/changelog/2020-01-deprecating-antecedents-to-the-conversations-api`);
    }
    else if (isDeprecated) {
        logger.warn(`${method} is deprecated. Please check on https://api.slack.com/methods for an alternative.`);
    }
}
/**
 * Log a warning when using chat.postMessage without text argument or attachments with fallback argument
 * @param method api method being called
 * @param logger instance of we clients logger
 * @param options arguments for the Web API method
 */
function warnIfFallbackIsMissing(method, logger, options) {
    const targetMethods = ['chat.postEphemeral', 'chat.postMessage', 'chat.scheduleMessage', 'chat.update'];
    const isTargetMethod = targetMethods.includes(method);
    const missingAttachmentFallbackDetected = (args) => Array.isArray(args.attachments)
        && args.attachments.some((attachment) => { return (!attachment.fallback || attachment.fallback.trim() === 0); });
    const isEmptyText = (args) => args.text === undefined || args.text === null || args.text === '';
    const buildWarningMessage = (missing) => `The \`${missing}\` argument is missing in the request payload for a ${method} call - ` +
        `It's a best practice to always provide a \`${missing}\` argument when posting a message. ` +
        `The \`${missing}\` is used in places where the content cannot be rendered such as: ` +
        'system push notifications, assistive technology such as screen readers, etc.';
    if (isTargetMethod && typeof options === 'object' && isEmptyText(options)) {
        if (missingAttachmentFallbackDetected(options)) {
            logger.warn(buildWarningMessage('fallback'));
        }
        else {
            logger.warn(buildWarningMessage('text'));
        }
    }
}
/**
 * Log a warning when thread_ts is not a string
 * @param method api method being called
 * @param logger instance of web clients logger
 * @param options arguments for the Web API method
 */
function warnIfThreadTsIsNotString(method, logger, options) {
    const targetMethods = ['chat.postEphemeral', 'chat.postMessage', 'chat.scheduleMessage', 'files.upload'];
    const isTargetMethod = targetMethods.includes(method);
    if (isTargetMethod && (options === null || options === void 0 ? void 0 : options.thread_ts) !== undefined && typeof (options === null || options === void 0 ? void 0 : options.thread_ts) !== 'string') {
        logger.warn(`The given thread_ts value in the request payload for a ${method} call is a float value. We highly recommend using a string value instead.`);
    }
}
//# sourceMappingURL=WebClient.js.map