// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.
package accesspolicy

import (
	"reflect"
	"sort"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/open-cluster-management/hcm-compliance/pkg/apis/accesspolicy/v1alpha1"
	"github.com/open-cluster-management/seed-sdk/pkg/client"
	sctx "github.com/open-cluster-management/seed-sdk/pkg/context"
	"github.com/open-cluster-management/seed-sdk/pkg/controller"
	"github.com/open-cluster-management/seed-sdk/pkg/testing/framework"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

func TestFindPattern(t *testing.T) {
	list := []string{"Hello-World", "World-Hello", "Hello-World-Hello", "nothing", "exact"}

	//testing PREFIX
	actualResult := findPattern("Hello*", list)
	expectedResult := []string{"Hello-World", "Hello-World-Hello"}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}

	//testing SUFFIX
	actualResult = findPattern("*Hello", list)
	expectedResult = []string{"World-Hello", "Hello-World-Hello"}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}

	//testing if it CONTAINS the pattern
	actualResult = findPattern("*Hello*", list)
	expectedResult = []string{"Hello-World", "World-Hello", "Hello-World-Hello"}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}

	//testing if it does NOT contain the pattern
	actualResult = findPattern("*xxx*", list)
	expectedResult = []string{}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}

	//testing if it  contains the EXACT pattern
	actualResult = findPattern("Hello-World", list)
	expectedResult = []string{"Hello-World"}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}

	//testing corner case
	actualResult = findPattern("*ku*be", list)
	expectedResult = []string{}

	if !reflect.DeepEqual(actualResult, expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	}
}

func TestDeduplicateItems(t *testing.T) {

	included := []string{"Hello-World", "World-Hello", "Hello-World-Hello", "nothing", "exact"}
	excluded := []string{"Hello-World", "Hello-World-Hello", "exact"}

	actualResult := deduplicateItems(included, excluded)
	expectedResult := []string{"World-Hello", "nothing"}
	if len(actualResult) != len(expectedResult) {
		t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
	} else {
		sort.Strings(expectedResult)
		sort.Strings(actualResult)
		if !reflect.DeepEqual(actualResult, expectedResult) {
			t.Fatalf("Expected %s but got %s", expectedResult, actualResult)
		}
	}
}

func TestCreateCRD(t *testing.T) {
	ctx, cancelFunc := sctx.NewGlobal()
	defer cancelFunc()

	tl := framework.NewIntegrationLogger(t)
	err := controller.New().
		Install(v1alpha1.CustomResourceDefinitions).
		Watch("accesspolicies.v1alpha1.accesspolicy.mcm.ibm.com", "AccessPolicy", &v1alpha1.SchemeBuilder).
		RunLocal(ctx, tl)
	if err != nil {
		t.Fatal(err)
	}

	// start controller & create client
	client, err := client.New(ctx.RESTConfig(), v1alpha1.SchemeGroupVersion, ctx.Scheme())
	if err != nil {
		t.Fatal(err)
	}

	// build access policy sample
	ap := buildCR()
	if err = client.Create(ap); err != nil {
		t.Fatal(err)
	}

	// retrieve access policy back
	err = client.Get(ap)
	assert.NoError(t, err)

	// verify that object was updated and received a guid
	assert.NotEmpty(t, ap.ObjectMeta.GetUID)

	assert.Equal(t, "source", ap.Spec.Source.Service)

	// // update object
	// policy.Spec.ComplianceType = v1alpha1.MustNotHave
	// client.Update(policy)

	// // object gets updated in place when retrieved
	// err = client.Get(policy)
	// assert.NoError(t, err)

	// // verify that object was updated with new val
	// assert.Equal(t, v1alpha1.MustNotHave, policy.Spec.ComplianceType)
}

func buildCR() *v1alpha1.AccessPolicy {
	// Create an instance of our custom resource
	name := "aptest1"
	rule := v1alpha1.AccessPolicyRule{
		Protocol:  "html",
		Actions:   []string{"get"},
		Resources: []string{"/*"},
	}
	cr := &v1alpha1.AccessPolicy{
		TypeMeta: metav1.TypeMeta{
			Kind:       "AccessPolicy",
			APIVersion: "accesspolicy.mcm.ibm.com/v1alpha1",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name:      name,
			Namespace: "default",
		},
		Spec: v1alpha1.AccessPolicySpec{
			Namespaces: "namespace",
			Source: v1alpha1.Microservice{
				Service: "source",
			},
			Destination: v1alpha1.Microservice{
				Service: "destination",
			},
			Rules: []v1alpha1.AccessPolicyRule{rule},
		},
	}
	return cr
}
