// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.

// Package compliance handles compliance controller logic
package compliance

import (
	"reflect"

	//"reflect"
	"sync"
	"time"

	"github.com/golang/glog"
	compliancev1alpha1 "github.com/open-cluster-management/hcm-compliance/pkg/apis/compliance/v1alpha1"
	policyv1alpha1 "github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"

	//	cplList "github.com/open-cluster-management/hcm-compliance/pkg/client/listers/compliance/v1alpha1"
	complianceLister "github.com/open-cluster-management/hcm-compliance/pkg/client/listers/compliance/v1alpha1"
	plcList "github.com/open-cluster-management/hcm-compliance/pkg/client/listers/policy/v1alpha1"

	common "github.com/open-cluster-management/hcm-compliance/pkg/common"

	resourceClient "github.com/open-cluster-management/seed-sdk/pkg/client"
	"github.com/open-cluster-management/seed-sdk/pkg/context"
	resv1 "github.com/open-cluster-management/seed-sdk/pkg/types/apis/resource/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/client-go/tools/cache"
)

const name = "compliance"

var myFinalizerNames = []string{name + common.Finalizer, common.OldFinalizer}

var (
	//CplClient used to communicate with K8s api-server
	CplClient *resourceClient.ResourceClient

	//PlcClient used to List obj
	PlcClient *resourceClient.ResourceClient

	//CplLister used to List obj
	CplLister complianceLister.ComplianceLister

	//PlcLister used to List obj
	PlcLister plcList.PolicyLister

	compliances map[string]*compliancev1alpha1.Compliance

	//GenericLister used to List obj
	GenericLister cache.GenericLister

	//ClusterName holds the cluster name
	ClusterName string

	//WatchedNamespace is the ns we watch
	WatchedNamespace string

	m sync.RWMutex
)

//Initialize the maps
func Initialize(ns, cluster string) {
	compliances = make(map[string]*compliancev1alpha1.Compliance)
	ClusterName = cluster
	WatchedNamespace = ns
}

// Reconcile handles Policy Spec changes
func Reconcile(ctx context.Context, obj *compliancev1alpha1.Compliance) error {
	if !reflect.DeepEqual(WatchedNamespace, obj.Namespace) {
		return nil
	}
	/*
		if reflect.DeepEqual(strings.ToLower(string(obj.Spec.RemediationAction)), strings.ToLower(string("ignore"))) {
			err := HandleCompliance(ctx, obj, false)
			if err != nil {
				glog.Errorf("Error cleaning up after deletion %v", err)
			} else {
				glog.Infof("Ignoring the Compliance `%v` and deleting its policies", err)
			}
			return nil
		}
	*/
	err := resv1.EnsureFinalizerAndPut(ctx, obj, name+common.Finalizer)
	if err != nil {
		glog.Errorf("Error ensuring finalizer %v ", err)
	}
	err = HandleCompliance(ctx, obj, true)
	if err != nil {
		glog.V(4).Infof("Error handling policy %v", err)
	}
	glog.V(4).Infof("Received Compliance %v", obj.Name)

	return err
}

// Finalizer removes the finalizer name
func Finalizer(ctx context.Context, obj *compliancev1alpha1.Compliance) error {
	err := HandleCompliance(ctx, obj, true)
	if err != nil {
		glog.Errorf("Error cleaning up after deletion %v", err)
	}
	glog.V(3).Info("Deleting resource")
	common.RemoveFinalizerAndPut(ctx, obj, myFinalizerNames)
	//resv1.RemoveFinalizerAndPut(ctx, obj, name + common.Finalizer)
	return nil
}

// HandleCompliance will parse the compliance and figure out teh actions needed to enforce it.
func HandleCompliance(ctx context.Context, cpl *compliancev1alpha1.Compliance, added bool) error {
	if added {
		m.Lock()
		for index, plc := range cpl.Spec.RuntimeRules {
			if plc.Namespace == "" {
				cpl.Spec.RuntimeRules[index].Namespace = cpl.Namespace
			}
		}
		compliances[cpl.Name] = cpl // update map

		m.Unlock()

		if !checkComplianceValidity(cpl) {
			glog.V(3).Infof("the compliance `%v` is invalid", cpl.Name)
		}
		for _, plc := range cpl.Spec.RuntimeRules {
			//fmt.Printf("compliance `%v` has policy `%v`\n", cpl.Name, plc.Name)
			OwnerReferences := []metav1.OwnerReference{
				*metav1.NewControllerRef(cpl, schema.GroupVersionKind{
					Group:   compliancev1alpha1.SchemeGroupVersion.Group,
					Version: compliancev1alpha1.SchemeGroupVersion.Version,
					Kind:    "Compliance",
				}),
			}
			plc.SetOwnerReferences(OwnerReferences)
			lbl := make(map[string]string)
			lbl["compliance"] = cpl.Name
			if cpl.Spec.Ignore == true {
				lbl["ignore"] = "true" //change only the label
			} else {
				lbl["ignore"] = "false"

			}
			plc.SetLabels(lbl)
			m.Lock()
			createPolicy(cpl, &plc)
			m.Unlock()

		}
	} else { //remove a compliance
		m.Lock()
		delete(compliances, cpl.Name) // delete compliacne from map
		m.Unlock()
		/*
			for _, plc := range cpl.Spec.RuntimeRules {
				fmt.Printf("compliance `%v` has policy `%v`\n", cpl.Name, plc.Name)
				if plc.Namespace == "" {
					plc.Namespace = cpl.Namespace
				}
				opt := &metav1.DeleteOptions{}
				err := ResClient.Delete(&plc, opt)
				if err != nil {
					glog.Errorf("Error deleting policy %v  based on the deletion of Compliance `%v`, the error is: %v ", plc.Name, cpl.Name, err.Error())
				} else {
					glog.V(2).Infof("Deleted the policy `%v` based on the deletion of Compliance `%v`", plc.Name, cpl.Name)
				}
			} */
	}

	return nil
}

//	^ -->
//	|	|
//	|   |
//	<-- v
// ReconcileCompliance a loop for reconciliation of Compliance
func ReconcileCompliance(frequency time.Duration, namespace string) {
	//constantly update the status based on it's policies statuses
	//ResClient.
	//var labelSelector labels.Selector
	for {
		for _, cpl := range compliances {
			cplCompliant := true
			updateNeeded := false

			if cpl.Status.Status == nil {
				cpl.Status.Status = make(map[string]*compliancev1alpha1.CompliancePerClusterStatus)
				//updateNeeded = true
			}
			if _, ok := cpl.Status.Status[ClusterName]; !ok {
				cpl.Status.Status[ClusterName] = &compliancev1alpha1.CompliancePerClusterStatus{ClusterName: ClusterName, AggregatePolicyStatus: map[string]*policyv1alpha1.PolicyStatus{}}
				//updateNeeded = true
			}

			for index, plc := range cpl.Spec.RuntimeRules {

				obj, err := PlcClient.GetObject(plc.Kind, plc.Namespace, plc.Name)
				if err != nil {
					glog.Errorf("Error getting policy `%v` from K8s api-server, the error is: %v", plc.Name, err)
					continue
				}

				newPlc := obj.(*policyv1alpha1.Policy)
				cpl.Spec.RuntimeRules[index].Status = newPlc.Status

				if cpl.Status.Status[ClusterName].AggregatePolicyStatus == nil {
					cpl.Status.Status[ClusterName].AggregatePolicyStatus = make(map[string]*policyv1alpha1.PolicyStatus)
					glog.V(3).Info("cpl.Status[ClusterName].AggregatePolicyStatus = make(map[string]policyv1alpha1.PolicyStatus)")
				}

				if _, ok := cpl.Status.Status[ClusterName].AggregatePolicyStatus[newPlc.Name]; ok {
					if !reflect.DeepEqual(*cpl.Status.Status[ClusterName].AggregatePolicyStatus[newPlc.Name], newPlc.Status) {
						cpl.Status.Status[ClusterName].AggregatePolicyStatus[newPlc.Name] = &newPlc.Status
						updateNeeded = true
					}
				} else {
					cpl.Status.Status[ClusterName].AggregatePolicyStatus[newPlc.Name] = &newPlc.Status
					updateNeeded = true
				}

				if newPlc.Status.ComplianceState != policyv1alpha1.Compliant {
					cplCompliant = false
				}
			}

			if !cplCompliant {

				if cpl.Status.Status[ClusterName].ComplianceState != policyv1alpha1.NonCompliant {
					//we can consider unknown compliancy here too
					cpl.Status.Status[ClusterName].ComplianceState = policyv1alpha1.NonCompliant
					updateNeeded = true

				}
			} else {
				if cpl.Status.Status[ClusterName].ComplianceState != policyv1alpha1.Compliant {
					cpl.Status.Status[ClusterName].ComplianceState = policyv1alpha1.Compliant
					updateNeeded = true
				}
			}
			if updateNeeded {

				updateCompliance(cpl)
			}
		}
		glog.V(3).Info("reconciling compliance")
		time.Sleep(frequency)
	}
}

func createPolicy(cpl *compliancev1alpha1.Compliance, plc *policyv1alpha1.Policy) error {
	//err := resourceClient.Get(plc)
	//var needComplianceUpdate bool
	if plc.Namespace == "" {
		plc.Namespace = cpl.Namespace
	}
	tmpPlc := plc.DeepCopy()
	//err := ResClient.Create(plc)
	err := PlcClient.Get(tmpPlc)
	if err != nil {
		glog.Errorf("policy %v, does not exist, and needs to be created: %v ", plc.Name, err.Error())
		err = PlcClient.Create(plc)
		if err != nil {
			glog.Errorf("Error creating policy %v  based on Compliance `%v`, the error is: %v ", plc.Name, cpl.Name, err.Error())
		} else {
			glog.V(2).Infof("Created the policy `%v` based on Compliance `%v`", plc.Name, cpl.Name)
		}
	} else {
		//the policy exist, we can update it
		plc.ResourceVersion = tmpPlc.ResourceVersion
		plc.Status = tmpPlc.Status
		/*
			if plc.Labels["ignore"] != tmpPlc.Labels["ignore"] {
				//update the compliance
				needComplianceUpdate = true
			}
		*/
		err = PlcClient.Update(plc)
		if err != nil {
			glog.Errorf("Error updating policy %v, the error is: %v", plc.Name, err.Error())
		} else {
			glog.V(2).Infof("Updated the policy `%v` based on Compliance `%v`", plc.Name, cpl.Name)
		}
		/*
			if needComplianceUpdate {
				updateCompliance(cpl)
				if err != nil {
					glog.Errorf("Error updating compliance %v, the error is: %v", cpl.Name, err.Error())
				}
			}
		*/

	}
	return err
}

func updateCompliance(cpl *compliancev1alpha1.Compliance) error {
	//err := resourceClient.Get(plc)
	copy := cpl.DeepCopy()
	tmpCpl := cpl.DeepCopy()
	err := CplClient.Get(tmpCpl)
	if err != nil {
		glog.Errorf("Error getting compliance %v to verify its resource version before updating it, the error is: %v", tmpCpl.Name, err)
	}
	copy.ResourceVersion = tmpCpl.ResourceVersion
	//copy.Status = tmpCpl.Status
	err = CplClient.Update(copy)
	if err != nil {
		glog.Errorf("Error update compliance %v, the error is: %v", cpl.Name, err)
	} else {
		glog.V(2).Infof("Successfully updated the compliance `%v`", cpl.Name)
	}
	return err
}

func checkComplianceValidity(cpl *compliancev1alpha1.Compliance) bool {
	complianceValidity := true
	//TODO add logic here
	return complianceValidity
}

/* LISTER:


//fmt.Printf("cpl  --->%v \nstatus = %v  err = %v \n", plc.Name, newPlc.Status, err)
//labelSelector, _ := labels.Parse(fmt.Sprintf("compliance=compliance1"))
//objects, err := GenericLister.List(labelSelector)
/*
	objects, err := GenericLister.ByNamespace("").List(labels.Everything()) //lists policies

	for _, obj := range objects {
		metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()

		spec := util.GetField(obj, "Status")
		fmt.Printf("%v  \n %v", metadata.GetName(), spec)
	}
	if err != nil {
		glog.Errorf("====error====", err)
	}*/
