// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.
// Copyright (c) 2020 Red Hat, Inc.

package propagator

import (
	"fmt"
	"reflect"
	"sort"
	"strings"
	"sync"
	"time"

	"github.com/golang/glog"
	"k8s.io/apimachinery/pkg/api/equality"
	k8sErrors "k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/labels"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/runtime/schema"
	"k8s.io/apimachinery/pkg/util/wait"
	"k8s.io/client-go/tools/cache"
	"k8s.io/client-go/tools/record"

	"github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"
	policyv1alpha1 "github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"
	placementRuleLister "github.com/open-cluster-management/hcm-compliance/pkg/client/listers/apps/v1"
	"github.com/open-cluster-management/hcm-compliance/pkg/common"
	placement "github.com/open-cluster-management/multicloud-operators-foundation/pkg/apis/mcm/v1alpha1"
	placementLister "github.com/open-cluster-management/multicloud-operators-foundation/pkg/client/listers_generated/mcm/v1alpha1"
	placementrule "github.com/open-cluster-management/multicloud-operators-placementrule/pkg/apis/apps/v1"
	"github.com/open-cluster-management/seed-sdk/pkg/client"
	"github.com/open-cluster-management/seed-sdk/pkg/context"
	resv1 "github.com/open-cluster-management/seed-sdk/pkg/types/apis/resource/v1"
	"github.com/open-cluster-management/seed-sdk/pkg/util"

	mcmSync "github.com/open-cluster-management/hcm-compliance/pkg/sync"
	clreg "k8s.io/cluster-registry/pkg/apis/clusterregistry/v1alpha1"
	clusterregistryv1alpha1 "k8s.io/cluster-registry/pkg/apis/clusterregistry/v1alpha1"
	clLister "k8s.io/cluster-registry/pkg/client/listers/clusterregistry/v1alpha1"
)

const (
	Spec              = "Spec"
	Status            = "Status"
	ComplianceState   = "ComplianceState"
	PlacementPolicies = "PlacementPolicies"
	PlacementBindings = "PlacementBindings"
	name              = "propagator"
)

// finalizers to clean up
var syncFinalizerNames = []string{mcmSync.Name + common.Finalizer, common.Finalizer}
var myFinalizerNames = []string{name + common.Finalizer, common.Finalizer}

type Propagator struct {
	ClusterLister *clLister.ClusterLister

	GenericLister *cache.GenericLister

	PBLister *placementLister.PlacementBindingLister

	PRLister *placementRuleLister.PlacementRuleLister

	ResourceClient *client.ResourceClient

	HcmNamespace string

	// map[Watched object name]map[placementBinding name]PlacementPolicy name
	deployedMap map[string]map[string]string

	mutex sync.RWMutex

	initialized bool

	tmpBindings map[*placement.PlacementBinding]struct{}

	ReconcilePeriod time.Duration

	CheckedKind string

	LabelBlackList []string

	Recorder record.EventRecorder
}

var emptyMap = map[string]*clreg.Cluster{}

func NewPropagator() *Propagator {
	return &Propagator{deployedMap: make(map[string]map[string]string),
		tmpBindings: make(map[*placement.PlacementBinding]struct{}), ReconcilePeriod: 30 * time.Second}
}

// Reconcile handles Generic Object changes
func (pr *Propagator) Reconcile(ctx context.Context, obj interface{}) error {
	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	allClusters, err := (*pr.ClusterLister).List(labels.Everything())

	glog.V(4).Infof("Reconcile Object: %v", metadata.GetSelfLink())

	if err != nil {
		glog.Errorf("Error: reconcileLoop error List all clusters returned: %s", err.Error())
		pr.mutex.Unlock()
		return nil

	}

	if !common.IsInClusterNamespace(metadata.GetNamespace(), allClusters) {
		return pr.handleHCMReconcile(ctx, obj)
	} else if common.HasOwnerReferencesOf(obj.(runtime.Object), "Policy") {
		return pr.handlePolicyReconcile(ctx, obj)
	} else if common.HasOwnerReferencesOf(obj.(runtime.Object), "Compliance") {
		// do not reconcile policy object created by compliance
		return nil
	}
	// reconcile any other objects
	// return pr.handleReconcile(ctx, obj)
	// do not reconcile any other objects
	return nil
}

// ReconcileBinding handles Cluster Spec changes
func (pr *Propagator) ReconcileBinding(ctx context.Context, pb *placement.PlacementBinding) error {
	glog.V(4).Infof("ReconcileBinding %s", pb.SelfLink)
	err := resv1.EnsureFinalizerAndPut(ctx, pb, name+common.Finalizer)
	if err != nil {
		glog.Errorf("Error EnsureFinalizerAndPut: %s", err.Error())
		return err
	}
	return pr.handleBindingReconcile(ctx, pb)
}

// ReconcileCluster handles Cluster, we need this method only to set the finalizer
func (pr *Propagator) ReconcileCluster(ctx context.Context, cls *clreg.Cluster) error {
	glog.V(6).Infof("ReconcileCluster %s, the real reconciliation will be done during reconciliation or resync of a Generic Object",
		cls.Name)
	err := resv1.EnsureFinalizerAndPut(ctx, cls, name+common.Finalizer)
	if err != nil {
		glog.Errorf("Error EnsureFinalizerAndPut: %s", err.Error())
		return err
	}
	// the real reconciliation will done as result of generic object reconciliation ore resync.
	return nil // handleClusterReconcile(ctx, cls)
}

// Finalizer handles removing generic directive from the HCM "root" namespace
func (pr *Propagator) Finalizer(ctx context.Context, obj interface{}) error {
	metadata := (obj).(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(4).Infof("Finalizer %s", metadata.GetSelfLink())

	allClusters, err1 := (*pr.ClusterLister).List(labels.Everything())
	if err1 != nil {
		glog.Errorf("Error: reconcileLoop error List all clusters returned: %s", err1.Error())
		pr.mutex.Unlock()
		return nil

	}
	var err error
	if !common.IsInClusterNamespace(metadata.GetNamespace(), allClusters) {
		err = pr.handleHCMFinalizer(ctx, obj)
	} else {
		err = pr.handleFinalizer(ctx, obj)
	}
	glog.V(6).Infof("Finalizer %s handle method returned", metadata.GetSelfLink())
	if err != nil {
		glog.Errorf("Finalizer Error: handle method for %s returned %s", metadata.GetSelfLink(), err.Error())
		return err
	}
	err = common.RemoveFinalizerAndPut(ctx, obj.(runtime.Object), myFinalizerNames)
	//err = resv1.RemoveFinalizerAndPut(ctx, obj.(runtime.Object), name+common.Finalizer)
	if err != nil {
		glog.Errorf("Finalizer Error: RemoveFinalizerAndPut for %s returned: %s", metadata.GetSelfLink(), err.Error())
		return err
	}
	glog.V(4).Infof("Finalizer %s DONE", metadata.GetSelfLink())
	return nil
}

// FinalizerCluster removes the finalizer name
func (pr *Propagator) FinalizerCluster(ctx context.Context, cls *clreg.Cluster) error {
	glog.V(4).Infof("Finalizer cluster %s", cls.GetSelfLink())

	ns := cls.GetNamespace()

	err := pr.handleClusterFinalizer(ctx, ns)
	if err != nil {
		glog.Errorf("handleClusterFinalizer for cluster %s returned error: %s", cls.Name, err.Error())
		return err
	}
	err = common.RemoveFinalizerAndPut(ctx, cls, myFinalizerNames)
	//err = resv1.RemoveFinalizerAndPut(ctx, cls, name+common.Finalizer)
	if err != nil {
		glog.Errorf("RemoveFinalizerAndPut for cluster %s returned error: %s", cls.Name, err.Error())
		return err
	}
	return nil
}

// FinalizerBinding removes the finalizer name
func (pr *Propagator) FinalizerBinding(ctx context.Context, pb *placement.PlacementBinding) error {
	glog.V(4).Infof("Finalizer PlacementBinding %s", pb.GetSelfLink())

	subjects := pb.Subjects
	pr.mutex.Lock()
	// TODO put in a func
	if len(subjects) != 0 {
		//pl := pb.PlacementPolicyRef
		for _, subj := range subjects {
			if subj.Kind != pr.CheckedKind {
				continue
			}
			depl, OK := pr.deployedMap[pb.GetNamespace()+"/"+subj.Name]
			if !OK || len(depl) == 0 {
				continue
			}
			// several PlacementBindings can map the same subject to the same PlacementPolicies, we have to remove one by one
			//pp := remove(depl, pl.Name, pr.HcmNamespace, false)
			delete(depl, pb.Name)
			glog.V(5).Infof("Finalizer PlacementBinding: subject %s PPolicies %v", subj.Name, depl)
			//pr.deployedMap[subj.Name] = depl
			if len(depl) == 0 {
				delete(pr.deployedMap, subj.Name)
			}
		}
	}
	pr.mutex.Unlock()
	err := common.RemoveFinalizerAndPut(ctx, pb, myFinalizerNames)
	//err := resv1.RemoveFinalizerAndPut(ctx, pb, name+common.Finalizer)
	if err != nil {
		glog.Errorf("RemoveFinalizerAndPut for PlacementBinding %s returned error: %s", pb.Name, err.Error())
		return err
	}
	return nil
}

func (pr *Propagator) handleBindingReconcile(ctx context.Context, pb *placement.PlacementBinding) error {
	glog.V(5).Infof("handleBindingReconcile Object: %v", pb.GetSelfLink())

	// PlacementPolicy
	pl := pb.PlacementPolicyRef

	glog.V(5).Infof("handleBindingReconcile: check PlacementPolicyRef: name = %s", pl.Name)

	pp, err := (*pr.PRLister).PlacementRules(pb.GetNamespace()).Get(pl.Name)

	// we have to update the deployedMap
	ppExsist := true
	if err != nil {
		if !k8sErrors.IsNotFound(err) {
			glog.Errorf("handleBindingReconcile: get PlacementPolicyRef: name = %v, namespace = %v returned %s",
				pl.Name, pb.GetNamespace(), err.Error())
			return err
		} else {
			glog.V(5).Infof("handleBindingReconcile: PlacementPolicyRef: name = %s, namespace = %s doesn't exist",
				pl.Name, pb.GetNamespace())
			ppExsist = false
		}
	}

	// Subject
	subjects := pb.Subjects
	if len(subjects) == 0 {
		// TODO print ?
		return nil
	}
	pr.mutex.Lock()
	defer pr.mutex.Unlock()
	for _, subj := range subjects {
		if subj.Kind != pr.CheckedKind {
			continue
		}
		// check that the subject exists
		_, err := (*pr.GenericLister).ByNamespace(pb.GetNamespace()).Get(subj.Name)
		if err != nil {
			if k8sErrors.IsNotFound(err) {
				glog.V(5).Infof("handleBindingReconcile: DeployedObject: name = %s, namespace = %s doesn't exist",
					subj.Name, pb.GetNamespace())
			} else {
				glog.Errorf("handleBindingReconcile: get DeployedObject: name = %s, namespace = %s returned %s",
					subj.Name, pb.GetNamespace(), err.Error())
				// TODO should we return error ?
			}
			delete(pr.deployedMap, subj.Name)
			glog.V(5).Infof("handleBindingReconcile: remove DeployedObject: name = %s, namespace = %s from the deployedMap",
				subj.Name, pb.GetNamespace())
		} else {
			// subject exists
			placements, ok := pr.deployedMap[pb.GetNamespace()+"/"+subj.Name]
			if ppExsist {
				if !ok {
					placements = make(map[string]string)
				}
				placements[pb.Name] = pp.Name
				//placements = append(placements, key{kind: pp.Kind, name: pp.Name, namespace: pp.Namespace})
				glog.V(5).Infof("handleBindingReconcile: DeployedObject: name = %s, namespace = %s placements = %v",
					subj.Name, pb.GetNamespace(), placements)
				pr.deployedMap[pb.GetNamespace()+"/"+subj.Name] = placements
			} else {
				if ok {
					delete(placements, pb.Name)
					//	placements = remove(placements, pr.HcmNamespace, pl.Name, true)
					if len(placements) == 0 {
						delete(pr.deployedMap, pb.GetNamespace()+"/"+subj.Name)
						glog.V(5).Infof(
							"handleBindingReconcile: remove DeployedObject: name = %s, namespace = %s from the deployedMap",
							subj.Name, pb.GetNamespace())
					}
				}
			}

		}

	}
	if !pr.initialized {
		pr.tmpBindings[pb] = struct{}{}
		bind, err := (*pr.PBLister).List(labels.Everything())
		if err != nil {
			glog.Errorf("handleBindingReconcile: get PlacementBindings: namespace = %s returned %s",
				pb.GetNamespace(), err.Error())
		} else {
			if len(pr.tmpBindings) >= len(bind) {
				pr.tmpBindings = nil
				go pr.runReconciliationLoop(ctx.Done())
				pr.initialized = true
				glog.V(5).Infof("handleBindingReconcile: start reconciliation loop")
			}
		}
	}
	return nil
}

// handleHCMReconcile handles a Generic Object  Reconcile event from the HCM "root" namespace
func (pr *Propagator) handleHCMReconcile(ctx context.Context, obj interface{}) error {
	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(5).Infof("handleHCMReconcile Object: %s", metadata.GetSelfLink())

	spec := util.GetField(obj, "Spec")
	if spec == nil {
		glog.Errorf("Error: handleHCMReconcile Object: %v doesn't have spec", metadata.GetSelfLink())
		return nil
	}
	err := resv1.EnsureFinalizerAndPut(ctx, obj.(runtime.Object), name+common.Finalizer)
	if err != nil {
		glog.Errorf("Error: handleHCMReconcile %s EnsureFinalizerAndPut: %s", metadata.GetSelfLink(), err.Error())
		return err
	}
	pr.mutex.Lock()
	defer pr.mutex.Unlock()

	_, OK := pr.deployedMap[metadata.GetNamespace()+"/"+metadata.GetName()]
	if !OK {
		pr.deployedMap[metadata.GetNamespace()+"/"+metadata.GetName()] = make(map[string]string, 0)
	}
	if !pr.initialized {
		bind, err := (*pr.PBLister).List(labels.Everything())
		if err != nil {
			glog.Errorf("Error: handleHCMReconcile %s getPlacementBindings: %s", metadata.GetSelfLink(), err.Error())
		} else {
			if len(bind) == 0 {
				glog.V(5).Infof("handleHCMReconcile start reconcilation loop")
				go pr.runReconciliationLoop(ctx.Done())
				pr.initialized = true
			}
		}
	}
	glog.V(5).Infof("handleHCMReconcile Object: %s END", metadata.GetSelfLink())
	return nil
}

func (pr *Propagator) handleReconcile(ctx context.Context, obj interface{}) error {

	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(6).Infof("handleReconcile Object: %v", metadata.GetSelfLink())

	objStStr := util.GetField(obj, Status)
	if objStStr == nil {
		glog.V(5).Infof("handleReconcile Object: %v doesn't have status str", metadata.GetSelfLink())
		return nil
	}
	objSt := util.GetField(objStStr, Status)
	if objSt == nil {
		glog.V(5).Infof("handleReconcile Object: %v doesn't have status", metadata.GetSelfLink())
		return nil
	}
	objValue := reflect.ValueOf(objSt)
	keys := objValue.MapKeys()
	if len(keys) == 0 {
		glog.V(5).Infof("Status of handleReconcile Object: %v doesn't have entries", metadata.GetSelfLink())
		return nil
	}
	hcmObj, err := (*pr.GenericLister).ByNamespace(pr.HcmNamespace).Get(metadata.GetName())
	if err != nil {
		if k8sErrors.IsNotFound(err) {
			// TODO remove other ?
			glog.Errorf("Error HCM root Object for %s doesn't exist", metadata.GetName())
			return nil
		}
		glog.Errorf("Error get Root Object with name %s from %s namespace returned : %s",
			metadata.GetName(), pr.HcmNamespace, err.Error())
		return err
	}
	rootStStr := util.GetField(hcmObj, Status)
	if rootStStr != nil {
		rootSt := util.GetField(rootStStr, Status)
		if rootSt != nil {
			rootValue := reflect.ValueOf(rootSt)
			rV := rootValue.MapIndex(keys[0])
			if rV != reflect.ValueOf(nil) && equality.Semantic.DeepEqual(rV.Interface(), objValue.MapIndex(keys[0]).Interface()) {
				glog.V(7).Infof("Status of handleReconcile Object: %v is included into the root object, key=%s, nothing to update ", metadata.GetSelfLink(), keys[0])
				return nil
			}
		}
	}

	newHcmObj := hcmObj.DeepCopyObject()
	//printStatus(newHcmObj, "newHCMobj")

	pr.mutex.Lock()
	defer pr.mutex.Unlock()

	copyStatus(newHcmObj, obj)

	err = pr.ResourceClient.Update(newHcmObj)
	if err != nil {
		m := newHcmObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
		return fmt.Errorf("error updating %s object: %s", m.GetSelfLink(), err)
	}

	return nil
}

func (pr *Propagator) handlePolicyReconcile(ctx context.Context, obj interface{}) error {
	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(5).Infof("handlePolicyReconcile Object: %v", metadata.GetSelfLink())

	plc := obj.(*policyv1alpha1.Policy)
	cls, err := (*pr.ClusterLister).Clusters(metadata.GetNamespace()).List(labels.Everything())
	if err != nil {
		glog.Errorf("Error: reconcileLoop List clusters in ns %s returned: %s", metadata.GetNamespace(), err.Error())
		return err
	} else if len(cls) == 0 {
		// there is an object without cluster
		glog.Errorf("there is no cluster in %s", metadata.GetNamespace())
		return nil
	} else if len(cls) > 1 {
		// there is more than one cluster found
		glog.Errorf("there is more than one cluster in %s", metadata.GetNamespace())
		return nil
	}
	clusterName := cls[0].Name
	policyPerClusterStatus := map[string]*policyv1alpha1.PolicyStatus{}
	policyPerClusterStatus[metadata.GetName()] = &policyv1alpha1.PolicyStatus{
		ComplianceState: plc.Status.ComplianceState,
	}
	clsOffline := false
	if len(cls[0].Status.Conditions) != 0 {
		condition := cls[0].Status.Conditions[len(cls[0].Status.Conditions)-1]
		if condition.Type != clusterregistryv1alpha1.ClusterOK {
			clsOffline = true
		}
	}
	labels := metadata.GetLabels()
	namespace := labels["parent-namespace"]
	name := labels["parent-policy"]
	if clsOffline {
		key := reflect.ValueOf(clusterName)
		hcmObj, err := (*pr.GenericLister).ByNamespace(namespace).Get(name)
		if err != nil {
			return err
		}
		newHcmObj := hcmObj.DeepCopyObject()
		if removeClusterFromStatus(newHcmObj, key) {
			glog.V(4).Infof("Cluster %v is offline; removing from status of %v", clusterName, metadata.GetName())
			err = pr.ResourceClient.Update(newHcmObj)
			if err != nil {
				return err
			}
		} else {
			glog.V(4).Infof("Cluster %v is offline; already removed from status of %v", clusterName, metadata.GetName())
		}
		return nil
	}
	plc.Status.Status = make(map[string]*policyv1alpha1.CompliancePerClusterStatus)
	plc.Status.Status[clusterName] = &policyv1alpha1.CompliancePerClusterStatus{
		ClusterName:           clusterName,
		AggregatePolicyStatus: policyPerClusterStatus,
		ComplianceState:       plc.Status.ComplianceState,
	}

	objStStr := util.GetField(plc, Status)
	if objStStr == nil {
		glog.V(5).Infof("handleReconcile Object: %v doesn't have status str", metadata.GetSelfLink())
		return nil
	}
	objSt := util.GetField(objStStr, Status)
	if objSt == nil {
		glog.V(5).Infof("handleReconcile Object: %v doesn't have status", metadata.GetSelfLink())
		return nil
	}
	objValue := reflect.ValueOf(objSt)
	keys := objValue.MapKeys()
	if len(keys) == 0 {
		glog.V(5).Infof("Status of handleReconcile Object: %v doesn't have entries", metadata.GetSelfLink())
		return nil
	}

	if labels == nil || labels["parent-namespace"] == "" || labels["parent-policy"] == "" {
		// this is an old policy pre 3.2.1 needs to be removed
		glog.V(5).Infof("Old policy in cluster namespace found. Deleting %s", metadata.GetSelfLink())
		pr.deleteGenericObject(obj)
		return nil
	}

	hcmObj, err := (*pr.GenericLister).ByNamespace(namespace).Get(name)
	if err != nil {
		if k8sErrors.IsNotFound(err) {
			// TODO remove other ?
			glog.Errorf("Error HCM root Object for %s doesn't exist", name)
			return nil
		}
		glog.Errorf("Error get Root Object with name %s from %s namespace returned : %s",
			name, namespace, err.Error())
		return err
	}

	rootStStr := util.GetField(hcmObj, Status)
	if rootStStr != nil {
		rootSt := util.GetField(rootStStr, Status)
		if rootSt != nil {
			rootValue := reflect.ValueOf(rootSt)
			rV := rootValue.MapIndex(keys[0])
			if rV != reflect.ValueOf(nil) && equality.Semantic.DeepEqual(rV.Interface(), objValue.MapIndex(keys[0]).Interface()) {
				glog.V(7).Infof("Status of handlePolicyReconcile Object: %v is included into the root object, key=%s, nothing to update ", metadata.GetSelfLink(), keys[0])
				return nil
			}
		}
	}

	newHcmObj := hcmObj.DeepCopyObject()
	pr.mutex.Lock()
	defer pr.mutex.Unlock()

	copyStatus(newHcmObj, plc)
	err = pr.ResourceClient.Update(newHcmObj)
	if err != nil {
		m := newHcmObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
		pr.Recorder.Event(newHcmObj, "Warning", "PolicyStatusAggregated", fmt.Sprintf("Policy %s status aggregation from %s failed, returned error: %s", plc.GetName(), plc.GetNamespace(), err))
		return fmt.Errorf("error updating %s object: %s", m.GetSelfLink(), err)
	}
	pr.Recorder.Event(newHcmObj, "Normal", "PolicyStatusAggregated", fmt.Sprintf("Policy %s status aggregated from %s", plc.GetName(), plc.GetNamespace()))

	return nil
}

func copyStatus(dest, src interface{}) {
	glog.V(5).Info("copystatus")
	srcStStr := util.GetField(src, Status)
	if srcStStr == nil {
		glog.V(4).Info("copyStatus: src Status is nil")
		return
	}
	srcSt := util.GetField(srcStStr, Status)
	srcValue := reflect.ValueOf(srcSt)

	keys := srcValue.MapKeys()
	if len(keys) == 0 {
		glog.V(4).Info("copyStatus: status has 0 entries")
		return
	}
	outStatus := getField(reflect.ValueOf(dest), Status)
	status := getField(outStatus, Status)
	//destValue := reflect.ValueOf(destSt)
	if status.IsNil() {
		status = reflect.MakeMap(reflect.TypeOf(srcSt))
	}

	//we can have only 1 key
	deepCopy := srcValue.MapIndex(keys[0]).MethodByName("DeepCopy")
	if deepCopy == reflect.ValueOf(nil) {
		//  there is no deep-copy
		glog.V(6).Info("copyStatus: copy without DeepCopy")
		status.SetMapIndex(keys[0], srcValue.MapIndex(keys[0]))
	} else {
		glog.V(6).Info("copyStatus: copy with DeepCopy")
		result := deepCopy.Call([]reflect.Value{})
		status.SetMapIndex(keys[0], result[0])
	}
	v := getField(outStatus, Status)
	if v.IsValid() && v.CanSet() {
		v.Set(status)
	}
	//util.SetField(dest, "Status", destValue.Interface())
	return
}

func removeClusterFromStatus(dest runtime.Object, clKey reflect.Value) bool {
	metadata := dest.(metav1.ObjectMetaAccessor).GetObjectMeta()
	destStStr := util.GetField(dest, Status)
	if destStStr == nil {
		glog.V(4).Infof("removeClusterFromStatus %s main Status is nil", metadata.GetSelfLink())
		return false
	}
	destSt := util.GetField(destStStr, Status)
	destValue := reflect.ValueOf(destSt)
	if destValue == reflect.ValueOf(nil) {
		// nothing to update
		glog.V(4).Infof("removeClusterFromStatus %s Status is nil", metadata.GetSelfLink())
		return false
	}
	if destValue.MapIndex(clKey) != reflect.ValueOf(nil) {
		glog.V(4).Infof("removeClusterFromStatus %s remove key %s", metadata.GetSelfLink(), clKey)
		destValue.SetMapIndex(clKey, reflect.ValueOf(nil))
		return true
	}
	return false
}

func (pr *Propagator) createOrUpdateObject(obj interface{}, clNs string) error {
	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(2).Infof("createOrUpdateObject %s in %s namespace", metadata.GetSelfLink(), clNs)

	rt := obj.(runtime.Object)
	newObj := rt.DeepCopyObject()

	util.SetFieldToZero(newObj, "Status")
	// delete labels from the black list
	if pr.LabelBlackList != nil && len(pr.LabelBlackList) > 0 {
		newMetadata := newObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
		labels := newMetadata.GetLabels()
		for _, label := range pr.LabelBlackList {
			delete(labels, label)
		}
		newMetadata.SetLabels(labels)
	}

	listener := (*pr.GenericLister).ByNamespace(clNs)
	lObj, err := listener.Get(metadata.GetNamespace() + "." + metadata.GetName())

	if err != nil {
		if k8sErrors.IsNotFound(err) {
			// new  Object
			glog.V(5).Infof("Object %s not found in %s namespace, creating replica...", metadata.GetName(), clNs)

			// create replica object in a cluster specific namespace

			newMetadata := newObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
			labels := newMetadata.GetLabels()
			if labels == nil {
				labels = map[string]string{"parent-namespace": newMetadata.GetNamespace(), "parent-policy": newMetadata.GetName()}
			} else {
				labels["parent-namespace"] = newMetadata.GetNamespace()
				labels["parent-policy"] = newMetadata.GetName()
			}
			newMetadata.SetName(newMetadata.GetNamespace() + "." + newMetadata.GetName())
			newMetadata.SetLabels(labels)
			newMetadata.SetNamespace(clNs)
			newMetadata.SetResourceVersion("")

			// only set ownerReferences when object is kind policy
			if newObj.(runtime.Object).GetObjectKind().GroupVersionKind().Kind == "Policy" {
				ownerReferences := []metav1.OwnerReference{
					*metav1.NewControllerRef(newObj.(metav1.Object), schema.GroupVersionKind{
						Group:   policyv1alpha1.SchemeGroupVersion.Group,
						Version: policyv1alpha1.SchemeGroupVersion.Version,
						Kind:    "Policy",
					}),
				}
				newMetadata.SetOwnerReferences(ownerReferences)
			}
			err = pr.ResourceClient.Create(newObj)
			if err != nil {
				glog.Errorf("Error: createOrUpdateObject [name %s, namespace %s] returned error: %s",
					newMetadata.GetName(), newMetadata.GetNamespace(), err)
				pr.Recorder.Event(rt, "Warning", "PolicyPropagated", fmt.Sprintf("Policy %s failed to propagate to %s, returned error: %s", metadata.GetName(), clNs, err))
				return err
			}
			pr.Recorder.Event(rt, "Normal", "PolicyPropagated", fmt.Sprintf("Policy %s propagated to %s", metadata.GetName(), clNs))
		} else {
			glog.Errorf("Error: createOrUpdateObject get Object from Listener: %s", err)
			return err
		}
	} else { // object exists in the cluster namespace. If there is a different set of specs, update the specs
		glog.V(5).Infof("Object %s found in %s cluster, checking if spec needs sync", metadata.GetName(), clNs)
		err = pr.syncSpecs(newObj, lObj)
		if err != nil {
			glog.Errorf("Error: createOrUpdateObject  sync object: %s", err)
			return err
		}
	}
	return nil
}

func (pr *Propagator) handleFinalizer(ctx context.Context, obj interface{}) error {

	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	glog.V(5).Infof("handleFinalizer for %s", metadata.GetSelfLink())

	labels := metadata.GetLabels()

	if labels == nil || labels["parent-namespace"] == "" || labels["parent-policy"] == "" {
		// this is an old policy pre 3.2.1, doing nothing
		return nil
	}
	namespace := labels["parent-namespace"]
	name := labels["parent-policy"]
	hcmObj, err := (*pr.GenericLister).ByNamespace(namespace).Get(name)
	if err != nil {
		if k8sErrors.IsNotFound(err) {
			// the main object might be removed before, so it is not a real error
			glog.V(5).Infof("handleFinalizer Error: Root Object %s doesn't exist", metadata.GetName())
			return nil
		}
		glog.Errorf("handleFinalizer Error: get Root Object %s returned : %s", metadata.GetName(), err.Error())
		return err
	}
	objStStr := util.GetField(obj, Status)
	if objStStr == nil {
		glog.V(5).Infof("handleFinalizer %s Object Status struct is nil", metadata.GetSelfLink())
		return nil
	}
	objSt := util.GetField(objStStr, Status)
	if objSt == nil {
		glog.V(5).Infof("handleFinalizer %s Object Status is nil", metadata.GetSelfLink())
		return nil
	}
	objValue := reflect.ValueOf(objSt)
	keys := objValue.MapKeys()
	if len(keys) == 0 {
		glog.V(5).Infof("handleFinalizer %s Object Status has no entries", metadata.GetSelfLink())
		return nil
	}
	newHcmObj := hcmObj.DeepCopyObject()
	glog.V(6).Infof("handleFinalizer %s going to remove the key %s", metadata.GetSelfLink(), keys[0])
	if removeClusterFromStatus(newHcmObj, keys[0]) {
		err := pr.ResourceClient.Update(newHcmObj)
		if err != nil {
			return err
		}
	}
	return nil
}

func (pr *Propagator) handleHCMFinalizer(ctx context.Context, obj interface{}) error {

	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	name := metadata.GetName()
	labelSelector, _ := labels.Parse(fmt.Sprintf("parent-policy=%s,parent-namespace=%s", name, metadata.GetNamespace()))
	objects, err := (*pr.GenericLister).List(labelSelector)
	if err != nil {
		glog.Errorf("Error handleHCMFinalizer %s List Objects returned: %s", metadata.GetSelfLink(), err)
		return err
	}

	for _, o := range objects {
		m := o.(metav1.ObjectMetaAccessor).GetObjectMeta()
		glog.V(5).Infof("handleHCMFinalizer delete %s %s", m.GetName(), m.GetNamespace())
		// if m.GetName() == name && m.GetNamespace() != pr.HcmNamespace {
		err = pr.deleteGenericObject(o)
		if err != nil {
			return err
		}
		// }
	}
	return nil
}

func (pr *Propagator) handleClusterFinalizer(ctx context.Context, ns string) error {
	objects, err := (*pr.GenericLister).ByNamespace(ns).List(labels.Everything())
	if err != nil {
		return err
	}
	for _, obj := range objects {
		(*pr).removeFinalizers(obj, syncFinalizerNames)
		pr.deleteGenericObject(obj)
	}
	return nil
}

func (pr *Propagator) deleteGenericObject(obj interface{}) error {
	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	name := metadata.GetName()
	namespace := metadata.GetNamespace()
	kind := reflect.TypeOf(obj).Elem().Name()
	glog.V(5).Infof("deleteGenericObject %s", metadata.GetSelfLink())
	err := pr.ResourceClient.DeleteObject(kind, namespace, name, &metav1.DeleteOptions{})
	if err != nil && !k8sErrors.IsNotFound(err) {
		glog.Errorf("Error: deleteGenericObject ( %s, %s, %s) returned : %s",
			kind, namespace, name, err)
		pr.Recorder.Event(obj.(runtime.Object), "Warning", "PolicyDeleted", fmt.Sprintf("Policy %s deletion from %s failed, returned err: %s", name, namespace, err))
		return err
	}
	pr.Recorder.Event(obj.(runtime.Object), "Normal", "PolicyDeleted", fmt.Sprintf("Policy %s deleted from %s", name, namespace))
	return nil
}

// sync the specs from remote to local object
func (pr *Propagator) syncSpecs(origObj, destObj runtime.Object) error {
	orgSpec := util.GetField(origObj, "Spec")
	destSpec := util.GetField(destObj, "Spec")
	if destObj.GetObjectKind().GroupVersionKind().Kind == "Policy" {
		destObj = common.RemoveStatusInSpec(destObj)
		destSpec = util.GetField(destObj, "Spec")
	}

	//check whether annotations need update
	annotationsUpdateNeeded := false
	rAnnotations := destObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetAnnotations()
	if rAnnotations != nil {
		oAnnotations := origObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetAnnotations()
		if !equality.Semantic.DeepEqual(rAnnotations, oAnnotations) {
			glog.V(5).Infof("Annotations of object %v have been modified, updating object...", name)
			annotationsUpdateNeeded = true
		}
	}
	orgMeta := origObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	destLink := destObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetSelfLink()
	if !equality.Semantic.DeepEqual(orgSpec, destSpec) || annotationsUpdateNeeded {
		glog.V(5).Infof("Spec for object %s needs sync. Updating ...", destLink)

		util.SetField(destObj, "Spec", orgSpec)
		destObj.(metav1.ObjectMetaAccessor).GetObjectMeta().SetAnnotations(orgMeta.GetAnnotations())

		err := pr.ResourceClient.Update(destObj)
		if err != nil {
			pr.Recorder.Event(destObj, "Warning", "PolicyUpdated", fmt.Sprintf("Policy %s spec update failed, returned error: %s", destObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetName(), err))
			return fmt.Errorf("Error syncSpecs updating object: %s", err)
		}
		pr.Recorder.Event(destObj, "Normal", "PolicyUpdated", fmt.Sprintf("Policy %s spec updated", destObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetName()))
	} else {
		glog.V(5).Infof("Spec for object %s doesn't need sync.", destLink)
	}
	return nil
}

func (pr *Propagator) removeStaleObjects(name string, namespace string, allClusters []*clreg.Cluster, deployed map[string]*clreg.Cluster) {

	for _, cluster := range allClusters {
		_, OK := deployed[cluster.Name]
		if OK {
			continue
		}
		obj, err := (*pr.GenericLister).ByNamespace(cluster.GetNamespace()).Get(namespace + "." + name)
		if err != nil {
			if k8sErrors.IsNotFound(err) {
				continue
			} else {
				// we just print it
				glog.Errorf("Error: removeStaleObjects  returned: %s", err)
			}
		} else {
			if obj != nil && !common.HasOwnerReferencesOf(obj, "Compliance") { // the policy object maybe created by policy controller in the namespace, so don't delete
				pr.deleteGenericObject(obj)
			}
		}
	}
}

// remove stale statuses from the main MCM object. Returns "true" if the object should be updated
func (pr *Propagator) removeStaleStatuses(obj runtime.Object, allClusters []*clreg.Cluster) bool {

	metadata := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
	objStStr := util.GetField(obj, Status)
	if obj == nil {
		glog.V(6).Infof("removeStaleStatuses from %s main Status is nil", metadata.GetSelfLink())
		return false
	}
	destSt := util.GetField(objStStr, Status)
	destValue := reflect.ValueOf(destSt)
	if destValue == reflect.ValueOf(nil) || destValue.Kind() != reflect.Map {
		// nothing to update
		glog.V(6).Infof("removeStaleStatuses from %s Status is nil", metadata.GetSelfLink())
		return false
	}
	statusKeys := destValue.MapKeys()
	if len(statusKeys) == 0 {
		// nothing to update
		glog.V(6).Infof("removeStaleStatuses from %s the status map is empty", metadata.GetSelfLink())
		return false
	}
	clustersMap := make(map[string]*clreg.Cluster)
	for _, cluster := range allClusters {
		clustersMap[cluster.Name] = cluster
	}
	ret := false
	for _, key := range statusKeys {
		clName := key.String()
		glog.V(6).Infof("removeStaleStatuses from %s key %s", metadata.GetSelfLink(), clName)
		cluster, ok := clustersMap[clName]
		if !ok {
			// cluster doesn't exist
			destValue.SetMapIndex(key, reflect.ValueOf(nil))
			ret = true
		} else {
			_, err := (*pr.GenericLister).ByNamespace(cluster.Namespace).Get(metadata.GetNamespace() + "." + metadata.GetName())
			if err != nil {
				if k8sErrors.IsNotFound(err) {
					glog.V(5).Infof("removeStaleStatuses remove status for %s from %s ", clName, metadata.GetSelfLink())
					destValue.SetMapIndex(key, reflect.ValueOf(nil))
					ret = true
				} else {
					glog.Errorf("Error: removeStaleStatuses from %s, key = %s, get object returned %s", metadata.GetSelfLink(), clName, err)
				}
			}
		}

	}
	return ret
}

// runReconciliationLoop periodically runs a reconciliation task
func (pr *Propagator) runReconciliationLoop(stopCh <-chan struct{}) {
	wait.Until(pr.reconcileLoop, pr.ReconcilePeriod, stopCh)
}

// reconcile loop
func (pr *Propagator) reconcileLoop() {
	glog.V(5).Infof("START reconcileLoop deployedMap: %v", pr.deployedMap)

	//[deployedObject][clusterName]*Cluster
	destMap := make(map[interface{}]map[string]*clreg.Cluster)

	// objects without clusters
	orphans := make([]runtime.Object, 0)
	updateObjects := make([]runtime.Object, 0)

	pr.mutex.Lock()
	allClusters, err := (*pr.ClusterLister).List(labels.Everything())
	if err != nil {
		glog.Errorf("Error: reconcileLoop error List all clusters returned: %s", err.Error())
		pr.mutex.Unlock()
		return

	}

	i := 0
	for _, cluster := range allClusters {
		if len(cluster.Status.Conditions) != 0 {
			allClusters[i] = cluster
			i++
		}
	}
	allClusters = allClusters[:i]

	for deployedName, ppMap := range pr.deployedMap {
		policyName := strings.Split(deployedName, "/")[1]
		policyNamespace := strings.Split(deployedName, "/")[0]
		glog.V(6).Infof("reconcileLoop: deployedName %s, ppolicies %v ", deployedName, ppMap)

		var deployObj runtime.Object
		deployObj, err = (*pr.GenericLister).ByNamespace(policyNamespace).Get(policyName)
		if err != nil {
			if k8sErrors.IsNotFound(err) {
				glog.V(5).Infof("reconcileLoop: deployed object %s was not found in %s ns",
					policyName, policyNamespace)
				delete(pr.deployedMap, deployedName)
				continue
			}
			glog.Warningf("reconcileLoop: getDeployedObject: from ns %s name %s returned error: %s",
				policyNamespace, policyName, err.Error())
			continue
		}
		statusPP := make([]string, 0)
		statusPB := make([]string, 0)
		// Several PlacementPolicies can point to the same clusters
		destClusters := make(map[string]*clreg.Cluster) //[clusterName]*Cluster
		policies := make(map[string]struct{})

		if ppMap == nil || len(ppMap) == 0 {
			glog.V(5).Infof("reconcileLoop: deployedName %s, there is no PlacementPolicies", deployedName)
			destMap[deployObj] = emptyMap
			delete(pr.deployedMap, deployedName)
		} else {
			for pb, pp := range ppMap {
				//validate renamed objects
				var binding *placement.PlacementBinding
				binding, err = (*pr.PBLister).PlacementBindings(policyNamespace).Get(pb)
				if err != nil {
					glog.Errorf("Error: reconcileLoop: get PlacementBinding  named %s from %s ns returned %s",
						pb, policyNamespace, err.Error())
					if k8sErrors.IsNotFound(err) {
						delete(ppMap, pb)
					}
					continue
				}
				exsist := false
				for _, subj := range binding.Subjects {
					if subj.Name == policyName && subj.Kind == deployObj.GetObjectKind().GroupVersionKind().Kind {
						exsist = true
						break
					}
				}
				if !exsist {
					// renamed object
					glog.V(5).Infof("reconcileLoop: deployedObject %s: was removed from %s PlacmentBindings",
						deployedName, pb)
					destMap[deployObj] = emptyMap
					delete(ppMap, pb)
					continue
				}
				statusPB = append(statusPB, pb)

				_, ok := policies[pp]
				if ok {
					// the PlacementPolicy has been checked
					continue
				}
				var placementObj *placementrule.PlacementRule
				placementObj, err = (*pr.PRLister).PlacementRules(policyNamespace).Get(pp)
				if err != nil {
					if k8sErrors.IsNotFound(err) {
						glog.V(5).Infof("reconcileLoop: deployedObject %s: PlacementPolicy object %s was not found in %s ns",
							deployedName, pp, policyNamespace)
						//  inexistent PlacementPolicy object will be remove from the table, during reconciliation of PolicyBindings
					} else {
						glog.Errorf("Error: reconcileLoop: deployedObject %s: get PlacementPolicy object %s from %s ns returned: %s",
							deployedName, pp, policyNamespace, err.Error())
					}
					continue
				}
				policies[pp] = struct{}{}
				statusPP = append(statusPP, pp)
				ppDecisions := placementObj.Status.Decisions
				for _, decision := range ppDecisions {
					cluster, err := (*pr.ClusterLister).Clusters(decision.ClusterNamespace).Get(decision.ClusterName)
					if err != nil {
						glog.Errorf("Error: reconcileLoop: failed to retrieve cluster %s in namespace %s",
							decision.ClusterName, decision.ClusterNamespace)
						continue
					}
					destClusters[cluster.Name] = cluster
				}

				// var labelSelector labels.Selector

				// clLabels := placementObj.Spec.ClusterLabels
				// clNames := placementObj.Spec.ClusterNames
				// clCond := placementObj.Spec.ClusterConditions
				// if clLabels != nil {
				// 	labelSelector, err = common.ConvertLabels(clLabels)
				// } else {
				// 	labelSelector = labels.Everything()
				// }

				// var clusters []*clreg.Cluster
				// clusters, err = (*pr.ClusterLister).List(labelSelector)
				// if err != nil {
				// 	glog.Errorf("Error: reconcileLoop: deployedObject %s: PlacementPolicy %s list clusters returned %s",
				// 		deployedName, placementObj.Name, err.Error())
				// 	continue
				// }
				// i := 0
				// for _, cluster := range clusters {
				// 	if len(cluster.Status.Conditions) != 0 {
				// 		clusters[i] = cluster
				// 		i++
				// 	}
				// }
				// clusters = clusters[:i]
				// if len(clusters) == 0 {
				// 	glog.V(5).Infof("reconcileLoop: deployedObject %s: PlacementPolicy %s: cluster labels match did not find any cluster",
				// 		deployedName, placementObj.Name)
				// 	continue
				// }
				// for _, cluster := range clusters {
				// 	if common.IsMached2(cluster, &clNames, &clCond) {
				// 		destClusters[cluster.Name] = cluster
				// 	}
				// }
			}
		}

		//make policy propagate to no clusters if disabled
		disabled := util.GetField(deployObj, Spec).(v1alpha1.PolicySpec).Disabled
		if disabled {
			destClusters = emptyMap
			glog.Infof("Policy %s is disabled, setting propagated clusters to empty list...", deployObj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetName())
		}

		//objSt := util.GetField(deployObj, Status)
		newHcmObj := deployObj.DeepCopyObject()
		v := reflect.ValueOf(newHcmObj)
		v = getField(v, Status)
		if !v.IsValid() {
			metadata := newHcmObj.(metav1.ObjectMetaAccessor).GetObjectMeta()
			glog.Errorf("Error: reconcileLoop Object: %s has invalid Status field", metadata.GetSelfLink())
			// TODO we should not be here
		} else {
			shouldUpdate := false
			sort.Strings(statusPP)
			v1 := getField(v, PlacementPolicies)
			if !reflect.DeepEqual(v1.Interface(), statusPP) {
				glog.V(6).Infof("reconcileLoop: PlacementPolicies should be updated status.PP =%v newPP=%v", v1, statusPP)
				if v1.IsValid() && v1.CanSet() {
					v1.Set(reflect.ValueOf(statusPP))
					shouldUpdate = true
				} else {
					glog.Errorf("Error: reconcileLoop cannot set placement policy list")
				}
			}
			v1 = getField(v, PlacementBindings)
			sort.Strings(statusPB)
			if !reflect.DeepEqual(v1.Interface(), statusPB) {
				glog.V(6).Infof("reconcileLoop: PlacementBindings should be updated status.PP=%v newPB=%v", v1, statusPB)
				if v1.IsValid() && v1.CanSet() {
					v1.Set(reflect.ValueOf(statusPB))
					shouldUpdate = true
				} else {
					glog.Errorf("Error: reconcileLoop cannot set placement bindings list")
				}
			}
			shouldUpdate = pr.removeStaleStatuses(newHcmObj, allClusters) || shouldUpdate

			if shouldUpdate {
				updateObjects = append(updateObjects, newHcmObj)
			}
		}

		if len(destClusters) == 0 {
			glog.V(5).Infof("reconcileLoop; deployed object %s: doesn't have any cluster destination",
				deployedName)
			destMap[deployObj] = emptyMap
		} else {
			glog.V(5).Infof("reconcileLoop: deploy object %s to %v", deployedName, destClusters)
			//			// deploy or delete
			destMap[deployObj] = destClusters
		}
	}
	// remove stale deployed objects which are not in the pr.deployedMap
	var allObjs []runtime.Object
	allObjs, err = (*pr.GenericLister).List(labels.Everything())
	if err != nil {
		glog.Errorf("Error: reconcileLoop  List all objects returned: %s", err.Error())
	} else {
		for _, obj := range allObjs {
			meta := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()

			// we have to remove objects without clusters
			ns := meta.GetNamespace()
			if !common.IsInClusterNamespace(ns, allClusters) {
				_, ok := pr.deployedMap[meta.GetNamespace()+"/"+meta.GetName()]
				if !ok {
					destMap[obj] = emptyMap
				}
			} else {
				cls, errL := (*pr.ClusterLister).Clusters(ns).List(labels.Everything())
				if errL != nil {
					glog.Errorf("Error: reconcileLoop List clusters in ns %s returned: %s", ns, err.Error())
				} else if len(cls) == 0 {
					// there is an object without cluster
					orphans = append(orphans, obj)
				}
			}
		}
	}
	pr.mutex.Unlock()
	// Now, after releasing the mutex, we can update statuses of deployed objects
	for _, obj := range updateObjects {
		err = pr.ResourceClient.Update(obj)
		if err != nil {
			m := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
			glog.Errorf("Error: reconcileLoop updating %s object: %s", m.GetSelfLink(), err)
		}
	}

	// update the destination
	for obj, destClusters := range destMap {
		meta := obj.(metav1.ObjectMetaAccessor).GetObjectMeta()
		if len(destClusters) == 0 {
			pr.removeStaleObjects(meta.GetName(), meta.GetNamespace(), allClusters, emptyMap)
		} else {
			for _, cl := range destClusters {
				err = pr.createOrUpdateObject(obj, cl.Namespace)
				if err != nil {
					glog.Errorf("Error: reconcileLoop createOrUpdateObject %v in %v returned %s", obj, cl.Namespace, err.Error())
				}
			}
			pr.removeStaleObjects(meta.GetName(), meta.GetNamespace(), allClusters, destClusters)
		}
	}
	// remove orphans
	for _, obj := range orphans {
		(*pr).removeFinalizers(obj, syncFinalizerNames)
		pr.deleteGenericObject(obj)
	}

	glog.V(5).Infof("END reconcileLoop deployedMap: %v", pr.deployedMap)
}

// getField gets a field in a struct using reflect
func getField(v reflect.Value, field string) reflect.Value {
	if v.Type().Kind() == reflect.Ptr {
		v = v.Elem()
	}
	return v.FieldByName(field)

}

func (pr *Propagator) removeFinalizers(obj runtime.Object, finalizers []string) {
	glog.V(6).Infof("removeFinalizers from %s finalizers %v", obj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetSelfLink(), finalizers)
	update := false
	for _, finName := range finalizers {
		if resv1.HasFinalizer(obj, finName) {
			resv1.RemoveFinalizer(obj, finName)
			update = true
		}
	}
	if update {
		err := pr.ResourceClient.Update(obj)
		if err != nil {
			glog.Errorf("removeFinalizers: update %s returned %s", obj.(metav1.ObjectMetaAccessor).GetObjectMeta().GetSelfLink(), err.Error())
		}
	}
}
