// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.

package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/golang/glog"
	policyv1alpha1 "github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"

	"github.com/open-cluster-management/hcm-compliance/pkg/common"
	"github.com/open-cluster-management/hcm-compliance/pkg/compliance"
	"github.com/open-cluster-management/seed-sdk/pkg/controller"

	compApis "github.com/open-cluster-management/hcm-compliance/pkg/apis/compliance/v1alpha1"

	"github.com/open-cluster-management/seed-sdk/pkg/client"
	"github.com/open-cluster-management/seed-sdk/pkg/scheme"
	"k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/fields"
	"k8s.io/client-go/tools/cache"
)

func main() {

	resyncFrequency := flag.Duration("resyncfrequency", time.Second*200, "time in seconds to reconcile, default is 200 seconds. example: \"30s\", or \"2h45m\". Valid time units are ms, s, m, h")
	namespace := flag.String("watch-ns", "", "Watched Kubernetes namespace")
	clusterName := flag.String("cluster-name", "mcm-managed-cluster", "Name of the cluster")
	kubeconfig := flag.String("kubeconfig", "", "Path to a kube config. Only required if out-of-cluster.")
	reconcileFrequency := flag.Duration("reconcilefrequency", time.Second*3, "time in seconds to reconcile, default is 30 seconds. example: \"30s\", or \"2h45m\". Valid time units are ms, s, m, h")

	flag.Parse()

	defer glog.Flush()

	if *namespace == "" {
		*namespace = "default"
	}
	// this is done for back-compatibility, but in general we need to provide the cluster name
	if *clusterName == "" {
		*clusterName = *namespace
	}
	glog.V(3).Infof("Starting compliance-contoller, it's watching `%s` namespace on cluster `%s`", *namespace, *clusterName)

	c := make(chan os.Signal)
	signal.Notify(c, os.Interrupt, syscall.SIGTERM)
	go func() {
		<-c
		cleanup()
	}()

	// build config for local cluster
	config, err := common.BuildConfig(kubeconfig)
	if err != nil {
		glog.Fatalf("cannot create kube config sucessfully: %v", err)
	}
	// create a compliance client
	gscheme := scheme.NewScheme()
	compApis.AddToScheme(gscheme.Scheme)
	gscheme.SetPlural(compApis.SchemeGroupVersion.WithKind("Compliance"), compApis.ComplianceResourcePlural)
	compliance.CplClient, err = client.New(config, compApis.SchemeGroupVersion, gscheme)
	if err != nil {
		panic(err)
	}
	// create a policy client
	scheme := scheme.NewScheme()
	policyv1alpha1.AddToScheme(scheme.Scheme)
	scheme.SetPlural(policyv1alpha1.SchemeGroupVersion.WithKind("Policy"), policyv1alpha1.PolicyResourcePlural)
	compliance.PlcClient, err = client.New(config, policyv1alpha1.SchemeGroupVersion, scheme)
	if err != nil {
		panic(err)
	}

	// Create Compliance Lister
	ctx, cancelFunc := context.WithCancel(context.Background())
	defer cancelFunc()
	/*
		inx, cntr := cache.NewIndexerInformer(
			cache.NewListWatchFromClient(compliance.CplClient.RESTClient, "compliances", v1.NamespaceAll, fields.Everything()),
			&compApis.Compliance{},
			*reconcileFrequency,
			cache.ResourceEventHandlerFuncs{},
			cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
		)
	*/
	inx, cntr := cache.NewIndexerInformer(
		cache.NewListWatchFromClient(compliance.PlcClient.RESTClient, "policies", v1.NamespaceAll, fields.Everything()),
		&policyv1alpha1.Policy{},
		*resyncFrequency,
		cache.ResourceEventHandlerFuncs{},
		cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
	)
	//cl := cache.NewGenericLister(inx, compApis.Resource("compliances"))
	cl := cache.NewGenericLister(inx, policyv1alpha1.Resource("policies"))
	gl := cl.(cache.GenericLister)
	compliance.GenericLister = gl
	go cntr.Run(ctx.Done())

	//compliance.PlcLister = plcLister

	compliance.Initialize(*namespace, *clusterName)
	/*
		// Create Compliance Lister
		inx, _ := cache.NewIndexerInformer(
			cache.NewListWatchFromClient(compliance.CplClient.RESTClient, "compliances", v1.NamespaceAll, fields.Everything()),
			&compApis.Compliance{},
			*reconcileFrequency,
			cache.ResourceEventHandlerFuncs{},
			cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
		)
		//cplLister := complianceLister.NewComplianceLister(inx)

		// Create Policy Lister

		indx, _ := cache.NewIndexerInformer(
			cache.NewListWatchFromClient(compliance.PlcClient.RESTClient, "policies", v1.NamespaceAll, fields.Everything()),
			&policyv1alpha1.Policy{},
			*reconcileFrequency,
			cache.ResourceEventHandlerFuncs{},
			cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
		)
		plcLister := policyLister.NewPolicyLister(indx)

		compliance.PlcLister = plcLister
	*/

	go compliance.ReconcileCompliance(*reconcileFrequency, *namespace)

	// Configure  compliance controller
	err = controller.New().
		ParseFlags().
		Watch("compliances.v1alpha1.compliance.mcm.ibm.com", "Compliance", &compApis.SchemeBuilder).
		From(config).
		Reconcile(compliance.Reconcile).
		Finalize(compliance.Finalizer).
		ResyncPeriod(*resyncFrequency).
		Install(compApis.CustomResourceDefinitions).
		Install(policyv1alpha1.CustomResourceDefinitions).
		Run()
	if err != nil {
		panic(err)
	}
}

// some clean up logic can be added here
func cleanup() {
	fmt.Println("cleaning up...")
	os.Exit(0)
}
