// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.
// Copyright (c) 2020 Red Hat, Inc.

package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/golang/glog"
	"github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"

	//
	//informers "github.com/open-cluster-management/hcm-compliance/pkg/client/informers/externalversions/policy/v1alpha1" //
	//
	policyv1alpha1 "github.com/open-cluster-management/hcm-compliance/pkg/apis/policy/v1alpha1"
	policyLister "github.com/open-cluster-management/hcm-compliance/pkg/client/listers/policy/v1alpha1"
	"github.com/open-cluster-management/hcm-compliance/pkg/common"
	"github.com/open-cluster-management/hcm-compliance/pkg/policy"
	resourceClient "github.com/open-cluster-management/seed-sdk/pkg/client"
	seedClient "github.com/open-cluster-management/seed-sdk/pkg/client"
	"github.com/open-cluster-management/seed-sdk/pkg/controller"
	"github.com/open-cluster-management/seed-sdk/pkg/scheme"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/fields"
	"k8s.io/client-go/kubernetes"
	eventLister "k8s.io/client-go/listers/core/v1"
	"k8s.io/client-go/tools/cache"
)

func main() {
	var err error
	reconcileFrequency := flag.Duration("reconcilefrequency", time.Second*3, "time in seconds to reconcile, default is 30 seconds. example: \"30s\", or \"2h45m\". Valid time units are ms, s, m, h")
	kubeconfig := flag.String("kubeconfig", "", "Path to a kube config. Only required if out-of-cluster.")
	clusterName := flag.String("cluster-name", "mcm-managed-cluster", "Name of the cluster")
	namespace := flag.String("watch-ns", "", "Watched Kubernetes namespace")

	flag.Parse()

	defer glog.Flush()

	glog.V(3).Infof("Started with cluster name: %v namespace: %v", *clusterName, *namespace)

	c := make(chan os.Signal)
	signal.Notify(c, os.Interrupt, syscall.SIGTERM)
	go func() {
		<-c
		cleanup()
	}()

	//create a kube client  kubernetes.Clientset

	initiatlized := make(chan int)

	config, err := common.BuildConfig(kubeconfig)
	if err != nil {
		glog.Fatalf("cannot create kube config sucessfully: %v", err)
	}

	client, err := kubernetes.NewForConfig(config)
	if err != nil {
		glog.Fatalf("cannot create kube client sucessfully: %v", err)
	}

	//fmt.Println("freq flag = %v", *reconcileFrequency)
	plcScheme := scheme.NewScheme()

	plcScheme.SetPlural(v1alpha1.SchemeGroupVersion.WithKind("Policy"), v1alpha1.PolicyResourcePlural)

	ResClient, err := resourceClient.New(config, v1alpha1.SchemeGroupVersion, plcScheme)
	if err != nil {
		glog.Fatalf("cannot create seed resource client sucessfully: %v", err)
	}

	//adding a policy lister
	policyv1alpha1.AddToScheme(plcScheme.Scheme)
	plcScheme.SetPlural(policyv1alpha1.SchemeGroupVersion.WithKind("Policy"), policyv1alpha1.PolicyResourcePlural)
	plcClient, err := seedClient.New(config, policyv1alpha1.SchemeGroupVersion, plcScheme)
	if err != nil {
		panic(err)
	}

	ctx, cancelFunc := context.WithCancel(context.Background())
	defer cancelFunc()

	inx, cntr := cache.NewIndexerInformer(
		cache.NewListWatchFromClient(plcClient.RESTClient, "policies", *namespace, fields.Everything()),
		&policyv1alpha1.Policy{},
		time.Second*30,
		cache.ResourceEventHandlerFuncs{},
		cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
	)
	go cntr.Run(ctx.Done())
	plcLister := policyLister.NewPolicyLister(inx)

	// adding Events Lister ================
	eventCtx, eventCancelFunc := context.WithCancel(context.Background())
	defer eventCancelFunc()

	eventIndexer, eventController := cache.NewIndexerInformer(
		cache.NewListWatchFromClient(client.CoreV1().RESTClient(), "events", *namespace, fields.Everything()),
		&corev1.Event{},
		time.Second*30,
		cache.ResourceEventHandlerFuncs{},
		cache.Indexers{cache.NamespaceIndex: cache.MetaNamespaceIndexFunc},
	)
	eventLister := eventLister.NewEventLister(eventIndexer)
	go eventController.Run(eventCtx.Done())

	policy.Initialize(config, client, ResClient, initiatlized, plcLister, *clusterName, eventLister)

	go policy.ReconcileResources(*client, *reconcileFrequency)

	// Configure policy controller
	err = controller.New().
		ParseFlags().
		Watch("policies.v1alpha1.policy.mcm.ibm.com", "Policy", &v1alpha1.SchemeBuilder).
		Namespace(*namespace).
		//	From(config). // seed-sdk issue #180
		Reconcile(policy.Reconcile).
		Finalize(policy.Finalizer).
		Install(v1alpha1.CustomResourceDefinitions).
		Run()
	if err != nil {
		fmt.Print(err)
	}
}

// some clean up logic can be added here
func cleanup() {
	fmt.Println("cleaning up...")
	os.Exit(0)
}
