// Licensed Materials - Property of IBM
// (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
// Note to U.S. Government Users Restricted Rights:
// Use, duplication or disclosure restricted by GSA ADP Schedule
// Contract with IBM Corp.
package main

import (
	"context"
	"flag"
	"fmt"
	"time"

	"github.com/golang/glog"
	"github.com/open-cluster-management/seed-sdk/pkg/controller"
	v1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	meta_v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/client-go/rest"

	"github.com/open-cluster-management/hcm-compliance/pkg/common"
	"github.com/open-cluster-management/hcm-compliance/pkg/sync"
	"k8s.io/client-go/kubernetes"
)

func main() {
	kubeconfig := flag.String("kubeconfig", "", "Path to a kube config. Only required if out-of-cluster")
	remoteKubeconfig := flag.String("remote-kubeconfig", "", "Path to remote kube config")
	watchNs := flag.String("watch-ns", "", "MCM namespace to watch")
	klusterletNs := flag.String("klusterlet-ns", "", "The klusterlet namespace")
	resyncPeriod := flag.Duration("resync", 2*time.Minute, "full object listing frequency")
	syncAccessPolicies := flag.Bool("access-policy", false, "synchronize AccessPolicy instance")
	certSecretName := flag.String("cert-secret-name", "", "Name of secret with client certificate")
	flag.Parse()

	if *klusterletNs == "" || *watchNs == "" {
		panic(fmt.Errorf("missing parameter: klusterlet-ns = \"%s\", watch-ns = \"%s\", cert-secret-name = \"%s\"",
			*klusterletNs, *watchNs, *certSecretName))
	}
	glog.V(3).Infof("Starting sync controller")
	defer glog.Flush()

	// build config for local cluster
	lConfig, err := common.BuildConfig(kubeconfig)
	if err != nil {
		panic(err)
	}

	// build config for remote cluster
	rConfig, err := sync.BuildRemoteConfig(*remoteKubeconfig, *certSecretName, *klusterletNs, lConfig)
	if err != nil {
		panic(err)
	}

	glog.V(3).Infof("Remote config: %#v", rConfig)

	if checkOrCreateNamespace(lConfig, *watchNs) != nil {
		panic(err)
	}

	// create cancel function to run remote controller
	ctx, cancelFn := context.WithCancel(context.Background())
	defer cancelFn()

	// glog.V(3).Infof("Starting watcher for local cluster at %s on namespace %s", lConfig.Host, *watchNs)
	// complianceSync, err := sync.NewComplianceSyncher(rConfig, lConfig)
	// if err != nil {
	// 	panic(err)
	// }

	glog.V(3).Infof("Starting watcher for local cluster at %s on namespace %s", lConfig.Host, *watchNs)
	policySync, err := sync.NewPolicySyncher(rConfig, lConfig)
	if err != nil {
		panic(err)
	}

	var accesspolicySync *sync.AccessPolicySyncher
	if *syncAccessPolicies {
		accesspolicySync, err = sync.NewAccessPolicySyncher(rConfig, lConfig)
		if err != nil {
			panic(err)
		}
	}

	// @todo provide namespace and resync as parameters to Watch* functions?
	builder := controller.New().ParseFlags()
	// builder = complianceSync.WatchStatusEvents(builder).
	// 	ResyncPeriod(*resyncPeriod).
	// 	Namespace(*watchNs)
	builder = policySync.WatchStatusEvents(builder).
		ResyncPeriod(*resyncPeriod).
		Namespace(*watchNs)
	if *syncAccessPolicies {
		builder = accesspolicySync.WatchStatusEvents(builder).
			ResyncPeriod(*resyncPeriod).
			Namespace(*watchNs)
	}
	if err = builder.RunWithContext(ctx); err != nil {
		panic(err)
	}

	// workaround for issue https://github.com/open-cluster-management/seed-sdk/issues/180
	if *certSecretName != "" {
		flag.Set("kubeconfig", sync.KubeconfigForCerts)
	} else {
		flag.Set("kubeconfig", *remoteKubeconfig)
	}
	glog.V(3).Infof("Starting watcher for remote cluster at %s on namespace %s", rConfig.Host, *watchNs)
	builder = controller.New().ParseFlags()
	// builder = complianceSync.WatchSpecEvents(builder).
	// 	ResyncPeriod(*resyncPeriod).
	// 	Namespace(*watchNs)
	builder = policySync.WatchSpecEvents(builder).
		ResyncPeriod(*resyncPeriod).
		Namespace(*watchNs)
	if *syncAccessPolicies {
		builder = accesspolicySync.WatchSpecEvents(builder).
			ResyncPeriod(*resyncPeriod).
			Namespace(*watchNs)
	}
	err = builder.Run()
	if err != nil {
		panic(err)
	}
}

func checkOrCreateNamespace(cfg *rest.Config, name string) error {

	clientset := kubernetes.NewForConfigOrDie(cfg)

	_, err := clientset.CoreV1().Namespaces().Get(name, meta_v1.GetOptions{})

	if errors.IsNotFound(err) {
		ns := &v1.Namespace{}
		ns.Name = name
		_, err = clientset.CoreV1().Namespaces().Create(ns)
		if err == nil || errors.IsAlreadyExists(err) {
			return nil
		}
	}
	if err != nil {
		glog.Errorf("Error: checkOrCreateNamespace namespace %s returned: %s", name, err.Error())
	}
	return err
}
