"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const debug = debug_1.default('core/command-tree');
debug('loading');
const command_1 = require("../models/command");
const events_1 = require("./events");
const usage_error_1 = require("./usage-error");
const oops_1 = require("./oops");
const plugins_1 = require("../plugins/plugins");
const tree_1 = require("../commands/tree");
const context_1 = require("../commands/context");
debug('finished loading modules');
let resolver;
exports.setPluginResolver = (_) => {
    resolver = _;
};
const exactlyTheSameRoute = (route, path) => {
    const routeAsPath = route.split('/').slice(1);
    for (let idx = 0; idx < routeAsPath.length; idx++) {
        if (routeAsPath[idx] !== path[idx]) {
            return false;
        }
    }
    if (routeAsPath.length !== path.length) {
        return false;
    }
    else {
        return true;
    }
};
const treeMatch = (model, path, readonly = false, hide = false, idxStart = 0, noWildcard = false) => {
    let parent = model;
    let cur;
    for (let idx = idxStart; idx < path.length; idx++) {
        cur = (parent.children && parent.children[path[idx]]);
        if (!cur) {
            if (readonly) {
                break;
            }
            else {
                if (!parent.children) {
                    parent.children = {};
                }
                cur = parent.children[path[idx]] = {
                    $: undefined,
                    parent: parent,
                    key: path[idx],
                    options: { hide: hide },
                    route: `${parent.route === '/' ? '' : parent.route}/${path[idx]}`
                };
            }
        }
        else {
        }
        parent = cur;
        cur = (cur.children && cur.children[path[idx]]);
    }
    if (!cur && !noWildcard) {
        cur = parent;
    }
    if (cur.options && cur.options.noArgs && !exactlyTheSameRoute(cur.route, path)) {
    }
    else {
        return cur;
    }
};
const match = (path, readonly) => {
    return treeMatch(tree_1.getModelInternal().root, path, readonly);
};
class DefaultCommandOptions {
}
const _listen = (model, route, handler, options = new DefaultCommandOptions()) => {
    const path = route.split('/').splice(1);
    const leaf = treeMatch(tree_1.getModelInternal().root, path, false, options.hide);
    if (leaf) {
        const prevOptions = leaf.options;
        if (options) {
            leaf.options = options;
        }
        if (leaf.$) {
            if (!leaf.options)
                leaf.options = {};
            if (prevOptions) {
                Object.assign(leaf.options, prevOptions);
            }
            leaf.options.override = leaf.$;
        }
        leaf.$ = handler;
        leaf.route = route;
        return leaf;
    }
};
const listen = (route, handler, options) => _listen(tree_1.getModelInternal().root, route, handler, options);
const _subtree = (route, options) => {
    const myListen = options.listen || listen;
    const path = route.split('/').splice(1);
    const leaf = match(path, false);
    if (leaf) {
        leaf.route = route;
        if (options) {
            leaf.options = options;
        }
        else {
            leaf.options = {};
        }
        leaf.options.isDirectory = true;
        const help = () => {
            const usage = options.usage || (options.synonymFor && options.synonymFor.options && options.synonymFor.options.usage);
            throw new usage_error_1.UsageError({ usage });
        };
        const opts = {
            noArgs: true,
            subtreeHandler: true,
            noAuthOk: true,
            inBrowserOk: true,
            requiresFullyQualifiedRoute: true
        };
        myListen(route, help, Object.assign({}, options, opts));
        myListen(`${route}/help`, help, Object.assign({}, options, opts));
        return leaf;
    }
};
const _subtreeSynonym = (route, master, options = master.options) => {
    if (route !== master.route) {
        const mySubtree = _subtree(route, Object.assign({}, options, { synonymFor: master }));
        if (!master.synonyms)
            master.synonyms = {};
        master.synonyms[mySubtree.route] = mySubtree;
    }
};
const _synonym = (route, handler, master, options = master.options) => {
    if (route !== master.route) {
        const node = listen(route, handler, Object.assign({}, options, { synonymFor: master }));
        if (!master.synonyms)
            master.synonyms = {};
        master.synonyms[node.route] = node;
    }
};
const commandNotFoundMessage = 'Command not found';
const commandNotFoundMessageWithPartialMatches = 'The following commands are partial matches for your request.';
const formatPartialMatches = (partialMatches, hide) => {
    return new usage_error_1.UsageError({
        message: hide ? '' : commandNotFoundMessage,
        usage: {
            header: commandNotFoundMessageWithPartialMatches,
            available: partialMatches.map(_ => _.usage)
        }
    }, { noBreadcrumb: true, noHide: true });
};
const suggestPartialMatches = (command, partialMatches, noThrow = false, hide = false) => {
    const availablePartials = partialMatches && partialMatches.filter(_ => _.usage);
    const anyPartials = availablePartials && availablePartials.length > 0;
    const error = anyPartials
        ? formatPartialMatches(availablePartials, hide)
        : new Error(`${commandNotFoundMessage}: ${command}`);
    error.code = 404;
    if (anyPartials) {
        error.partialMatches = availablePartials.map(_ => ({
            command: _.route
                .split('/')
                .slice(1)
                .join(' '),
            usage: _.usage
        }));
    }
    else {
        error.hide = hide;
    }
    if (noThrow) {
        return error;
    }
    else {
        throw error;
    }
};
const withEvents = (evaluator, leaf, partialMatches) => {
    const event = {};
    if (leaf) {
        event.route = leaf.route;
        event.plugin = leaf.options.plugin || 'builtin';
    }
    const handler = {
        subtree: leaf,
        route: leaf.route,
        eval: evaluator,
        options: leaf && leaf.options,
        success: ({ tab, type, command, isDrilldown = false, parsedOptions }) => {
            event.tab = tab;
            event.execType = type;
            event.command = command;
            event.isDrilldown = isDrilldown;
            if (parsedOptions && leaf.options && leaf.options.okOptions) {
                const opts = leaf.options.okOptions.filter(_ => parsedOptions[_]);
                if (opts) {
                    event.options = opts;
                }
            }
            if (leaf && events_1.default) {
                events_1.default.emit('/command/complete', event);
                if (event.execType !== command_1.ExecType.Nested) {
                    events_1.default.emit('/command/complete/fromuser', tab);
                }
            }
        },
        error: (command, tab, execType, err) => {
            event.tab = tab;
            event.execType = execType;
            event.command = command;
            event.error = oops_1.oopsMessage(err);
            if (leaf && events_1.default) {
                events_1.default.emit('/command/complete', event);
                if (event.execType !== command_1.ExecType.Nested) {
                    events_1.default.emit('/command/complete/fromuser', tab);
                }
            }
            if (err.code === 127 && partialMatches) {
                const suggestions = suggestPartialMatches(command, partialMatches, true, err.hide);
                return suggestions;
            }
            return err;
        }
    };
    return handler;
};
const _read = (model, argv, contextRetry, originalArgv, tryCatchalls) => __awaiter(void 0, void 0, void 0, function* () {
    let leaf = treeMatch(tree_1.getModelInternal().root, argv, true);
    let evaluator = (leaf && leaf.$);
    if (!evaluator) {
        const route = `/${argv.join('/')}`;
        yield resolver.resolve(route, { tryCatchalls: tryCatchalls && (!contextRetry || contextRetry.length === 0) });
        leaf = treeMatch(tree_1.getModelInternal().root, argv, true);
        evaluator = (leaf && leaf.$);
    }
    if (!evaluator) {
        if (!contextRetry) {
            return false;
        }
        else if (contextRetry.length === 0) {
            return _read(tree_1.getModelInternal().root, originalArgv, undefined, originalArgv, tryCatchalls);
        }
        else if (contextRetry.length > 0 &&
            contextRetry[contextRetry.length - 1] !== originalArgv[originalArgv.length - 1]) {
            const maybeInContextRetry = _read(tree_1.getModelInternal().root, contextRetry.concat(originalArgv), contextRetry.slice(0, contextRetry.length - 1), originalArgv, tryCatchalls);
            if (maybeInContextRetry) {
                return maybeInContextRetry;
            }
            const newContext = context_1.getCurrentContext()
                .concat(originalArgv)
                .filter((elt, idx, A) => elt !== A[idx - 1]);
            const maybeInDefaultContext = _read(tree_1.getModelInternal().root, newContext, contextRetry.slice(0, contextRetry.length - 1), originalArgv, tryCatchalls);
            return maybeInDefaultContext;
        }
        else {
            return false;
        }
    }
    else {
        if (leaf.options && leaf.options.requiresFullyQualifiedRoute) {
            const routeWithoutContext = `/${originalArgv.join('/')}`;
            if (leaf.route !== routeWithoutContext) {
                if (argv.length === originalArgv.length && argv.every((elt, idx) => elt === originalArgv[idx])) {
                    return false;
                }
                else {
                    return _read(tree_1.getModelInternal().root, originalArgv, undefined, originalArgv, tryCatchalls);
                }
            }
        }
        return withEvents(evaluator, leaf);
    }
});
const internalRead = (model, argv, tryCatchalls) => {
    if (argv[0] === 'kui')
        argv.shift();
    return _read(tree_1.getModelInternal().root, argv, context_1.Context.current, argv, tryCatchalls);
};
const commandNotFound = (argv, partialMatches, execOptions, tryCatchalls = true) => {
    if (!execOptions || !execOptions.failWithUsage) {
        const catchallHandler = tryCatchalls &&
            tree_1.getModelInternal()
                .catchalls.filter(({ offer }) => offer(argv))
                .sort(({ prio: prio1 }, { prio: prio2 }) => prio2 - prio1)[0];
        if (catchallHandler) {
            return withEvents(catchallHandler.eval, catchallHandler, partialMatches);
        }
        events_1.default.emit('/command/resolved', {
            error: `${commandNotFoundMessage}: ${argv.join(' ')}`,
            command: argv[0]
        });
    }
    return suggestPartialMatches(argv.join(' '), partialMatches);
};
const findPartialMatchesAt = (usage, partial) => {
    const matches = [];
    function maybeAdd(match) {
        if (match.route.indexOf(partial) >= 0 && (!match.usage || (!match.usage.synonymFor && !match.usage.hide))) {
            matches.push(match);
        }
    }
    for (const route in usage) {
        maybeAdd(usage[route]);
        for (const cmd in usage[route].children) {
            maybeAdd(usage[route].children[cmd]);
        }
    }
    return matches;
};
exports.read = (root, argv, noRetry = false, execOptions, tryCatchalls = true) => __awaiter(void 0, void 0, void 0, function* () {
    let cmd;
    if (!noRetry) {
        yield resolver.resolve(`/${argv.join('/')}`, { tryCatchalls: false });
        cmd = yield internalRead(root, argv, tryCatchalls);
    }
    if (!cmd) {
        let matches;
        if (argv.length === 1) {
            matches = yield findPartialMatchesAt(plugins_1.prescanModel().usage, argv[0]);
        }
        else {
            const allButLast = argv.slice(0, argv.length - 1);
            const last = argv[argv.length - 1];
            const parent = yield internalRead(root, allButLast, tryCatchalls);
            if (parent) {
                matches = yield findPartialMatchesAt(plugins_1.prescanModel().usage, last);
            }
        }
        if (matches && matches.length > 0) {
        }
        else {
            matches = undefined;
        }
        return commandNotFound(argv, matches, execOptions, tryCatchalls);
    }
    else {
        return cmd;
    }
});
class ImplForPlugins {
    constructor(plugin) {
        this.plugin = plugin;
    }
    catchall(offer, handler, prio = 0, options = new DefaultCommandOptions()) {
        return tree_1.getModelInternal().catchalls.push({
            route: '*',
            offer,
            eval: handler,
            prio,
            plugin: this.plugin,
            options,
            $: undefined,
            key: undefined,
            parent: undefined
        });
    }
    listen(route, handler, options) {
        return listen(route, handler, Object.assign({}, options, { plugin: this.plugin }));
    }
    synonym(route, handler, master, options) {
        return _synonym(route, handler, master, options && Object.assign({}, options, { plugin: this.plugin }));
    }
    subtree(route, options) {
        return _subtree(route, options);
    }
    subtreeSynonym(route, master, options = master.options) {
        return _subtreeSynonym(route, master, options);
    }
    override(route, fromPlugin, overrideHandler, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const currentHandler = (yield this.find(route, fromPlugin)).$;
            if (!currentHandler) {
                throw new Error(`Cannot find desired command handler for ${route} from plugin ${fromPlugin}`);
            }
            const handler = (args) => overrideHandler(args, currentHandler);
            return this.listen(route, handler, options);
        });
    }
    find(route, fromPlugin, noOverride = true) {
        return __awaiter(this, void 0, void 0, function* () {
            const cmd = match(route.split('/').slice(1), true);
            if (!cmd || cmd.route !== route || (!noOverride && resolver && resolver.isOverridden(cmd.route))) {
                if (resolver) {
                    if (fromPlugin) {
                        yield resolver.resolveOne(fromPlugin);
                    }
                    else {
                        yield resolver.resolve(route, { tryCatchalls: false });
                    }
                }
                return match(route.split('/').slice(1), true);
            }
            else {
                return cmd;
            }
        });
    }
}
exports.ImplForPlugins = ImplForPlugins;
function proxy(plugin) {
    return new ImplForPlugins(plugin);
}
exports.proxy = proxy;
//# sourceMappingURL=command-tree.js.map