"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const path_1 = require("path");
const home_1 = require("../util/home");
const capabilities_1 = require("./capabilities");
const debug = debug_1.default('core/find-file');
debug('loading');
const specialPaths = [];
const defaultSpecial = {
    prefix: '',
    filepath: path_1.join(__dirname, '..')
};
const withMatchingTrailingSlash = (original, resolved) => {
    if (original.charAt(original.length - 1) === '/' && resolved.charAt(resolved.length - 1) !== '/') {
        return `${resolved}/`;
    }
    else {
        return resolved;
    }
};
exports.isSpecialDirectory = (filepath) => path_1.basename(filepath).charAt(0) === '@';
exports.findFileWithViewer = (filepath, { safe = false, keepRelative = false } = {}) => {
    if (!filepath) {
        if (!safe) {
            throw new Error('Please specify a file');
        }
        else {
            return { resolved: '' };
        }
    }
    else if (filepath.charAt(0) === '@') {
        const desiredPrefix = filepath.endsWith('/')
            ? filepath.slice(0, filepath.length - 1)
            : path_1.dirname(filepath) === '.'
                ? filepath
                : path_1.dirname(filepath);
        const longestMatchingSpecial = specialPaths
            .filter(({ prefix }) => filepath === prefix || desiredPrefix.indexOf(prefix) === 0)
            .sort((a, b) => b.prefix.length - a.prefix.length)[0];
        const special = longestMatchingSpecial || defaultSpecial;
        debug('resolving @ file', filepath, desiredPrefix, special);
        return {
            resolved: withMatchingTrailingSlash(filepath, path_1.join(special.filepath, filepath)),
            viewer: special.command
        };
    }
    else if (keepRelative) {
        return { resolved: filepath };
    }
    else {
        debug('resolving normal file', filepath);
        return {
            resolved: withMatchingTrailingSlash(filepath, path_1.resolve(home_1.default(filepath)))
        };
    }
};
exports.findFile = (filepath, { safe = false, keepRelative = false } = {}) => {
    return exports.findFileWithViewer(filepath, { safe, keepRelative }).resolved;
};
exports.viewer = (prefix) => {
    const special = specialPaths.find(_ => _.prefix === prefix);
    if (!special) {
        throw new Error('bad special prefix');
    }
    else {
        return special.command || undefined;
    }
};
exports.addPath = (filepath, { prefix = path_1.basename(filepath), command = '' } = {}) => {
    if (!capabilities_1.inBrowser()) {
        debug('addPath', filepath);
        try {
            const appModulePath = require('app-module-path');
            appModulePath.addPath(path_1.resolve(filepath));
        }
        catch (err) {
            console.error('error in addPath', err);
        }
    }
    if (prefix.charAt(0) === '@') {
        specialPaths.push({ prefix, filepath: path_1.dirname(filepath), command });
    }
};
//# sourceMappingURL=find-file.js.map