"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const debug = debug_1.default('main/headless');
debug('loading');
const exec_1 = require("../repl/exec");
const mimic_dom_1 = require("../util/mimic-dom");
const plugins_1 = require("../plugins/plugins");
const usage_error_1 = require("../core/usage-error");
const capabilities_1 = require("../core/capabilities");
capabilities_1.setMedia(capabilities_1.Media.Headless);
let exitCode = 0;
process.on('uncaughtException', (err) => __awaiter(void 0, void 0, void 0, function* () {
    debug('uncaughtException');
    debug(err);
    const colors = yield Promise.resolve().then(() => require('colors/safe'));
    console.error(colors.red(err.toString()));
    process.exit(1);
}));
process.on('exit', code => {
    debug('exiting', code);
});
let argStart = 1;
if (process.env.DEVMODE) {
    argStart = 2;
}
const commandContextPattern = /--command-context/;
let commandContext;
const log = console.log;
const error = console.error;
debug('modules loaded');
let graphicalShellIsOpen = false;
const noAuth = false;
const success = (quit) => (out) => __awaiter(void 0, void 0, void 0, function* () {
    debug('success!');
    if (process.env.KUI_REPL_MODE) {
        debug('returning repl mode result');
        return out;
    }
    const { print } = yield Promise.resolve().then(() => require('./headless-pretty-print'));
    yield print(out, log, process.stdout);
    if (!graphicalShellIsOpen) {
        quit();
        process.exit(exitCode);
    }
    else {
        debug('graphical shell is open');
    }
});
const failure = (quit, execOptions) => (err) => __awaiter(void 0, void 0, void 0, function* () {
    if (execOptions && execOptions.rethrowErrors) {
        throw err;
    }
    const code = err.code || err.statusCode;
    debug('failure', code, err);
    let completion = Promise.resolve();
    if (!noAuth) {
        if (process.env.TEST_CODE_ONLY && err.statusCode) {
            error(err.statusCode);
        }
        else {
            let msg;
            if (usage_error_1.default.isUsageError(err)) {
                msg = usage_error_1.default.getFormattedMessage(err);
            }
            else {
                const { oopsMessage } = yield Promise.resolve().then(() => require('../core/oops'));
                msg = oopsMessage(err);
            }
            if (typeof msg === 'string') {
                const colors = yield Promise.resolve().then(() => require('colors/safe'));
                if (process.env.TEST_INCLUDE_CODE && err.statusCode) {
                    error(colors.blue(err.statusCode.toString()) + ' ' + colors.red(msg));
                }
                else {
                    error(colors.red(msg));
                }
            }
            else {
                const { print } = yield Promise.resolve().then(() => require('./headless-pretty-print'));
                completion = print(msg, error, process.stderr, undefined, 'error') || Promise.resolve();
            }
        }
    }
    else {
        error(`No credentials found. Consider trying again with "kui help" command.`);
    }
    return completion.then(() => {
        if (!graphicalShellIsOpen) {
            exitCode = typeof code === 'number' ? code : 1;
            if (!process.env.KUI_REPL_MODE) {
                process.exit(exitCode > 128 ? exitCode - 256 : exitCode);
                if (quit) {
                    quit();
                }
            }
        }
        return false;
    });
});
const insufficientArgs = (argv) => argv.length === 0 || (argv.length === 1 && /(-h)|(--help)/.test(argv[0]));
let electronCreateWindowFn;
exports.createWindow = (argv, subwindowPlease, subwindowPrefs) => __awaiter(void 0, void 0, void 0, function* () {
    try {
        graphicalShellIsOpen = true;
        const { setGraphicalShellIsOpen } = yield Promise.resolve().then(() => require('./headless-pretty-print'));
        setGraphicalShellIsOpen();
        const commandLine = argv;
        if (commandContext) {
            commandLine.push(commandContext);
        }
        electronCreateWindowFn(commandLine, subwindowPlease, subwindowPrefs);
    }
    catch (err) {
        error(err);
    }
});
const initCommandContext = (commandContext) => __awaiter(void 0, void 0, void 0, function* () {
    if (commandContext) {
        try {
            debug('setting command context', commandContext);
            (yield Promise.resolve().then(() => require('../commands/context'))).setDefaultCommandContext(JSON.parse(commandContext.substring(commandContext.indexOf('=') + 1)));
        }
        catch (err) {
            debug('Error initializing command context', err);
        }
    }
});
function initCommandRegistrar() {
    return __awaiter(this, void 0, void 0, function* () {
        const { init } = yield Promise.resolve().then(() => require('../commands/tree'));
        yield init();
    });
}
exports.main = (app, mainFunctions, rawArgv = process.argv, execOptions) => __awaiter(void 0, void 0, void 0, function* () {
    debug('main');
    const waitForPlugins = plugins_1.init();
    const waitForCommands = yield initCommandRegistrar();
    const ourCommandContext = rawArgv.find(_ => commandContextPattern.test(_));
    debug('commandContext', ourCommandContext);
    if (!commandContext && ourCommandContext) {
        commandContext = ourCommandContext;
    }
    const argv = rawArgv
        .slice(argStart)
        .filter(arg => !commandContextPattern.test(arg) &&
        arg !== '--kui-headless' &&
        arg !== '-v' &&
        arg !== '--raw-output' &&
        arg !== '--no-color' &&
        arg !== '--no-colors' &&
        arg !== '--color=always' &&
        arg !== '--ui');
    debug('argv', argv);
    const { quit } = app;
    exec_1.installOopsHandler(() => failure(quit, execOptions));
    electronCreateWindowFn = mainFunctions.createWindow;
    mimic_dom_1.default();
    const evaluate = (cmd) => Promise.resolve(exec_1.exec(cmd, execOptions)).then(success(quit));
    debug('bootstrap');
    return Promise.all([waitForPlugins, waitForCommands])
        .then(([initWasPerformed]) => __awaiter(void 0, void 0, void 0, function* () {
        debug('plugins initialized');
        if (initWasPerformed) {
            yield initCommandContext(commandContext);
        }
        const maybeRetry = (err) => {
            return failure(quit, execOptions)(err);
        };
        if (initWasPerformed) {
            debug('invoking plugin preloader');
            yield plugins_1.preload();
            debug('invoking plugin preloader... done');
        }
        if (insufficientArgs(argv)) {
            debug('insufficient args, invoking help command');
            return evaluate('help');
        }
        const cmd = argv
            .map(_ => (_.match(/\s+/) ? `"${_}"` : _))
            .join(' ')
            .trim();
        if (cmd && cmd.length > 0) {
            debug('about to execute command');
            return evaluate(cmd).catch(maybeRetry);
        }
        else {
            debug('exiting, no command');
            if (!process.env.KUI_REPL_MODE) {
                process.exit(0);
            }
        }
    }))
        .catch(failure(quit, execOptions));
});
function initHeadless(argv, force = false, isRunningHeadless = false, execOptions) {
    if (force || isRunningHeadless) {
        debug('initHeadless');
        const app = {
            quit: () => process.exit(0)
        };
        try {
            return exports.main(app, {
                createWindow: (executeThisArgvPlease, subwindowPlease, subwindowPrefs) => __awaiter(this, void 0, void 0, function* () {
                    const { createWindow: createElectronWindow } = yield Promise.resolve().then(() => require('./spawn-electron'));
                    return createElectronWindow(true, executeThisArgvPlease, subwindowPlease, subwindowPrefs);
                })
            }, argv, execOptions);
        }
        catch (err) {
            console.error('Internal Error, please report this bug:');
            console.error(err);
            if (!process.env.KUI_REPL_MODE) {
                process.exit(1);
            }
            else {
                throw err;
            }
        }
    }
    else {
    }
}
exports.initHeadless = initHeadless;
//# sourceMappingURL=headless.js.map