/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
const debug = Debug('webapp/pip');
debug('loading');
import { removeAllDomChildren } from './util/dom';
import { getSidecar, getActiveView } from './views/sidecar';
import { popupListen } from './listen';
import Presentation from './views/presentation';
import { pexec, qexec } from '../repl/exec';
import { css, rawCSS } from './bottom-stripe';
import { isCapturable } from './models/capturable';
const remove = (elt, cls) => elt.classList.remove(cls);
const add = (elt, cls) => elt.classList.add(cls);
const _highlight = (op) => (highlightThis) => {
    if (highlightThis) {
        if (Array.isArray(highlightThis)) {
            highlightThis.forEach(_ => op(_, 'picture-in-picture-highlight'));
        }
        else {
            op(highlightThis, 'picture-in-picture-highlight');
        }
    }
};
const dehighlight = _highlight(remove);
const highlight = _highlight(add);
/**
 * Make an DOM event handler that will restore the given pippedContainer
 *
 */
const restore = (tab, pippedContainer, previousPresentation, sidecarClass, capturedHeaders, highlightThis, escapeHandler, options) => () => {
    debug('restore');
    const sidecar = getSidecar(tab);
    const parent = (options && options.parent) || sidecar.querySelector('.custom-content');
    if (pippedContainer !== true) {
        removeAllDomChildren(parent);
    }
    if (previousPresentation) {
        debug('restoring presentation mode', previousPresentation, Presentation[previousPresentation].toString());
        document.body.setAttribute('data-presentation', Presentation[previousPresentation].toString());
    }
    // restore escape handler
    if (escapeHandler) {
        document.onkeyup = escapeHandler;
    }
    // restore sidecar header state
    sidecar.className = sidecarClass;
    capturedHeaders.forEach(({ selector, node, redraw, nextSibling }) => {
        const curHeader = document.querySelector(selector);
        const curHeaderParent = curHeader.parentNode;
        curHeaderParent.removeChild(curHeader);
        curHeaderParent.insertBefore(node, nextSibling);
        if (redraw) {
            redraw(node);
        }
    });
    // pippedContainer.classList.remove('picture-in-picture')
    // pippedContainer.classList.add('picture-in-picture-stage1')
    // setTimeout(() => {
    // sidecar.classList.add('custom-content')
    // pippedContainer.classList.remove('picture-in-picture-stage1')
    if (pippedContainer !== true && pippedContainer !== false) {
        if (!pippedContainer.classList.contains('sidecar-content')) {
            if (pippedContainer.parentNode) {
                pippedContainer.parentNode.removeChild(pippedContainer);
            }
            parent.appendChild(pippedContainer);
        }
    }
    // pippedContainer.onclick = null
    // }, 300)
    dehighlight(highlightThis);
};
/**
 *
 *
 */
const pip = (tab, container, previousPresentation, capturedHeaders, highlightThis, returnTo = 'previous view', options) => {
    try {
        if (container !== true && container !== false) {
            if (!container.classList.contains('sidecar-content')) {
                container.parentNode.removeChild(container);
            }
        }
    }
    catch (e) {
        // ok
    }
    const sidecar = getSidecar(tab);
    const sidecarClass = sidecar.className;
    const escapeHandler = undefined; // we don't want to override the escape key behavior
    const backContainer = css.backContainer(tab);
    const backButton = css.backButton(tab);
    const restoreFn = restore(tab, container, previousPresentation, sidecarClass, capturedHeaders, highlightThis, escapeHandler, options);
    debug('returnTo', returnTo);
    backButton.setAttribute('data-balloon', `Return to ${returnTo}`);
    backContainer.classList.add('has-back-button');
    backButton.addEventListener('click', () => __awaiter(void 0, void 0, void 0, function* () {
        // Notes: wait for any drilldown in the current tab to finish
        // before actuating the back button behavior.
        // https://github.com/IBM/kui/issues/3114
        yield (tab.state.drilldownInProgress || Promise.resolve());
        restoreFn();
        backContainer.classList.remove('has-back-button');
    }), { once: true });
    return restoreFn;
};
/**
 * Capture and clone the given selector
 *
 */
const capture = (tab, selector, redraw) => {
    // capture the current dom via deep clone
    const node = tab.querySelector(selector);
    const clone = node.cloneNode(true);
    // remember this, so we can reattach in the right place (using insertBefore)
    const parent = node.parentNode;
    const nextSibling = node.nextSibling;
    node.remove();
    parent.insertBefore(clone, nextSibling);
    return {
        selector,
        node,
        redraw,
        nextSibling
    };
};
/**
 * Drill down to a more detailed view, using the given command to
 * populate the new view.
 *
 */
export const drilldown = (tab, command, highlightThis, ccontainer, returnTo, options) => (event) => __awaiter(void 0, void 0, void 0, function* () {
    if (event)
        event.stopPropagation();
    // maybe ccontainer is a query selector
    const container = typeof ccontainer === 'string' ? document.querySelector(ccontainer) : ccontainer || getActiveView(tab);
    debug('drilldown', command, container, returnTo);
    // capture the current header and other DOM state, before the `command` overwrites it
    const alreadyPipped = document.querySelector('body > .picture-in-picture');
    const presentationString = document.body.getAttribute('data-presentation');
    const presentation = presentationString && Presentation[presentationString];
    const capturedHeader = capture(tab, '.sidecar-header-text', popupListen);
    const capturedHeader2 = capture(tab, '.header-right-bits .custom-header-content');
    const capturedHeader3 = capture(tab, '.header-right-bits .action-content');
    const capturedHeader4 = capture(tab, '.sidecar-header-icon');
    const capturedHeader5 = capture(tab, '.sidecar-header-secondary-content');
    // for the footer, we need to capture the modeButton renderer, so we can reattach the click events
    const modeContainer = css.modeContainer(tab);
    const modeButtons = isCapturable(modeContainer) && modeContainer.capture;
    const capturedFooter = capture(tab, rawCSS.buttons, modeButtons && modeButtons());
    const capturedHeaders = [
        capturedHeader,
        capturedHeader2,
        capturedHeader3,
        capturedHeader4,
        capturedHeader5,
        capturedFooter
    ];
    // make the transition
    const restoreFn = container && !alreadyPipped
        ? pip(tab, container, presentation, capturedHeaders, highlightThis, returnTo, options)
        : () => true;
    highlight(highlightThis);
    // now we can safely begin executing the command
    debug('executing command', command);
    // eslint-disable-next-line no-async-promise-executor
    tab.state.drilldownInProgress = new Promise((resolve) => __awaiter(void 0, void 0, void 0, function* () {
        const done = () => {
            tab.state.drilldownInProgress = undefined;
            resolve();
        };
        if (typeof command === 'string') {
            debug('drilling down with string command');
            const execOptions = Object.assign({}, {
                isDrilldown: true,
                preserveBackButton: true,
                rethrowErrors: true,
                reportErrors: true
            }, (options && options.execOptions) || {});
            if (!options || options.exec === 'pexec') {
                return pexec(command, execOptions)
                    .then(done)
                    .catch(restoreFn);
            }
            else {
                return qexec(command, undefined, undefined, execOptions)
                    .then(done)
                    .catch(restoreFn);
            }
        }
        else if (typeof command === 'function') {
            return command()
                .then(done)
                .catch(restoreFn);
        }
        else {
            console.error('no viewer for content');
            done();
        }
    }));
    yield tab.state.drilldownInProgress;
});
export default drilldown;
//# sourceMappingURL=picture-in-picture.js.map