/*
 * Copyright 2017 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import * as assert from 'assert';
import { Common, CLI, Keys, ReplExpect, Selectors, SidecarExpect } from '@kui-shell/test';
describe(`clear the console ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const doPrompt = (opts) => () => __awaiter(this, void 0, void 0, function* () {
        const { enteredString, enteredPlaceholder = '', expectedPlaceholder = 'Test prompt', cancel = false } = opts;
        try {
            const res = yield CLI.command(`prompt ${enteredPlaceholder}`, this.app);
            yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                const placeholder = yield this.app.client.getAttribute(Selectors.PROMPT_N(res.count), 'placeholder');
                return placeholder === expectedPlaceholder;
            }));
            if (cancel) {
                yield this.app.client.keys(Keys.ctrlC);
                return ReplExpect.blank(res);
            }
            else {
                this.app.client.keys(`${enteredString}${Keys.ENTER}`);
                return ReplExpect.okWithString(enteredString)(res);
            }
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    });
    const enteredString = 'does this work?';
    const enteredString2 = 'does this also work?';
    it(`quick test of prompt`, doPrompt({ enteredString }));
    it(`another quick test of prompt`, doPrompt({
        enteredString: enteredString2,
        enteredPlaceholder: 'foo',
        expectedPlaceholder: 'foo'
    }));
    it(`cancel test of prompt`, doPrompt({
        enteredPlaceholder: 'foo3',
        expectedPlaceholder: 'foo3',
        cancel: true
    }));
    // get something on the screen
    it(`should sleep`, () => CLI.command('sleep 1', this.app).catch(Common.oops(this, true)));
    it('should clear the console', () => CLI.command('clear', this.app)
        .then(() => ReplExpect.consoleToBeClear(this.app))
        .then(() => SidecarExpect.closed)
        .catch(Common.oops(this, true)));
    // get something on the screen
    it(`should sleep again`, () => CLI.command('sleep 1', this.app).catch(Common.oops(this, true)));
    const JUNK = 'junk text that should stay';
    it('should clear the console with ctrl+l', () => CLI.command(JUNK, this.app, true)
        .then(() => __awaiter(this, void 0, void 0, function* () {
        yield this.app.client.keys([Keys.CONTROL, 'l', 'NULL']); // use control-l to clear
        return ReplExpect.consoleToBeClear(this.app);
    }))
        .then(() => this.app.client.getValue(Selectors.CURRENT_PROMPT))
        .then(text => assert.strictEqual(text, JUNK))
        .catch(Common.oops(this, true)));
    // hit enter, and expect that JUNK to fail
    it(`should fail with command not found`, () => {
        return CLI.command('nope', this.app)
            .then(ReplExpect.error(127))
            .catch(Common.oops(this, true));
    });
    // get something on the screen
    it(`should sleep yet again`, () => CLI.command('sleep 1', this.app).catch(Common.oops(this, true)));
    // FIXME prompt does not work in webpack+proxy
    it('should clear properly despite existing prompt', () => CLI.command('prompt', this.app) // wipe will change the placeholder text
        .then(() => __awaiter(this, void 0, void 0, function* () {
        yield this.app.client.keys([Keys.CONTROL, 'l', 'NULL']); // use control-l to clear
        return ReplExpect.consoleToBeClear(this.app);
    }))
        .catch(Common.oops(this, true)));
});
//# sourceMappingURL=clear.js.map