"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const libraryName = 'composer';
const composerPattern = new RegExp(`${libraryName}.`);
const strings = {
    literal: {
        documentation: 'Inject a constant value',
        detail: 'Inject a constant value',
        args: ['aConstantValue']
    },
    if: {
        documentation: 'Conditionally execute a task',
        detail: 'if (condTask) thenTask; else elseTask',
        args: ['condTask', 'thenTask', 'elseTask']
    },
    sequence: {
        documentation: 'Sequence two or more tasks into a pipeline',
        detail: 'task1 -> task2',
        args: ['task1', 'task2']
    },
    try: {
        documentation: 'Catch errors in tasks with a handler task',
        detail: 'try { task1 } catch { task2 }',
        args: ['task', 'errorHandler']
    },
    let: {
        documentation: 'Bind a variable to a value, then execute a sequence of tasks',
        detail: 'var=value; task1 -> task2',
        args: ['variable', 'value', 'task1', 'task2']
    },
    mask: {
        documentation: 'Execute a given composition masking any enclosing variable bindings',
        detail: 'mask enclosing variable bindings',
        args: ['composition']
    },
    while: {
        documentation: 'While a condition holds, execute a task',
        detail: 'while (cond) task',
        args: ['cond', 'task']
    },
    repeat: {
        documentation: 'Repeat a task N times',
        detail: 'for(i=0;i<N;i++) task',
        args: ['N', 'task']
    },
    retry: {
        documentation: 'If a task fails, retry at most N times',
        detail: 'for(i=0;i<N);i++) if(task) break',
        args: ['N', 'task']
    },
    retain: {
        documentation: 'Execute a task, then re-inject the input',
        detail: 'output=task(input); return Object.assign(output, input)',
        args: ['task']
    },
    task: {
        documentation: 'You may optionally choose to merge the task input and output (options.merge), or project a given field (options.output)',
        detail: 'Perform a single task',
        args: ['task', 'options']
    }
};
const makeProposal = languages => keyword => Object.assign({
    label: keyword,
    kind: languages.CompletionItemKind.Method,
    insertText: {
        value: `${keyword}(${strings[keyword].args.map((_, idx) => `\${${idx + 1}:${_}}`).join(', ')})`
    }
}, strings[keyword]);
const proposals = languages => Object.keys(strings).map(makeProposal(languages));
exports.default = languages => ({
    language: 'javascript',
    provider: {
        triggerCharacters: ['.'],
        provideCompletionItems: (model, position) => {
            const lastChars = model.getValueInRange({
                startLineNumber: position.lineNumber,
                startColumn: 0,
                endLineNumber: position.lineNumber,
                endColumn: position.column
            });
            const words = lastChars.replace('\t', '').split(' ');
            const activeTyping = words[words.length - 1];
            const isPrefix = activeTyping.match(composerPattern);
            if (isPrefix) {
                return proposals(languages);
            }
            else {
                return [];
            }
        }
    }
});
//# sourceMappingURL=composer-javascript.js.map