"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const path_1 = require("path");
const store_1 = require("../models/store");
const home_1 = require("../util/home");
const capabilities_1 = require("../core/capabilities");
const debug = debug_1.default('core/userdata');
exports.userDataDir = () => {
    if (capabilities_1.inBrowser()) {
        throw new Error('Unsupported operation');
    }
    else {
        const name = '@kui-shell/settings';
        switch (process.platform) {
            case 'darwin':
                return path_1.join(process.env.HOME, 'Library', 'Application Support', name);
            case 'linux': {
                const home = process.env.XDG_CONFIG_HOME || home_1.default('~/.config');
                return path_1.join(home, name);
            }
            case 'win32':
                return path_1.join(process.env.APPDATA, name);
        }
    }
};
const preferencesFilepath = () => path_1.join(exports.userDataDir(), 'prefs.json');
const fsyncPreferences = (prefs) => __awaiter(void 0, void 0, void 0, function* () {
    if (capabilities_1.inBrowser()) {
        store_1.default().setItem('kui.userprefs', JSON.stringify(prefs));
    }
    else {
        const { mkdirp, writeFile } = yield Promise.resolve().then(() => require('fs-extra'));
        yield mkdirp(exports.userDataDir());
        yield writeFile(preferencesFilepath(), JSON.stringify(prefs));
    }
    return prefs;
});
function readFile(filepath) {
    return __awaiter(this, void 0, void 0, function* () {
        const { readFile } = yield Promise.resolve().then(() => require('fs'));
        return new Promise((resolve, reject) => {
            readFile(filepath, (err, data) => {
                if (err) {
                    reject(err);
                }
                else {
                    resolve(data.toString());
                }
            });
        });
    });
}
const preferences = () => __awaiter(void 0, void 0, void 0, function* () {
    if (capabilities_1.inBrowser()) {
        debug('reading preferences from browser localStorage');
        const prefs = store_1.default().getItem('kui.userprefs');
        if (!prefs) {
            return {};
        }
        else {
            try {
                return JSON.parse(prefs);
            }
            catch (err) {
                debug('error parsing preference model', prefs);
                console.error('error parsing preference model', err);
                return {};
            }
        }
    }
    try {
        const filepath = preferencesFilepath();
        debug('reading persisted preference model', filepath);
        const raw = yield readFile(filepath);
        try {
            return JSON.parse(raw);
        }
        catch (err) {
            debug('error parsing preference model', raw);
            console.error('error parsing preference model', err);
            return {};
        }
    }
    catch (err) {
        if (err.code === 'ENOENT') {
            return fsyncPreferences({});
        }
        else {
            throw err;
        }
    }
});
exports.purgePreferences = () => __awaiter(void 0, void 0, void 0, function* () {
    debug('purgePreferences');
    if (capabilities_1.inBrowser()) {
        store_1.default().removeItem('kui.userprefs');
    }
    else {
        const { unlink } = yield Promise.resolve().then(() => require('fs-extra'));
        yield unlink(preferencesFilepath());
    }
});
exports.clearPreference = (key) => __awaiter(void 0, void 0, void 0, function* () {
    debug('clearPreference', key);
    const prefs = yield preferences();
    const value = prefs[key];
    delete prefs[key];
    yield fsyncPreferences(prefs);
    return value;
});
exports.getPreference = (key) => __awaiter(void 0, void 0, void 0, function* () {
    const prefs = yield preferences();
    const value = prefs[key];
    debug('getPreference', key, value);
    return value;
});
exports.setPreference = (key, value) => __awaiter(void 0, void 0, void 0, function* () {
    debug('setPreference', key, value);
    const prefs = yield preferences();
    prefs[key] = value;
    yield fsyncPreferences(prefs);
    return value;
});
//# sourceMappingURL=userdata.js.map