/*
 * Copyright 2019 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { isCustomSpec } from '../../webapp/views/custom-content';
import { addModeButtons } from '../../webapp/bottom-stripe';
import { isTable } from '../../webapp/models/table';
import { isButton } from './types';
import { hasContent, isScalarContent, isCommandStringContent, isStringWithOptionalContentType, isFunctionContent } from './content-types';
/**
 * Turn a Resource into content that can be presented in a sidecar tab
 *
 */
export function formatForTab(tab, mmr, resource, args) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!hasContent(resource)) {
            // then we have a plain resource. the rest of this function
            // assumes a Content structure, so wrap it up as such
            return formatForTab(tab, mmr, { content: resource }, args);
        }
        else if (isFunctionContent(resource)) {
            // then resource.content is a function that will provide the information
            return formatForTab(tab, mmr, yield resource.content(tab, mmr, args), args);
        }
        else if (isCommandStringContent(resource)) {
            const content = yield tab.REPL.qexec(resource.contentFrom);
            if (resource.contentType && typeof content === 'string') {
                return formatForTab(tab, mmr, {
                    content,
                    contentType: resource.contentType
                }, args);
            }
            else {
                return formatForTab(tab, mmr, content, args);
            }
        }
        else if (isCustomSpec(resource.content)) {
            return resource.content;
        }
        else if (isTable(resource.content)) {
            return resource.content;
        }
        else {
            // otherwise, we have string or HTMLElement content
            return Object.assign({ resource: mmr, toolbarText: mmr.toolbarText, kind: mmr.kind, metadata: mmr.metadata, type: 'custom' }, resource);
        }
    });
}
function wrapTable(tab, table, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const dom1 = document.createElement('div');
        const dom2 = document.createElement('div');
        dom1.classList.add('scrollable', 'scrollable-auto');
        dom2.classList.add('result-as-table', 'repl-result');
        dom1.appendChild(dom2);
        const { formatTable } = yield import('../../webapp/views/table');
        formatTable(tab, table, dom2, args);
        return dom1;
    });
}
function renderContent(tab, bearer, content, args) {
    return __awaiter(this, void 0, void 0, function* () {
        if (isStringWithOptionalContentType(content)) {
            return content;
        }
        else if (isFunctionContent(content)) {
            const actualContent = (yield content.content(tab, bearer, args));
            if (!isScalarContent(actualContent)) {
                if (isTable(actualContent)) {
                    return {
                        content: yield wrapTable(tab, actualContent, args)
                    };
                }
                else {
                    return {
                        content: actualContent
                    };
                }
            }
            else {
                return actualContent;
            }
        }
        else if (isScalarContent(content)) {
            if (isTable(content.content)) {
                return {
                    content: yield wrapTable(tab, content.content, args)
                };
            }
            else {
                return content;
            }
        }
        else if (isTable(content)) {
            return {
                content: yield wrapTable(tab, content, args)
            };
        }
        else if (isCommandStringContent(content)) {
            return {
                content: yield tab.REPL.qexec(content.contentFrom),
                contentType: content.contentType
            };
        }
    });
}
/**
 * Render a MultiModalResponse to the sidecar
 *
 */
export function show(tab, mmr, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const modes = mmr.modes;
        // const buttons = mmr.buttons ? formatButtons(tab, mmr, mmr.buttons) : ([] as SidecarMode[])
        const ourModesWithButtons = modes.concat(mmr.buttons || []);
        // first, do a "modelOnly" pass, to get the full list of modes
        // see https://github.com/IBM/kui/issues/3589
        const modesWithButtons = addModeButtons(tab, ourModesWithButtons, mmr, args, {
            preserveBackButton: true,
            show: mmr.defaultMode,
            modelOnly: true
        });
        const defaultMode = modesWithButtons.find(_ => !isButton(_) && _.defaultMode) || modesWithButtons.find(_ => !isButton(_));
        if (isButton(defaultMode)) {
            console.error('default mode is a button', defaultMode.mode, modesWithButtons, mmr);
            throw new Error('default mode is a button');
        }
        const content = hasContent(defaultMode) ? yield renderContent(tab, mmr, defaultMode, args) : undefined;
        // now that we've rendered the initial/default content, do a pass
        // over the modes and add them to the UI; see
        // https://github.com/IBM/kui/issues/3589
        addModeButtons(tab, ourModesWithButtons, mmr, args, {
            preserveBackButton: true,
            show: mmr.defaultMode
        });
        if (content) {
            const custom = Object.assign({
                type: 'custom',
                resource: mmr,
                modes: modesWithButtons,
                toolbarText: mmr.toolbarText,
                prettyName: mmr.prettyName,
                nameHash: mmr.nameHash
            }, content);
            const { showCustom } = yield import('../../webapp/views/sidecar');
            return showCustom(tab, custom, { leaveBottomStripeAlone: true });
        }
        else {
            console.error('empty content');
        }
    });
}
//# sourceMappingURL=show.js.map