/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { getCurrentTab } from './tab';
import { isHTML } from '../util/types';
function stripe() {
    return document.querySelector('#kui--status-stripe');
}
function region(pos) {
    return stripe().querySelector(`.kui--status-stripe-${pos}`);
}
function hasId(fragment) {
    return typeof fragment.id === 'string';
}
function isTextWithIcon(fragment) {
    const twi = fragment;
    return typeof twi.icon === 'string' && isHTML(twi.text);
}
/**
 * Implementation of StatusStripeController which uses a DOM Element
 * to wrap the fragment. It also listens to a set of default events:
 * new tab, tab switch, and command completion.
 *
 */
class ElementController {
    // eslint-disable-next-line no-useless-constructor
    constructor(element, fragment) {
        this.element = element;
        this.fragment = fragment;
    }
    showAs(level) {
        this.element.setAttribute('data-view', level);
        return this;
    }
    unlisten(myListener) {
        return __awaiter(this, void 0, void 0, function* () {
            const { default: eventBus } = yield import('../core/events');
            eventBus.off('/tab/new', myListener);
            eventBus.off('/tab/switch', myListener);
            eventBus.off('/command/complete/fromuser', myListener);
        });
    }
    listen(listener) {
        return __awaiter(this, void 0, void 0, function* () {
            const myListener = (tab) => {
                // since we have a single-global status, make sure that the
                // event pertains to the current tab; we also listen to
                // /tab/switch (just below), so we will reflect any events that
                // we drop (here) when the user switches to that other tab
                if (tab === getCurrentTab()) {
                    Promise.resolve(listener(tab, this, this.fragment)).catch(err => {
                        console.error('error refreshing status stripe fragment', err);
                        this.showAs('hidden');
                    });
                }
            };
            const { default: eventBus } = yield import('../core/events');
            eventBus.on('/tab/new', myListener);
            eventBus.on('/tab/switch', myListener);
            eventBus.on('/command/complete/fromuser', myListener);
            return myListener;
        });
    }
}
/**
 * The StatusStripe API.
 *
 */
class StatusStripeAPI {
    render(fragment) {
        if (isTextWithIcon(fragment)) {
            const frag = document.createDocumentFragment();
            const icon = document.createElement('a');
            icon.href = '#'; //
            icon.classList.add(fragment.iconIsNarrow ? 'tiny-right-pad' : 'small-right-pad', 'kui--status-stripe-icon');
            icon.innerHTML = fragment.icon;
            if (fragment.onclick) {
                if (fragment.onclick.text) {
                    const clickable = document.createElement('a');
                    clickable.href = '#';
                    clickable.classList.add('clickable', 'kui--status-stripe-text');
                    clickable.appendChild(fragment.text);
                    fragment.text = clickable;
                    fragment.text.onclick = () => getCurrentTab().REPL.pexec(fragment.onclick.text);
                }
                if (fragment.onclick.icon) {
                    icon.classList.add('clickable');
                    icon.onclick = () => getCurrentTab().REPL.pexec(fragment.onclick.icon);
                }
            }
            frag.appendChild(icon);
            frag.appendChild(fragment.text);
            return frag;
        }
        else {
            throw new Error('unsupported fragment type');
        }
    }
    wrap(fragment) {
        const wrapper = document.createElement('div');
        wrapper.classList.add('kui--status-stripe-element');
        wrapper.appendChild(this.render(fragment));
        if (hasId(fragment)) {
            wrapper.classList.add(fragment.id);
        }
        return wrapper;
    }
    /**
     * Append the given `fragment` at the given position `pos`. Note:
     * that the fragment will be hidden by default. This is to avoid the
     * fragment showing in a partial form prior to the first
     * availability of data.
     *
     * @return a controller that can be used e.g. to change whether or
     * not the fragment is visible
     *
     */
    addTo(pos, fragment) {
        const element = this.wrap(fragment);
        region(pos).appendChild(element);
        // re: initial invisibility, see
        // https://github.com/IBM/kui/issues/3538
        return new ElementController(element, fragment).showAs('hidden');
    }
    remove(pos, fragment) {
        region(pos).removeChild(fragment);
    }
}
/**
 * Export the API impl
 *
 */
export default new StatusStripeAPI();
//# sourceMappingURL=status-stripe.js.map