/*
 * Copyright 2017-20 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { readdir } from 'fs';
import { basename, join, dirname as pathDirname } from 'path';
import { expandHomeDir } from '@kui-shell/core';
import { shellescape, isDirectory, registerTabCompletionEnumerator } from '@kui-shell/plugin-core-support/tab-completion';
/**
 * Tab completion handler for local files
 *
 */
function completeLocalFiles(tab, commandLine, { toBeCompleted }) {
    return __awaiter(this, void 0, void 0, function* () {
        return (yield tab.REPL.rexec(`fscomplete ${toBeCompleted}`)).content;
    });
}
function doComplete(args) {
    const last = args.command.substring(args.command.indexOf('fscomplete ') + 'fscomplete '.length);
    // dirname will "foo" in the above example; it
    // could also be that last is itself the name
    // of a directory
    const lastIsDir = last.charAt(last.length - 1) === '/';
    const dirname = lastIsDir ? last : pathDirname(last);
    // debug('suggest local file', dirname, last)
    if (dirname) {
        // then dirname exists! now scan the directory so we can find matches
        return new Promise((resolve, reject) => {
            const dirToScan = expandHomeDir(dirname);
            readdir(dirToScan, (err, files) => __awaiter(this, void 0, void 0, function* () {
                if (err) {
                    console.error('fs.readdir error', err);
                    reject(err);
                }
                else {
                    const partial = shellescape(basename(last) + (lastIsDir ? '/' : ''));
                    const partialHasADot = partial.startsWith('.');
                    const matches = files.filter(_f => {
                        const f = shellescape(_f);
                        // exclude dot files from tab completion, also emacs ~ temp files just for fun
                        return ((lastIsDir || f.indexOf(partial) === 0) &&
                            !f.endsWith('~') &&
                            f !== '.' &&
                            f !== '..' &&
                            (partialHasADot || !f.startsWith('.')));
                    });
                    // add a trailing slash to any matched directory names
                    resolve({
                        mode: 'raw',
                        content: yield Promise.all(matches.map((match) => __awaiter(this, void 0, void 0, function* () {
                            const completion = lastIsDir ? match : match.substring(partial.length);
                            if (yield isDirectory(join(dirToScan, match))) {
                                return `${completion}/`;
                            }
                            else {
                                return { completion, addSpace: true };
                            }
                        })))
                    });
                }
            }));
        });
    }
}
/**
 * Entry point to register tab completion handlers
 *
 */
export function preload() {
    registerTabCompletionEnumerator(completeLocalFiles);
}
export function plugin(registrar) {
    registrar.listen('/fscomplete', doComplete, { hidden: true, requiresLocal: true });
}
//# sourceMappingURL=tab-completion.js.map