"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const core_1 = require("@kui-shell/core");
const traffic_light_1 = require("../../model/traffic-light");
const util_1 = require("../../util/util");
const strings = core_1.i18n('plugin-kubeui');
const debug = debug_1.default('k8s/view/modes/containers');
const headerModel = (pod) => {
    const statuses = pod.status && pod.status.containerStatuses;
    const specAttrs = [{ value: 'PORTS', outerCSS: 'header-cell pretty-narrow' }];
    const statusAttrs = !statuses
        ? []
        : [
            { value: 'RESTARTS', outerCSS: 'header-cell very-narrow' },
            { value: 'READY', outerCSS: 'header-cell very-narrow' },
            { value: 'STATE', outerCSS: 'header-cell pretty-narrow' },
            { value: 'MESSAGE', outerCSS: 'header-cell' }
        ];
    return {
        type: 'container',
        name: 'IMAGE',
        outerCSS: 'header-cell not-too-wide',
        attributes: specAttrs.concat(statusAttrs)
    };
};
const showLogs = (tab, { pod, container }, args) => {
    const podName = core_1.encodeComponent(pod.metadata.name);
    const containerName = core_1.encodeComponent(container.name);
    const ns = core_1.encodeComponent(pod.metadata.namespace);
    return `${util_1.getCommandFromArgs(args)} logs ${podName} ${containerName} -n ${ns}`;
};
const bodyModel = (tab, pod, args) => {
    const statuses = pod.status && pod.status.containerStatuses;
    const bodyModel = pod.spec.containers
        .map(container => {
        const status = statuses && statuses.find(_ => _.name === container.name);
        if (!status) {
            return;
        }
        debug('container status', container.name, status.restartCount, status);
        const stateKey = Object.keys(status.state)[0];
        const stateBody = status.state[stateKey];
        const statusAttrs = !status
            ? []
            : [
                {
                    key: 'restartCount',
                    value: status.restartCount,
                    outerCSS: 'very-narrow'
                },
                {
                    key: 'ready',
                    value: status.ready,
                    fontawesome: status.ready ? 'fas fa-check-circle' : 'far fa-dot-circle',
                    css: status.ready ? 'green-text' : 'yellow-text'
                },
                {
                    key: 'state',
                    value: stateKey,
                    tag: 'badge',
                    outerCSS: 'capitalize',
                    css: stateKey === 'running'
                        ? traffic_light_1.default.Green
                        : stateKey === 'terminated'
                            ? traffic_light_1.default.Red
                            : traffic_light_1.default.Yellow
                },
                {
                    key: 'message',
                    outerCSS: 'smaller-text not-too-wide',
                    value: stateBody.startedAt || stateBody.reason
                }
            ];
        const portsAttr = {
            key: 'ports',
            outerCSS: 'not-too-wide',
            value: (container.ports || []).map(({ containerPort, protocol }) => `${containerPort}/${protocol}`).join(' ')
        };
        const specAttrs = [portsAttr];
        return {
            type: 'container',
            name: container.name,
            onclick: showLogs(tab, { pod, container }, args),
            attributes: specAttrs.concat(statusAttrs)
        };
    })
        .filter(_ => _);
    debug('body model', bodyModel);
    return bodyModel;
};
function renderContainers(tab, resource, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const fetchPod = `${util_1.getCommandFromArgs(args)} get pod ${core_1.encodeComponent(resource.metadata.name)} -n "${resource.metadata.namespace}" -o json`;
        debug('issuing command', fetchPod);
        try {
            const podResource = yield tab.REPL.qexec(fetchPod, undefined, undefined, { raw: true });
            debug('renderContainers.response', podResource);
            return {
                header: headerModel(podResource),
                body: bodyModel(tab, podResource, args),
                noSort: true,
                title: 'Containers'
            };
        }
        catch (err) {
            if (err.code === 404) {
                return { body: [] };
            }
            else {
                throw err;
            }
        }
    });
}
function hasContainers(resource) {
    return resource.spec && resource.spec.containers;
}
exports.containersMode = {
    when: hasContainers,
    mode: {
        mode: 'containers',
        label: strings('containers'),
        content: renderContainers
    }
};
exports.default = exports.containersMode;
//# sourceMappingURL=containers.js.map